# w.c.s. - web application for online forms
# Copyright (C) 2005-2025  Entr'ouvert
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see <http://www.gnu.org/licenses/>.

import io
import os
import re
import zipfile
from datetime import datetime
from xml.etree import ElementTree as ET

from django.core.management.base import BaseCommand
from django.utils.encoding import force_str

from wcs.qommon import ezt, template
from wcs.qommon.publisher import get_publisher_class

dry_run = False
now = datetime.now().strftime('%Y%m%d-%H%M%S')


class Command(BaseCommand):
    help = 'Translate EZT templates to Django'

    def add_arguments(self, parser):
        parser.set_defaults(verbosity=1)
        parser.add_argument('-d', '--domain', '--vhost', metavar='DOMAIN')
        parser.add_argument(
            '--dry-run',
            action='store_true',
            help='Just show what translations would be made; do not actually write them.',
        )

    def handle(self, verbosity, domain=None, **options):
        global dry_run  # noqa pylint: disable=global-statement
        dry_run = options['dry_run']

        if domain:
            domains = [domain]
        else:
            domains = [x.hostname for x in get_publisher_class().get_tenants()]

        publisher_class = get_publisher_class()
        publisher_class.register_cronjobs()
        publisher = publisher_class.create_publisher()

        for domain in domains:
            print('domain: %s' % domain)
            publisher.set_tenant_by_hostname(domain)
            ezt_to_django_models(publisher)
            print()


def is_ezt(string):
    if not isinstance(string, str):
        return False
    if template.Template(string).format == 'ezt':
        try:
            ezt.Template().parse(string)
        except ezt.EZTException:
            return False
        if re.match(r'\[[^]]*[A-Z][^]]*\]', string):
            # don't consider leading [] expression if it has uppercases,
            # this typically happens as initial "tag" in an email subjet.
            return False
        return True
    return False


#
# Convert a EZT template to a Django one
#

pattern = re.compile('(?P<before>[^[]*?)\\[(?P<inside>.+?)\\](?P<after>.*)$', re.MULTILINE | re.DOTALL)
# "inside" is the first detected [bracket]
# "before" and "after" are, well, before and after the bracket
identifier = re.compile('^[a-zA-Z][a-zA-Z0-9_.-]*$')

EXCEPTIONS = {
    'is is_in_backoffice "False"': ('{% if is_in_backoffice %}', '{% endif %}'),
    'if-any user_email': ('{% if form_user %}', '{% endif %}'),
    'if !supportLists': None,
    'if lt IE 9': None,
    'endif': None,
    'date': '{{ form_receipt_datetime }}',
    'before': '{{ form_previous_status }}',
    'after': '{{ form_status }}',
    'today': '{{ today }}',
    'i': '[i]',  # used in some javascripts
    'r': '[r]',  # from "les B[r]on plans" (not ezt)
    'TL1': '[TL1]',  # demarches.portail.pfwb.be (not ezt)
    'ou': '[ou]',  # formulaires.demarches.metropole-rouen-normandie.fr-workflow:111
    'Automatique': '[Automatique]',  # formulaires.interne.rouen.fr-workflow:36
    'J-1': '[J-1]',  # services.formulaireintranet.grandlyon.com-workflow:79
    'adresse': '[adresse]',  # services.formulaireintranet.grandlyon.com-workflow:79
    'Commune': '[Commune]',  # services.formulaireintranet.grandlyon.com-workflow:79
    'motif': '[motif]',  # services.formulaireintranet.grandlyon.com-workflow:79
    'commune': '[commune]',  # services.formulaireintranet.grandlyon.com-workflow:79
    'field': '[field]',  # teleservices.calvados.fr-workflow:30
    'operator': '[operator]',  # teleservices.calvados.fr-workflow:30
    'value': '[value]',  # teleservices.calvados.fr-workflow:30
}


def legacy(varname):
    varname = varname.replace('-', '_')
    if varname.startswith('_'):
        return 'x' + varname
    if varname in (
        'user',
        'name',
        'details',
        'comment',
        'number',
        'url',
        'evolution',
        'tracking_code',
    ) or varname.startswith('var_'):
        return f'form_{varname}'
    if varname.startswith('option_var'):
        return 'form_option_' + varname[11:]
    if (
        not varname.startswith(('form_', 'comment_', 'session_user_'))
        and not varname.endswith('_url')
        and not '_response_' in varname
        and not '_var_' in varname
    ):
        print(f'WARNING strange varname: {varname}')
    return varname


def ezt_to_django(template, print_bugs=False, raise_on_bugs=False):
    output = ''
    ends = []

    while True:
        bracket = pattern.match(template)
        if bracket is None:
            output += template
            break
        orig_inside = bracket.group('inside')
        inside = orig_inside.strip()
        output += bracket.group('before')
        template = bracket.group('after')

        if inside == '[':  # [[]ezt syntax to dismiss bracket]
            output += '['
            continue

        if '[' in inside:
            raise SyntaxError(f'[{inside}]: [ in brackets')

        if inside == 'else':
            output += '{% else %}'
            continue

        if inside == 'end':
            if ends:
                output += ends.pop()
                continue
            raise SyntaxError('[end] without if-any/is/for')

        words = [word.strip() for word in inside.split(' ') if word.strip()]
        if not words:  # empty or space-only content
            output += '[%s]' % orig_inside
            continue

        if inside in EXCEPTIONS:
            translate = EXCEPTIONS.get(inside)
            if translate is None:
                output += '[%s]' % orig_inside
                continue
            if isinstance(translate, str):
                output += translate
                continue
            output += translate[0]
            if end := translate[1]:
                ends.append(end)
            continue

        if words[0] == 'if-any':
            if len(words) < 2:
                raise SyntaxError('empty "if-any" detected')
            output += '{%% if %s %%}' % legacy(words[1])
            ends.append('{% endif %}')
            continue

        if words[0] == 'for':
            if len(words) < 2:
                raise SyntaxError('empty "for" detected')
            item = legacy(words[1])
            output += '{%% for %s in %s %%}' % (item, item)
            ends.append('{% endfor %}')
            continue

        if words[0] == 'is':
            if len(words) < 3:
                raise SyntaxError('"is" needs 2 parameters')
            if words[2].startswith('"'):
                compare = '"' + inside.split('"')[1] + '"'
            else:
                compare = legacy(words[2])
            output += '{%% if %s == %s %%}' % (legacy(words[1]), compare)
            ends.append('{% endif %}')
            continue

        legacy_inside = legacy(inside)
        if identifier.match(legacy_inside) is not None:
            output += '{{ %s }}' % legacy_inside
            continue

        if raise_on_bugs:
            raise SyntaxError(f'cannot process: [{inside}]')
        if print_bugs:
            print(f'WARNING cannot process: [{inside}]')
        output += '[%s]' % orig_inside

    return output


#
# Formdefs
#
# TODO

#
# Workflows
#
# TODO

#
# DataSources
#
# TODO

#
# WSCall
#
# TODO

#
# mail_template
#
# TODO


#
# OpenDocument model files
#
def ezt_to_django_opendocument(stream, outstream, process_root):
    """Take a file-like object containing an ODT, ODS, or any open-office
    format, parse context.xml with element tree and apply process to its root
    node.
    """
    with zipfile.ZipFile(stream, mode='r') as zin, zipfile.ZipFile(outstream, mode='w') as zout:
        assert 'content.xml' in zin.namelist()
        for filename in zin.namelist():
            # first pass to process meta.xml, content.xml and styles.xml
            if filename not in ('meta.xml', 'content.xml', 'styles.xml'):
                continue
            content = zin.read(filename)
            root = ET.fromstring(content)
            process_root(root)
            content = ET.tostring(root)
            zout.writestr(filename, content)

        for filename in zin.namelist():
            # second pass to copy/replace other files
            if filename in ('meta.xml', 'content.xml', 'styles.xml'):
                continue
            content = zin.read(filename)
            zout.writestr(filename, content)


OO_TEXT_NS = 'urn:oasis:names:tc:opendocument:xmlns:text:1.0'
OO_OFFICE_NS = 'urn:oasis:names:tc:opendocument:xmlns:office:1.0'
OO_STYLE_NS = 'urn:oasis:names:tc:opendocument:xmlns:style:1.0'
OO_DRAW_NS = 'urn:oasis:names:tc:opendocument:xmlns:drawing:1.0'
OO_FO_NS = 'urn:oasis:names:tc:opendocument:xmlns:xsl-fo-compatible:1.0'
XLINK_NS = 'http://www.w3.org/1999/xlink'
USER_FIELD_DECL = '{%s}user-field-decl' % OO_TEXT_NS
USER_FIELD_GET = '{%s}user-field-get' % OO_TEXT_NS
SECTION_NODE = '{%s}section' % OO_TEXT_NS
SECTION_NAME = '{%s}name' % OO_TEXT_NS
STRING_VALUE = '{%s}string-value' % OO_OFFICE_NS
DRAW_FRAME = '{%s}frame' % OO_DRAW_NS
DRAW_NAME = '{%s}name' % OO_DRAW_NS
DRAW_IMAGE = '{%s}image' % OO_DRAW_NS
XLINK_HREF = '{%s}href' % XLINK_NS
NAME = '{%s}name' % OO_TEXT_NS


def ezt_to_django_oo(stream):
    global translated
    translated = False

    def process_root(root):
        if root.tag == '{%s}document-styles' % OO_OFFICE_NS:
            return
        # cache for keeping computed user-field-decl value around
        user_field_values = {}

        def process_text(t):
            global translated
            text = force_str(t)
            if is_ezt(text):
                django = ezt_to_django(text)
                if bool(text != django):
                    if dry_run:
                        if not translated:
                            print()
                        print('[dry-run] FROM EZT:', text)
                        print('[dry-run] → DJANGO:', django)
                    translated = True
                    return django
            return t

        nodes = []
        for node in root.iter():
            nodes.append(node)
        for node in nodes:
            # apply template to user-field-decl and update user-field-get
            if node.tag == USER_FIELD_DECL and STRING_VALUE in node.attrib:
                node.attrib[STRING_VALUE] = process_text(node.attrib[STRING_VALUE])
                if NAME in node.attrib:
                    user_field_values[node.attrib[NAME]] = node.attrib[STRING_VALUE]
            if node.tag == USER_FIELD_GET and NAME in node.attrib and node.attrib[NAME] in user_field_values:
                node.text = user_field_values[node.attrib[NAME]]

            for attr in ('text', 'tail'):
                value = getattr(node, attr)
                if not value:
                    continue
                setattr(node, attr, process_text(value))

    outstream = io.BytesIO()
    ezt_to_django_opendocument(stream, outstream, process_root)
    outstream.seek(0)
    if translated:
        return outstream
    return None


def is_opendocument(stream):
    try:
        with zipfile.ZipFile(stream) as z:
            if 'mimetype' in z.namelist():
                return z.read('mimetype').startswith(b'application/vnd.oasis.opendocument.')
    except zipfile.BadZipfile:
        return False
    finally:
        stream.seek(0)


def ezt_to_django_models(publisher):
    models_dir = os.path.join(publisher.app_dir, 'models')
    if not os.path.exists(models_dir):
        return
    for filename in os.listdir(models_dir):
        if filename.endswith('.invalid'):
            continue
        filename = os.path.join(models_dir, filename)
        with open(filename, 'rb') as stream:
            if not is_opendocument(stream):
                if dry_run:
                    print('[dry-run] untouched (not opendocument):', filename)
                continue
            outstream = ezt_to_django_oo(stream)
            if outstream is not None:
                if not dry_run:
                    print('UPDATED', filename)
                    with open(filename + '__django', 'wb') as outfp:
                        outfp.write(outstream.read())
                        outfp.flush()  # superstition...
                        outfp.close()
                    os.link(filename, filename + '__ezt_to_django-%s.orig.invalid' % now)
                    os.rename(filename + '__django', filename)
                else:
                    print('[dry-run] to be updated:', filename)
                    print()
            elif dry_run:
                print('[dry-run] untouched (no ezt):', filename)
