import contextlib
import hashlib
import hmac
import json
import logging
import os
import time

import flask
import redminelib
import requests

REDMINE_URL = os.environ['REDMINE_URL']
REDMINE_API_KEY = os.environ['REDMINE_API_KEY']
REDMINE_DEFAULT_PROJECT = os.environ.get('REDMINE_DEFAULT_PROJECT_ID', 'suivi-des-traces')
LOG_LEVEL = os.environ.get('LOG_LEVEL', 'INFO')
SENTRY_URL = os.environ.get('SENTRY_URL', 'https://sentry-test.entrouvert.org/')
SENTRY_TOKEN = os.environ['SENTRY_TOKEN']
SENTRY_CLIENT_SECRET = os.environ.get('SENTRY_CLIENT_SECRET')
SENTRY_ORGANIZATION_ID = os.environ.get('SENTRY_ORGANIZATION_ID', '1')

logging.basicConfig(level=LOG_LEVEL, format='%(levelname)s %(message)s')

REDMINE_CLIENT = redminelib.Redmine(REDMINE_URL, key=REDMINE_API_KEY, raise_attr_exception=False)


app = flask.Flask(__name__)


class Sentry:
    def __init__(self):
        self.url = SENTRY_URL
        self.token = SENTRY_TOKEN
        self.organization_id = SENTRY_ORGANIZATION_ID

    def get_user(self, user_id: str):
        url = self.url + f'api/0/organizations/{self.organization_id}/users/{user_id}/'
        headers = {'Authorization': f'Bearer {self.token}'}
        response = requests.get(url, timeout=1, headers=headers)
        return response.json()


SENTRY_CLIENT = Sentry()


@contextlib.contextmanager
def impersonate(username):
    current_value = REDMINE_CLIENT.engine.requests['headers'].get('X-Redmine-Switch-User')
    try:
        REDMINE_CLIENT.engine.requests['headers']['X-Redmine-Switch-User'] = username
        yield
    finally:
        if current_value:
            REDMINE_CLIENT.engine.requests['headers']['X-Redmine-Switch-User'] = current_value
        else:
            del REDMINE_CLIENT.engine.requests['headers']['X-Redmine-Switch-User']


@contextlib.contextmanager
def log():
    sentry_headers = {key: value for key, value in flask.request.headers if 'sentry' in key.lower()}
    if flask.request.method == 'POST' and flask.request.is_json:
        app.logger.debug(
            '%s %s payload %s headers %s',
            flask.request.method,
            flask.request.path + f' {flask.request.args}',
            flask.request.json,
            sentry_headers,
        )
    else:
        app.logger.debug(
            '%s %s headers %s',
            flask.request.method,
            flask.request.path + f' {flask.request.args}',
            sentry_headers,
        )
    yield


@contextlib.contextmanager
def authenticate():
    if SENTRY_CLIENT_SECRET:
        body = flask.request.data
        key = SENTRY_CLIENT_SECRET.encode()
        expected = (
            flask.request.headers.get('sentry-app-signature')
            or flask.request.headers.get('sentry-hook-signature')
            or ''
        )
        digest = hmac.new(
            key=key,
            msg=body,
            digestmod=hashlib.sha256,
        ).hexdigest()

        ok = hmac.compare_digest(digest, expected)
        app.logger.debug(
            'authenticate body %r key %r expected %s digest %s ok %s', body, key, expected, digest, ok
        )
        if not ok:
            # flask.abort(401)
            pass
    yield


_projects = None
_trackers = None


def get_projects():
    global _projects

    if not _projects:
        _projects = list(REDMINE_CLIENT.project.all())
    return _projects


def get_project(identifier):
    for project in get_projects():
        if str(project.id) == str(identifier) or project.identifier == identifier:
            return project
    return None


def get_trackers():
    global _trackers

    if not _trackers:
        _trackers = list(REDMINE_CLIENT.tracker.all())
    return _trackers


def get_tracker(name):
    for tracker in get_trackers():
        if str(tracker.name) == str(name) or tracker.name == name:
            return tracker
    return None


@app.route('/api/items/project/', methods=['GET'])
@authenticate()
@log()
def items_projects():
    projects = get_projects()
    query = flask.request.args.get('query')
    shown_projects = []
    for project in projects:
        if project.identifier == REDMINE_DEFAULT_PROJECT:
            shown_projects.insert(0, {'label': project.name, 'value': project.identifier, 'default': True})
            continue
        if query and (
            query.lower().strip() in project.identifier.lower()
            or query.lower().strip() in project.name.lower()
        ):
            shown_projects.append({'label': project.name, 'value': project.identifier, 'default': False})
    return flask.Response(json.dumps(shown_projects), mimetype='application/json')


@app.route('/api/items/issue/', methods=['GET'])
@authenticate()
@log()
def items_issue():
    query = flask.request.args.get('query', '0')

    try:
        issue = REDMINE_CLIENT.issue.get(query)
        project = get_project(issue.project.id)
        project_name = project.name if project else ''
        data = [
            {'label': f'{project_name} #{issue.id}: {issue.subject}', 'value': str(issue.id), 'default': True}
        ]
    except redminelib.exceptions.ResourceNotFoundError:
        data = []
    return flask.Response(json.dumps(data), mimetype='application/json')


@app.route('/api/issue/create/', methods=['POST'])
@authenticate()
@log()
def create_issue():
    '''Handle payload like:

    {
       "actor" : {
          "id" : 1,
          "name" : "Benjamin Dauvergne",
          "type" : "user"
       },
       "fields" : {
          "title" : "JSONDecodeError: Expecting ',' delimiter: line 27 column 59 (char 900)"
          "description" : "Comment on the trace.",
          "trace": "<trace>"
       },
       "installationId" : "7d0fde13-2a9d-4de0-87a2-e37018dea605",
       "issueId" : 11,
       "project" : {
          "id" : 3,
          "slug" : "publik"
       },
       "webUrl" : "https://sentry-test.entrouvert.org/organizations/entrouvert/issues/11/"
    }
    '''
    payload = flask.request.json
    return create_issue_handler(SENTRY_CLIENT, REDMINE_CLIENT, payload)


def create_issue_handler(sentry, redmine, payload):
    try:
        fields = payload['fields']
        title = fields['title']
        description = fields['description']
        trace = fields['trace']
        project_id = fields.get('project') or REDMINE_DEFAULT_PROJECT
        actor = payload['actor']
        url = payload['webUrl']
    except KeyError as e:
        return {'err': 1, 'err_desc': f'Invalid payload: missing key {e}'}, 400
    except ValueError as e:
        raise
        return {'err': 1, 'err_desc': f'Invalid payload: {e}'}, 400

    trace = trace.splitlines()
    trace = trace[3:-1]
    trace = '\n'.join(trace)

    user = sentry.get_user(actor['id'])
    username = user['email'].split('@')[0]

    project_identifier = None
    for project in get_projects():
        if project.identifier == project_id:
            project_identifier = project.identifier
            break

    kwargs = {
        'project_id': project_id,
        'subject': title,
        'description': f'{url}\n{description}\n<pre>{trace}</pre>',
    }

    tracker = get_tracker('Bug')
    if tracker:
        kwargs['tracker_id'] = tracker.id

    with impersonate(username):
        issue = redmine.issue.create(**kwargs)
        new_url = issue.url
    app.logger.info('created %s -> %s in project %s', url, new_url, project_identifier)
    return {'webUrl': new_url, 'project': '', 'identifier': str(issue.id)}


@app.route('/api/issue/link/', methods=['POST'])
@authenticate()
@log()
def link_issue():
    '''Handle payload like:

    {
       "actor" : {
          "id" : 1,
          "name" : "Benjamin Dauvergne",
          "type" : "user"
       },
       "fields" : {
          "title" : "JSONDecodeError: Expecting ',' delimiter: line 27 column 59 (char 900)"
          "description" : "<trace>",
       },
       "installationId" : "7d0fde13-2a9d-4de0-87a2-e37018dea605",
       "issueId" : 11,
       "project" : {
          "id" : 3,
          "slug" : "publik"
       },
       "webUrl" : "https://sentry-test.entrouvert.org/organizations/entrouvert/issues/11/"
    }
    '''
    payload = flask.request.json
    return link_issue_handler(SENTRY_CLIENT, REDMINE_CLIENT, payload)


def link_issue_handler(sentry, redmine, payload):
    try:
        fields = payload['fields']
        issue_id = fields['issue_id']
        actor = payload['actor']
        url = payload['webUrl']
    except KeyError as e:
        return {'err': 1, 'err_desc': f'Invalid payload: missing key {e}'}, 400
    except ValueError as e:
        raise
        return {'err': 1, 'err_desc': f'Invalid payload: {e}'}, 400

    user = sentry.get_user(actor['id'])
    username = user['email'].split('@')[0]

    with impersonate(username):
        issue = redmine.issue.get(issue_id)
        new_url = issue.url
        project = issue.project
        project.refresh()
    app.logger.info('link %s -> %s in project %s', url, new_url, project.identifier)
    return {'webUrl': new_url, 'project': f'{project.name} ', 'identifier': str(issue.id)}
