#!/usr/bin/python

import sys
import os
import subprocess

from setuptools import setup, find_packages
from setuptools.command.install_lib import install_lib as _install_lib
from distutils.command.build import build as _build
from distutils.command.sdist import sdist
from distutils.cmd import Command


class compile_translations(Command):
    description = 'compile message catalogs to MO files via django compilemessages'
    user_options = []

    def initialize_options(self):
        pass

    def finalize_options(self):
        pass

    def run(self):
        try:
            from django.core.management import call_command
            for path, dirs, files in os.walk('src'):
                if 'locale' not in dirs:
                    continue
                curdir = os.getcwd()
                os.chdir(os.path.realpath(path))
                call_command('compilemessages')
                os.chdir(curdir)
        except ImportError:
            sys.stderr.write('!!! Please install Django to build translations\n')


class build(_build):
    sub_commands = [('compile_translations', None)] + _build.sub_commands


class eo_sdist(sdist):

    def run(self):
        print("creating VERSION file")
        if os.path.exists('VERSION'):
            os.remove('VERSION')
        version = get_version()
        version_file = open('VERSION', 'w')
        version_file.write(version)
        version_file.close()
        sdist.run(self)
        print("removing VERSION file")
        if os.path.exists('VERSION'):
            os.remove('VERSION')


class install_lib(_install_lib):
    def run(self):
        self.run_command('compile_translations')
        _install_lib.run(self)


def get_version():
    if os.path.exists('VERSION'):
        version_file = open('VERSION', 'r')
        version = version_file.read()
        version_file.close()
        return version
    if os.path.exists('.git'):
        p = subprocess.Popen(['git', 'describe', '--dirty', '--match=v*'], stdout=subprocess.PIPE)
        result = p.communicate()[0]
        if p.returncode == 0:
            version = str(result.split()[0][1:])
            version = version.replace('-', '.')
            return version
    return '0'


setup(
    name='authentic2-wallonie-connect',
    version=get_version(),
    license='AGPLv3',
    description='Authentic2 plugin for the Wallonie Connect usecase',
    author="Entr'ouvert",
    author_email="info@entrouvert.com",
    packages=find_packages('src'),
    package_dir={
        '': 'src',
    },
    package_data={
        'authentic2_wallonie_connect': [
        ],
    },
    install_requires=[
        'authentic2',
    ],
    entry_points={
        'authentic2.plugin': [
            'authentic2-wallonie-connect = authentic2_wallonie_connect:Plugin',
        ],
    },
    cmdclass={
        'build': build,
        'install_lib': install_lib,
        'compile_translations': compile_translations,
        'sdist': eo_sdist,
    },
    zip_safe=False,
)
