# Script pour importer les numéros de puce des bacs relevés

# Sftp sur sftp -P 33335 -o KexAlgorithms=+diffie-hellman-group-exchange-sha1,diffie-hellman-group1-sha1,diffie-hellman-group14-sha1 <login>@84.239.67.17
#

import configparser
import contextlib
import re
import os.path
import urllib.parse
import logging
import logging.config

import requests

import paramiko
from paramiko._version import __version_info__

# .ini file must contain
# [paprec]
# sftp_url = sftp://user:password@ip:port/
# wcs_url = https://domain/api/cards/remontee-bacs/import-csv
# wcs_login = user
# wcs_password = password
#
# [loggers]
# keys=root
#
# [handlers]
# keys=journald
#
# [formatters]
# keys=journald
#
# [logger_root]
# handlers=journald
#
# [handler_journald]
# class=systemd.journal.JournalHandler
# formatter=journald
# level=INFO
#
# [formatter_journald]
# format=%(levelname)s %(name)s %(message)s
#
# You can read logs with:
#
#   journalctl -t import-puce-coved.py

logging.basicConfig(level=logging.DEBUG, format='%(asctime)s %(levelname)-5s %(name)-30s %(message)s')
logging.getLogger('paramiko.transport').propagate = False


def main():
    INI_FILES = [
        os.path.expanduser('~/.paprec.ini'),
        '/var/lib/wcs/tenants/demarches-tm.test.entrouvert.org/paprec.ini',
        '/var/lib/wcs/tenants/demarches-tm.eservices.toulouse-metropole.fr/paprec.ini',
    ]

    inifile = configparser.ConfigParser()
    read_ini_files = inifile.read(INI_FILES)

    logging.config.fileConfig(inifile)

    logging.info('Importing remontees-bacs')
    logging.info('Reading configuration from %s', ', '.join(read_ini_files))

    if 'paprec' not in inifile:
        logging.error('Could not find [paprec] section in files')
        raise SystemExit(1)

    debug = inifile['paprec'].get('debug')
    if debug and debug.lower().strip() in ['on', 'true', '1']:
        logging.getLogger().handlers[0].setLevel('DEBUG')
        logging.getLogger('paramiko.transport').propagate = True

    sftp_url, wcs_url, wcs_login, wcs_password = (
        inifile['paprec'].get('sftp_url'),
        inifile['paprec'].get('wcs_url'),
        inifile['paprec'].get('wcs_login'),
        inifile['paprec'].get('wcs_password'),
    )

    if not sftp_url:
        logging.error('Could not find sftp_url in [paprec] section')
        raise SystemExit(1)
    parsed = urllib.parse.urlparse(sftp_url)
    logging.debug(
        'sftp_url %s',
        urllib.parse.urlunparse(parsed._replace(netloc=parsed.netloc.split('@')[-1])),
    )

    if not wcs_url or not wcs_login or not wcs_password:
        logging.error('Could not find wcs_url or wcs_login or wcs_password in [paprec] section')
        raise SystemExit(1)
    parsed = urllib.parse.urlparse(wcs_url)
    logging.debug(
        'wcs_url %s',
        urllib.parse.urlunparse(parsed._replace(netloc=parsed.netloc.split('@')[-1])),
    )

    do_move = not (inifile['paprec'].get('move', '').strip().lower() in ['off', 'false', '0'])

    class SFTP:
        def __init__(self, url):
            self.url = url
            parsed = urllib.parse.urlparse(url)
            if not parsed.scheme == 'sftp':
                raise ValueError('invalid scheme %s' % parsed.scheme)
            if not parsed.hostname:
                raise ValueError('missing hostname')
            self.username = parsed.username or None
            self.password = parsed.password or None
            self.hostname = parsed.hostname
            self.port = parsed.port or 22
            self.path = parsed.path.strip('/')
            self.private_key = None
            self._client = None
            self._transport = None

        def __json__(self):
            return {
                'url': self.url,
            }

        def __str__(self):
            return re.sub(r'://([^/]*:[^/]*?)@', '://***:***@', self.url)

        def __eq__(self, other):
            return isinstance(other, SFTP) and other.url == self.url

        # Paramiko can hang processes if not closed, it's important to use it as a
        # contextmanager
        @contextlib.contextmanager
        def client(self):
            ssh = paramiko.SSHClient()
            try:
                if __version_info__ < (2, 2):
                    ssh.set_missing_host_key_policy(paramiko.client.AutoAddPolicy())
                else:
                    ssh.set_missing_host_key_policy(paramiko.client.AutoAddPolicy)
                ssh.connect(
                    hostname=self.hostname,
                    port=self.port,
                    timeout=25,
                    pkey=self.private_key,
                    look_for_keys=False,
                    allow_agent=False,
                    username=self.username,
                    password=self.password,
                )
                client = ssh.open_sftp()
                try:
                    if self.path:
                        client.chdir(self.path)
                        base_cwd = client._cwd
                        old_adjust_cwd = client._adjust_cwd

                        def _adjust_cwd(path):
                            path = old_adjust_cwd(path)
                            if not os.path.normpath(path).startswith(base_cwd):
                                raise ValueError(
                                    'all paths must be under base path %s: %s' % (base_cwd, path)
                                )
                            return path

                        client._adjust_cwd = _adjust_cwd
                    yield client
                finally:
                    client.close()
            finally:
                ssh.close()

    sftp = SFTP(sftp_url)
    with sftp.client() as client:
        files = sorted([name for name in client.listdir() if name.endswith('.txt') and name.count('_') == 5])
        logging.debug('Found %d files: %s', len(files), ', '.join(files))
        for file in files:
            with client.open(file, 'rt') as fd:
                lines = fd.readlines()
            real_lines = lines[1:-1]
            content = ''.join(real_lines)

            response = requests.put(
                wcs_url + '?update-mode=update',
                auth=(wcs_login, wcs_password),
                headers={'content-type': 'text/csv'},
                data=content.encode(),
            )
            if not response.ok or response.json()['err']:
                logging.error('Could not import file %s', file)
                continue
            logging.info('Imported %s', file)
            if do_move:
                client.rename(file, 'ARCHIVES/' + file)
    logging.info('Finished')


try:
    main()
except Exception:
    logging.exception('Script crashed')
