import os
import sys
import subprocess
import tempfile

from quixote import get_publisher
from wcs.carddef import CardDef
from wcs.formdef import FormDef
from wcs.qommon.storage import Equal
from wcs.qommon.upload_storage import PicklableUpload


tenant = os.path.basename(get_publisher().app_dir)

SETTINGS = '/etc/publik/clients/cd59/compile-pdf-convocations-%s.conf' % tenant
if not os.path.exists(SETTINGS):
    print('missing settings %s' % SETTINGS)
    sys.exit(1)

# settings par defaut (à personnaliser dans le fichier SETTINGS indiqué ci-dessus)

FORMDEF_URL_NAME = 'rendez-vous-orientation'  # nom du formulaire contenant les convocations PDF
# note : pour indiquer plusieurs formulaires, utiliser une liste de str
FIELD_VARNAME = 'courrier'  # identifiant du champ PDF dans ce formulaire
STATUS = 'RdV réservé'  # nom du statut des formdata à prendre en compte
TRIGGER = 'impression-ok'  # trigger à déclencher après prise en charge
CARDDEF_URL_NAME = 'convocations'  # nom de la fiche pour créer avec la concat des PDF
COPY = None  # fichier où copier le PDF généré
# exemple : COPY = '/var/lib/wcs/tenant.wcs/static/convocations-<clésecrète>.pdf'
# qui permet d'avoir une URL "secrète" du genre https://tenant.wcs/static/convocations-<clésecrète>.pdf

with open(SETTINGS) as fd:
    exec(fd.read())

formdefs = {}

# on cherche tous les éléments de la configuration

url_names = FORMDEF_URL_NAME
if not isinstance(url_names, list):
    url_names = [url_names]

for url_name in url_names:
    for formdef in FormDef.select():
        if formdef.url_name == url_name:
            break
    else:
        print('cannot find formdef %s' % url_name)
        sys.exit(1)

    for pdf_field in formdef.get_all_fields():
        if pdf_field.varname == FIELD_VARNAME:
            break
    else:
        print('cannot find field %s in formdef %s' % (FIELD_VARNAME, formdef))
        sys.exit(1)

    workflow = formdef.get_workflow()

    for status in workflow.possible_status:
        if status.name == STATUS:
            break
    else:
        print('cannot find status %s in workflow %s (formdef %s)' % (STATUS, workflow, formdef))
        sys.exit(1)

    for jump in status.items:
        if jump.key == 'jump' and jump.trigger == TRIGGER:
            break
    else:
        print(
            'cannot find jump trigger %s in status %s (workflow %s, formdef %s)'
            % (TRIGGER, status, workflow, formdef)
        )
        sys.exit(1)

    formdefs[url_name] = {'formdef': formdef, 'pdf_field': pdf_field, 'status': status, 'jump': jump}

for carddef in CardDef.select():
    if carddef.url_name == CARDDEF_URL_NAME:
        break
else:
    print('cannot find carddef %s' % CARDDEF_URL_NAME)
    sys.exit(1)

for create_pdf_field in carddef.get_all_fields():
    if create_pdf_field.key == 'file':
        break
else:
    print('cannot find file field in carddef %s' % carddef)
    sys.exit(1)


# c'est bon, on a tout, on y va

# récupération des PDF et concaténation via pdftk

with tempfile.TemporaryDirectory(prefix='cd59-convocations-pdf-') as tmpdirname:
    command = ['/usr/bin/pdftk']
    formdatas = []

    for url_name, f in formdefs.items():
        formdef = f['formdef']
        pdf_field = f['pdf_field']
        status = f['status']
        jump = f['jump']

        for formdata in formdef.data_class().select([Equal('status', 'wf-%s' % status.id)]):
            pdf = formdata.data.get(pdf_field.id)
            if not isinstance(pdf, PicklableUpload):
                # no file found in formdata
                continue
            filename = os.path.join(tmpdirname, '%s-%s.pdf' % (formdef.id, formdata.id))
            with open(filename, 'wb') as pdf_fd:
                pdf_fd.write(pdf.get_content())
            command.append(filename)
            formdatas.append((formdata, jump))

    if not formdatas:
        sys.exit(0)
    command += ['output', '-']
    run = subprocess.run(command, timeout=120, capture_output=True, cwd=tmpdirname, check=True)
    upload = PicklableUpload('output.pdf', 'application/pdf')
    upload.receive([run.stdout])

# création d'une fiche avec le PDF résultat

card = carddef.data_class()()
card.data = {create_pdf_field.id: upload}
card.just_created()
card.store()
card.perform_workflow()

# copie du PDF dans un fichier local

if COPY:
    with open(COPY, 'wb') as f:
        f.write(upload.get_content())

# jump sur chaque formulaire

for formdata, jump in formdatas:
    formdata.jump_status(jump.status)
    formdata.perform_workflow()
