# passerelle - © Entr'ouvert

import pytest

from passerelle.utils import eval as utils_eval
from passerelle.utils.eval import DynamicCheckError, StaticCheckError, safe_compile, safe_eval


@pytest.mark.parametrize(
    'expression',
    [
        '"aaa"',
        '11',
        '11 * 12',
        '11 - 12',
        'y - 12',
        'x.lower()',
        "[1 for x in '1' for y in '1']",
        'len(x)',
        'str(x)',
        'float(y)',
        'int(y)',
        'sorted(x)',
    ],
)
def test_ok(expression):
    safe_eval(expression, {'x': 'a', 'y': 10})


@pytest.mark.parametrize(
    'expression',
    [
        # pow operator is forbidden
        '11 ** 12',
        # comprehensions of depth more than 2 are forbidden
        "[1 for x in '1' for y in '1' for z in 'x']",
        "[[[1 for x in '1'] for y in '1'] for z in 'x']",
        # dict comprehensions
        '{x: y for x, y in b}',
        # name starting with an underscore are forbidden
        '_coin',
        '__import__',
        'x.__sizeof__',
        # assignment through comprehension target is blocked
        "[1 for x.i in '1']",
        # prevent dangerous methods
        'open()',
        'x.clear()',
        'x.discard(1)',
        "x.write('1')",
        'f"{open()}"',
        'x[open():1]',
    ],
)
def test_statically_bad_expression(expression):
    with pytest.raises(StaticCheckError, match=r'unauthorized'):
        safe_eval(expression, {'x': 'a', 'y': 10})


def test_statically_self_visible():
    with pytest.raises(StaticCheckError, match=r'unknown name'):
        safe_compile('a == 1', {'b'})


@pytest.mark.parametrize(
    'expression',
    [
        '"s" * 100000000000',
        '"s" * "y"',
        '[0] * 100000000000',
        '[x for x in l]',
        '100000000000000000000 * 1000000000000000000000000000',
    ],
)
def test_dynamically_bad_expression(expression, monkeypatch):
    monkeypatch.setattr(utils_eval, 'MAX_CACHE_SIZE', 2)
    with pytest.raises(DynamicCheckError, match=r'unauthorized'):
        safe_eval(expression, {'l': [1] * 101})


@pytest.mark.parametrize(
    'expression',
    [
        'text',
        "text == query ['text']",
        "(id == query.get('etab1')) or (id == query.get('etab2')) or (id == query.get('etab3'))",
        'id',
        "text== query['discipline']",
        "code+' / '+libelle",
        "query.get('day', '') >= debut and query.get('day', '') <= fin",
        'id',
        "query.get('code_postal') in ['13200']",
        'justificatif_domicile == "O"',
        "int(query.get('numero', '0')) <= int(numero_fin)",
        "(int(query.get('numero', '0')) % 2) == int(parite)",
        "idpresta == query['id_prestation']",
        'codeinsee',
        'id',
        'text+", "+salle+", "+capacite+" personnes maximum"',
        'cp',
        "cp == query['cp']",
        'text',
        'id',
        'disabled!=1',
        'cp',
        "cp == query['cp']",
        'cp',
        "cp == query['cp']",
        "text == query ['text']",
        'id',
        "text + ' (' + commune2019 + ')'",
        'nbhabitants|numeric',
        'text,cp,commune',
        'cp',
        "cp == query['cp']",
        'text',
        'id',
        "type == query['type']",
        "type == query['type']",
        'id',
        "id == query['id']",
        'id',
        "niveau == query ['niveau']",
        'niveau',
        "zipcode + ' - ' + text",
        "text == query['text']",
        'id',
        "text == query['a'] and academie == query['ac']",
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        'metro == query.get("metro")',
        "sstheme == query['sstheme']",
        'id',
        "slug == query.get('slug')",
        'text',
        'code_postal + " - " + text',
        'text+" : "+reponse_type_l1+" - "+reponse_type_l2',
        "epci_slug == query['epci']",
        'text',
        'text.replace("É" , "E")',
        'text[4:-4]',
        "Service == query['service']",
        'id',
        "text+' '+Prenom",
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        'cp',
        'cp+"-"+text',
        "motcle== query['theme']",
        'id',
        'text+", "+salle+", "+capacite+" personnes maximum"',
        "Fonction== query['Fonction']",
        'id,data1',
        'email',
        "email.lower() == query['email'].lower()",
        'id',
        "entite == query['entite']",
        "type_personne == query['type_personne']",
        'libelle_acte',
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        'id',
        "id == query['id']",
        'text+ " (" + code_postal + ")"',
        'destinataire == query.get("destinataire")',
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        'text+" - "+adresse+" -  "+ville',
        '"slug_formulaire" not in query or slug_formulaire == query.get("slug_formulaire")',
        'slug_formulaire == query.get("slug_formulaire") and statut == query.get("statut")',
        "role_n1.startswith('GC')",
        "int(query.get('revenu')) >= int(debut)",
        "query.get('type') == type",
        'activite_id',
        "id_thematique == query['id_thematique'] if 'id_thematique' in query else True",
        'disponible == "oui"',
        'id_lieu == "rdv-composteur" or id_lieu == query[\'id_lieu\'] if \'id_lieu\' in query else True',
        'id_thematique + "-" + id_preference == query[\'id\'] if \'id\' in query else True',
        'id_thematique +"-"+ id_preference',
        'text',
        "idemandeur == query['idemandeur']",
        'id_cat',
        "id_cat == query['cat']",
        'id',
        "id == query['insee']",
        'id',
        'dga',
        'affichage_liste_deroulante != "Non"',
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        'text',
        "ligne == query['id']",
        'cp',
        "cp == query['cp']",
        'slug',
        '"professionnel_referent" not in query or text == query.get("professionnel_referent")',
        'text == query.get("ecole")',
        'commune + "-" + text',
        'id',
        '"professionnel_referent" not in query or text == query.get("professionnel_referent")',
        'text == query.get("ecole")',
        'commune + "-" + text',
        'code_commune_INSEE',
        "id == query['id']",
        "int(query['num']) <= int(num_max)",
        'text',
        'text + " (" + dep + ")"',
        'codepostal',
        'categorie == query["categorie"] and niveau == "destination"',
        'type_demande in (query["type_demande"], "COMMUN")',
        'id',
        'fr+" ("+id+")"',
        'code_ada',
        'type == "CE"',
        "filtre7 == 'o'",
        'id',
        "id == query['code']",
        'code_commune',
        "code_postal + ' ' + libelle_commune",
        "( uk == query.get('uk') ) if ('uk' in query) else True",
        "text == query['text']",
        'classe',
        'id',
        "NORME_POSTALE==query['voie']",
        'text.replace("\'", "")==query[\'lieu_de_vote\']',
        'fr+" ("+id+")"',
        '" ".join([Num,Libelle, Codpos])',
        'text.replace("\'", "")',
        '"mat " + Ecole if Type == "MAT" else "elem " + Ecole',
        "statut == query['statut'] and electeur == query['electeur']",
        'poste,',
        'f"{num} {Lib_Off}"',
        'id',
        "nature==query['nature']",
        "genre == query['genre']",
        'id',
        "(query['adulte'] == 'true' and text == 'adulte')  or (query['annee'] in text.split(','))",
        'art + " " + text',
        'id',
        'text + " (" + codepostal + ")"',
        "CP == query['CP']",
        'type + " à " + lieu + " (" + pays + ")"',
        'text',
        "ufr == query['ufr'] and niveau == query['niveau']",
        "text==query['signalement']",
        "ext == query['critere1']",
        'text',
        '"mat " + Ecole if Type == "MAT" else "elem " + Ecole',
        "ext == query['critere1']",
        'text',
        '"mat " + Ecole if Type == "MAT" else "elem " + Ecole',
        '"nom_commune" not in query or text == query.get("nom_commune")',
        'id',
        'code',
        "text==query['text']",
        "type_rdv==query.get('type_rdv') and nb_personnes==query.get('nb_personnes')",
        'code-uo.startswith(15)',
        'code-uo',
        'text',
        "ecole_id == query['ecole_id']",
        'id',
        "id+' '+text",
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        "(date_debut >= query.get('date_debut')) and (date_fin <=  query.get('date_fin'))",
        '"Semaine "+num_semaine+" - Du "+date_debut_texte+" au "+date_fin_texte',
        "(per_ville == query.get('per_ville') or per_agglo == query.get('per_agglo') or per_rdle == query.get('per_rdle'))",
        'competence + " - " + text + " - " + soustype',
        "query.get('commune') == commune",
        "ecole_id == query['ecole_id']",
        'id',
        "(date_debut >= query.get('date_debut')) and (date_fin <=  query.get('date_fin'))",
        '"Semaine "+num_semaine+" - Du "+date_debut_texte+" au "+date_fin_texte',
        "query.get('commune') == commune",
        "categorie == 'Art et culture'",
        'ville',
        'cp.rjust(5,\'0\') + " - " + text',
        'text + " (" + codepostal + ")"',
        'codepostal',
        "type == query['type']",
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        'email',
        "email == query['email']",
        'referent',
        "text == query.get('text')",
        'text + "-" + sous_type',
        'id',
        "text == query.get('text')",
        "text == query.get('text') and nivo == query.get('nivo') and theme == query.get('theme')",
        '"categorie" not in query or text == query.get("categorie")',
        'text,contact',
        'id',
        "nom == query['nom']",
        'categorie',
        "type == query['type']",
        'id',
        'semaine == query["text"]',
        "text == query['text']",
        'direction',
        "affectation == query['affectation']",
        'affectation',
        'categorie == "cat1"',
        "'toto'",
        'nomcat',
        'int(id)',
        'employeur',
        "employeur == query.get('employeur')",
        'slugquartier == query.get("slugquartier")',
        '"slug_formulaire" not in query or slug_formulaire == query.get("slug_formulaire")',
        'text',
        'jour+" - "+text+" - Produits "+type_produit',
        "code_postal == query['code_postal']",
        'text',
        'code_postal + " - " + text',
        'capacite_min + " - " + capacite_max + " places"',
        "normalize(query.get('voie', '').lower()) == normalize(text.lower())",
        '(parite == "" or int(\'\'.join([s for s in query.get(\'numero\', \'0\').split() if s.isdigit()]) or \'1\') % 2 == int(parite))',
        'ponctuel == query.get("ponctuel") and int(capacite_min) <= int(query.get("capacite")) and int(capacite_max) >= int(query.get("capacite"))',
        "ressourcesEuro == query.get('ressourcesEuro')",
        'aide',
        "query['domaine']==text",
        'int(ordre or 10000)',
        'formulaire+" / "+selection',
        'int(id) < int("1000")',
        'id',
        'cp+" - "+ville',
        'int(secteur)',
        "id == query ['insee']",
        "(id == query.get('etab1')) or (id == query.get('etab2')) or (id == query.get('etab3'))",
        'id',
        "text== query['discipline']",
        "code+' / '+libelle",
        "idpresta == query['id_prestation']",
        'codeinsee',
        'text',
        'id',
        'disabled!=1',
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        'metro == query.get("metro")',
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        "text == query.get('text')",
        "query.get('day', '') >= debut and query.get('day', '') <= fin",
        'id',
        "query.get('code_postal') in ['13200']",
        'justificatif_domicile == "O"',
        "int(query.get('numero', '0')) <= int(numero_fin)",
        "(int(query.get('numero', '0')) % 2) == int(parite)",
        "code_postal == query['code_postal']",
        'text',
        'code_postal + " - " + text',
        'capacite_min + " - " + capacite_max + " places"',
        "normalize(query.get('voie', '').lower()) == normalize(text.lower())",
        '(parite == "" or int(\'\'.join([s for s in query.get(\'numero\', \'0\').split() if s.isdigit()]) or \'1\') % 2 == int(parite))',
        'ponctuel == query.get("ponctuel") and int(capacite_min) <= int(query.get("capacite")) and int(capacite_max) >= int(query.get("capacite"))',
        "Type==query['type_ecole']",
        'RNE',
        '"mat " + Ecole if Type == "MAT" else "elem " + Ecole',
        "id == query['id']",
        'id',
        'referent',
        "text == query['centre']",
        'id',
        "int(debut) <= int(query['debut']) and int(fin)>=int(query['debut'])",
        '"type" not in query or text == query.get("type")',
        "type == query['type']",
        'id',
        "ecole_id == query['ecole_id']",
        'id',
        'J+" - "+text',
        "(date_debut >= query.get('date_debut')) and (date_fin <=  query.get('date_fin'))",
        '"Semaine "+num_semaine+" - Du "+date_debut_texte+" au "+date_fin_texte',
        "query.get('commune') == commune",
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        "id==query['siret']",
        'id',
        'employeur',
        "employeur == query['employeur']",
        "id == query['id']",
        'id',
        "id == query['id'] and famille == query['mdp']",
        "zipcode + ' - ' + text",
        "Service == query['service']",
        'id',
        "text+' '+Prenom",
        'dga',
        'affichage_liste_deroulante != "Non"',
        'id',
        'cp',
        "cp == query['cp']",
        "motcle== query['theme']",
        'motcle',
        'text',
        "text == query['text']",
        'nom, prenom',
        "idemandeur == query['idemandeur']",
        'text',
        'id',
        'text+ " (" + code_postal + ")"',
        'code_postal+" "+text',
        "domaine ==  query [ 'domaine']",
        'slug',
        '"professionnel_referent" not in query or text == query.get("professionnel_referent")',
        'text == query.get("ecole")',
        'commune + "-" + text',
        "domaine == query['domaine']",
        'text',
        'text + " (" + dep + ")"',
        'codepostal',
        'id',
        'fr+" ("+id+")"',
        'code_ada',
        'type == "CE"',
        "genre == query['genre']",
        'id',
        "categorie == 'Art et culture'",
        'text',
        'cp.rjust(5,\'0\') + " " + text',
        'id',
        'text+", "+salle+", "+capacite+" personnes maximum"',
        'id',
        "NORME_POSTALE==query['voie']",
        'text.replace("\'", "")==query[\'lieu_de_vote\']',
        'fr+" ("+id+")"',
        'fr+" / "+en',
        'text.replace("\'", "")',
        '"mat " + Ecole if Type == "MAT" else "elem " + Ecole',
        "statut == query['statut'] and electeur == query['electeur']",
        'poste,',
        "categorie == query.get('categorie')",
        'text',
        "Nom+' '+Prenom",
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        'metro == query.get("metro")',
        "code_insee==query['insee']",
        'id',
        "id+' '+text",
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        "(per_ville == query.get('per_ville') or per_agglo == query.get('per_agglo') or per_rdle == query.get('per_rdle'))",
        'competence + " - " + text + " - " + soustype',
        "int(query.get('qf')) >= int(qfmin) and int(query.get('qf')) <= int(qfmax)",
        "query.get('commune') == commune",
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        'type_demande in (query["type_demande"], "COMMUN")',
        'cp',
        "cp == query['cp']",
        'cp',
        "cp == query['cp']",
        'cp',
        "cp == query['cp']",
        'activite_id',
        "identifiant == query['identifiant'] if 'identifiant' in query else True",
        'disponible == "oui"',
        'id_lieu == "rdv-composteur" or id_lieu == query[\'id_lieu\'] if \'id_lieu\' in query else True',
        'id_thematique + "-" + id_preference == query[\'id\'] if \'id\' in query else True',
        'id_thematique +"-"+ id_preference',
        'iddirection',
        "idrue == query['idrue']",
        "int(query['numero']) > int(de)",
        "id == query['id']",
        'id',
        "int(query['num']) <= int(num_max)",
        'email',
        "email == query['email']",
        'id_cat',
        "id_cat == query['cat']",
        "( uk == query.get('uk') ) if ('uk' in query) else True",
        'ordre',
        'art + " " + text',
        'id',
        'text + " (" + codepostal + ")"',
        "CP == query['CP']",
        'type + " à " + lieu + " (" + pays + ")"',
        'code',
        "code==query.get('code')",
        'id',
        "epci_slug == query['epci']",
        'text',
        'text.replace("É" , "E")',
        'cp',
        "cp == query['cp']",
        "sousobjet_slug == query['sousobjet_slug']",
        'id',
        'id',
        'cp+" - "+ville',
        'int(secteur)',
        "id == query ['insee']",
        "text == query['text']",
        'direction',
        'desactive == ""',
        'id',
        'slugquartier == query.get("slugquartier")',
        '"slug_formulaire" not in query or slug_formulaire == query.get("slug_formulaire")',
        'text',
        'jour+" - "+text+" - Produits "+type_produit',
        'destinataire == query.get("destinataire")',
        "code_postal.startswith(query['q']) if query.get('q', '').isdigit() else True",
        'code_postal + " - " + text',
        'id',
        '"slug_formulaire" not in query or slug_formulaire == query.get("slug_formulaire")',
        'slug_formulaire == query.get("slug_formulaire") and statut == query.get("statut")',
        "role_n1.startswith('GC')",
        "int(query.get('revenu')) >= int(debut)",
        "query.get('type') == type",
        'id',
        "text == query.get('text')",
        "text == query.get('text') and nivo == query.get('nivo') and theme == query.get('theme')",
        'text,jauge,horaire',
        "adhesion == query['adhesion']",
        'text',
        "text == query.get('text')",
        'nom',
        'int(ordre or 10000)',
        'int(id) < int("1000")',
    ],
)
def test_statically_good_expressions(expression):
    safe_compile(expression, authorized_functions=['normalize'])
