# Copyright (C) 2025 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import uuid
from base64 import b64decode
from os.path import splitext
from urllib.parse import urlparse

from django.conf import settings
from django.db import models, transaction
from django.http import HttpResponse
from django.shortcuts import get_object_or_404
from django.urls import reverse
from django.utils.http import urlencode
from django.utils.translation import gettext_lazy as _
from requests import ConnectionError, RequestException, Timeout

from passerelle.base.models import BaseResource, OAuth2Resource
from passerelle.base.signature import sign_url
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError

UUID_PATTERN = '(?P<uuid>[0-9|a-f]{8}-[0-9|a-f]{4}-[0-9|a-f]{4}-[0-9|a-f]{4}-[0-9a-f]{12})'

ALERT_SCHEMA = {
    'type': 'object',
    'title': 'CITEOS alert',
    'description': '',
    'required': [
        'name',
        'date',
        'address',
        'coordinates',
    ],
    'properties': {
        'name': {'description': _('Alert name'), 'type': 'string'},
        'date': {
            'description': _('Alert date'),
            'type': 'string',
            'format': 'date',
        },
        'address': {'description': _('Alert address'), 'type': 'string'},
        'status_trigger': {'description': _('Status update trigger URL'), 'type': 'string'},
        'coordinates': {
            'description': _('Coordinates'),
            'type': 'object',
            'properties': {
                'lat': {'type': 'number'},
                'lon': {'type': 'number'},
            },
            'additionalProperties': False,
        },
        'comment': {'description': _('Comment'), 'type': 'string'},
        'documents': {
            'description': _('Documents'),
            'type': 'array',
            'items': {
                'type': 'object',
                'required': ['content'],
                'properties': {
                    'content': {
                        'type': 'string',
                    },
                    'content_type': {
                        'type': 'string',
                    },
                    'filename': {
                        'type': 'string',
                    },
                },
            },
        },
    },
}

UPDATE_ALERT_SCHEMA = {
    'type': 'object',
    'title': 'Update citeos alert schema',
    'description': '',
    'required': [
        'uuid',
        'status',
    ],
    'properties': {
        'uuid': {'description': _('Alert UUID'), 'type': 'string', 'pattern': UUID_PATTERN},
        'status': {'description': _('New status'), 'type': 'string'},
    },
}


class CiteosConnector(BaseResource, OAuth2Resource):
    add_alert_url = models.URLField(_('Add Alert Endpoint'))
    category = _('Business Process Connectors')

    _can_update_alert_status_description = _('Updating alert status is limited to the following API users:')

    class Meta:
        verbose_name = 'CITEOS'

    @classmethod
    def get_verbose_name(cls):
        return cls._meta.verbose_name

    @endpoint(
        name='add-alert',
        description=_('Create an alert'),
        post={'request_body': {'schema': {'application/json': ALERT_SCHEMA}}},
    )
    def add_alert(self, request, post_data):
        with transaction.atomic():
            documents = []
            alert = self.alerts.create(
                status_update_trigger_url=post_data.get('status_update_trigger_url', None)
            )

            for file in post_data.get('documents', []):
                document = self.documents.create(
                    alert=alert, content=b64decode(file['content']), content_type=file.get('content_type')
                )
                filename, extension = splitext(file['filename'])
                documents.append(
                    {
                        'libelle': filename,
                        'extension': extension,
                        'url': document.get_url(request),
                        'source': 'publik',
                    }
                )

            try:
                response = self.requests.post(
                    self.add_alert_url,
                    json=[
                        {
                            'sourcecode': str(alert.uuid),
                            'name': post_data['name'],
                            'sourcesystem': 'publik',
                            'signaldate': post_data['date'],
                            'field': 'EP',
                            'priority': 1,
                            'address': post_data['address'],
                            'coordonnees': f"POINT ({post_data['coordinates']['lat']} {post_data['coordinates']['lon']})",
                            'comment': post_data['comment'],
                            'documents': documents,
                        }
                    ],
                )

                json = response.json()

                if not response.ok:
                    raise APIError(f'Citeos error : {json.get("message")}')

                return {'data': {'alert_uuid': alert.uuid}}
            except (ConnectionError, Timeout) as e:
                raise APIError('HTTP request failed: %s' % e)

    @endpoint(
        name='get-document',
        description=_('Download a document'),
        perm='OPEN',
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('Document identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def get_document(self, request, uuid):
        document = get_object_or_404(self.documents, uuid=uuid)
        return HttpResponse(content=document.content, content_type=document.content_type)

    @endpoint(
        name='update-alert-status',
        perm='can_update_alert_status',
        description=_('Update alert status'),
        post={'request_body': {'schema': {'application/json': UPDATE_ALERT_SCHEMA}}},
    )
    def update_alert_status(self, request, post_data):
        self.add_job('status_update_job', alert_uuid=post_data['uuid'], status=post_data['status'])

    def status_update_job(self, alert_uuid, status):
        alert = self.alerts.filter(uuid=alert_uuid).first()

        if alert is None:
            self.logger.error('Trying to update status of unknown alert %s', alert_uuid)
            return

        alert.trigger_status_update(status)


class Alert(models.Model):
    uuid = models.UUIDField(verbose_name=_('UUID'), unique=True, default=uuid.uuid4)
    status_update_trigger_url = models.CharField('status-update-trigger-url', blank=True, null=True)
    resource = models.ForeignKey(CiteosConnector, on_delete=models.CASCADE, related_name='alerts')

    def trigger_status_update(self, status):
        url = self.get_trigger_status_update_url()
        if not url:
            return

        try:
            response = self.resource.requests.post(
                url,
                json={
                    'status': status,
                },
            )
            response.raise_for_status()

        except RequestException as exception:
            self.resource.logger.error('Error while triggering %s: %s', url, exception)
            raise

    def get_trigger_status_update_url(self):
        trigger_url = self.status_update_trigger_url
        if not trigger_url:
            return None

        scheme, netloc, _, _, _, _ = urlparse(trigger_url)
        services = settings.KNOWN_SERVICES.get('wcs', {})
        for service in services.values():
            remote_url = service.get('url')
            service_scheme, service_netloc, _, _, _, _ = urlparse(remote_url)
            if service_scheme == scheme and service_netloc == netloc:
                orig = service.get('orig')
                if orig:
                    trigger_url = (
                        trigger_url + ('&' if '?' in trigger_url else '?') + urlencode({'orig': orig})
                    )
                secret = service.get('secret')
                return sign_url(trigger_url, secret)

        self.resource.logger.error("Can't find a suitable configured WCS service for url %s", trigger_url)

        return None


class Document(models.Model):
    uuid = models.UUIDField(verbose_name=_('UUID'), unique=True, default=uuid.uuid4)
    alert = models.ForeignKey(Alert, on_delete=models.CASCADE, related_name='documents')
    resource = models.ForeignKey(CiteosConnector, on_delete=models.CASCADE, related_name='documents')
    content = models.BinaryField('content')
    content_type = models.CharField('content-type', blank=True, null=True)

    def get_url(self, request):
        relative_url = reverse(
            'generic-endpoint',
            kwargs={
                'slug': self.alert.resource.slug,
                'connector': self.alert.resource.get_connector_slug(),
                'endpoint': 'get-document',
                'rest': str(self.uuid),
            },
        )
        return request.build_absolute_uri(relative_url)
