# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import os

import pytest
import responses
from django.core.cache import cache
from zeep import Settings

from passerelle.contrib.selligent.models import PARAMS_SEPARATOR, Selligent
from passerelle.utils.jsonresponse import APIError
from tests.utils import ResponsesSoap, generic_endpoint_url, setup_access_rights

pytestmark = pytest.mark.django_db


@pytest.fixture
def requests_mock():
    return responses.RequestsMock(assert_all_requests_are_fired=False)


def get_xml_file(filename):
    with open(os.path.join(os.path.dirname(__file__), 'data', 'selligent', filename), 'rb') as desc:
        return desc.read()


@pytest.fixture()
def session_service(requests_mock):
    responses_soap = ResponsesSoap(
        wsdl_url='http://selligent.example.org/CDMWebServices/SessionServices.asmx?WSDL',
        wsdl_content=get_xml_file('SessionServices.xml'),
        settings=Settings(strict=False, xsd_ignore_sequence_order=True),
        requests_mock=requests_mock,
    )
    with responses_soap() as soap_mock:
        yield soap_mock


@pytest.fixture()
def script_service(requests_mock):
    responses_soap = ResponsesSoap(
        wsdl_url='http://selligent.example.org/CDMWebServices/ScriptServices.asmx?WSDL',
        wsdl_content=get_xml_file('ScriptServices.xml'),
        settings=Settings(strict=False, xsd_ignore_sequence_order=True),
        requests_mock=requests_mock,
    )
    with responses_soap() as soap_mock:
        yield soap_mock


@pytest.fixture
def connector(db):
    return setup_access_rights(
        Selligent.objects.create(
            slug='test',
            base_api_url='http://selligent.example.org/CDMWebServices/',
            user='foo',
            password='secret',
            database='example',
        )
    )


def test_get_session_token(app, connector, session_service):
    session_service.add_soap_response('Login', get_xml_file('SessionToken.xml'))
    connector.check_status()
    assert connector.session_token == 'selligent_rec-mgl01-acee7f17-2fd5-4b9c-ae19-1ecf65e08c22'
    assert len(session_service.soap_requests) == 1
    soap_request = session_service.soap_requests[0]
    assert type(soap_request).__name__ == 'Login'
    # check if the token is in cache
    assert connector.session_token == 'selligent_rec-mgl01-acee7f17-2fd5-4b9c-ae19-1ecf65e08c22'
    assert len(session_service.soap_requests) == 1

    # token get failure
    cache.clear()
    session_service.replace_soap_response('Login', get_xml_file('SessionTokenFailure.xml'))
    with pytest.raises(APIError, match="Nom d'utilisateur ou mot de passe incorrect."):
        assert connector.session_token


def test_parameters_build(app, connector):
    assert (
        connector.build_params_from_payload({'field': 'value\nnew line&other ending with;'})
        == PARAMS_SEPARATOR
        + 'field=valueret-char1otnew lineesperluettother ending withpo1-virgul'
        + PARAMS_SEPARATOR
    )


def test_create_demand(app, connector, session_service, script_service):
    session_service.add_soap_response('Login', get_xml_file('SessionToken.xml'))
    script_service.add_soap_response('ExecuteServerScript', get_xml_file('DemandCreationFailure.xml'))
    payload = {
        'CU_PRENOM': 'Bar',
        'CU_EMAIL': 'foo.bar@example.org',
    }
    url = generic_endpoint_url('selligent', 'create-demand', slug=connector.slug)
    resp = app.post_json('%s?scriptNrid=27821677706740&code_prestation=xyz' % url, payload)
    assert len(session_service.soap_requests) == 1
    assert len(script_service.soap_requests) == 1
    soap_request = script_service.soap_requests[0]
    assert soap_request.sessionParams == 'selligent_rec-mgl01-acee7f17-2fd5-4b9c-ae19-1ecf65e08c22'
    assert soap_request.scriptNrid == '27821677706740'
    assert soap_request.paramsValue.startswith('xyz;|$|')
    err_json = resp.json
    assert err_json['err']
    assert err_json['err_desc'].startswith('ERREUR')
    assert 'CU_NOM est obligatoire' in err_json['err_desc']
    payload['CU_NOM'] = 'Foo'
    script_service.replace_soap_response('ExecuteServerScript', get_xml_file('DemandCreation.xml'))
    resp = app.post_json('%s?scriptNrid=27821677706740&code_prestation=xyz' % url, payload)
    assert len(session_service.soap_requests) == 1
    assert len(script_service.soap_requests) == 2
    assert resp.json == {'data': 'D-20211123-0001', 'err': 0}


def test_add_demand_file(app, connector, session_service, script_service):
    session_service.add_soap_response('Login', get_xml_file('SessionToken.xml'))
    script_service.add_soap_response('ExecuteServerScript', get_xml_file('DemandCreation.xml'))
    payload = {'CU_PRENOM': 'Bar', 'CU_EMAIL': 'foo.bar@example.org', 'CU_NOM': 'Foo'}

    url = generic_endpoint_url('selligent', 'create-demand', slug=connector.slug)
    resp = app.post_json('%s?scriptNrid=27821677706740&code_prestation=xyz' % url, payload)
    demand_number = resp.json['data']
    assert len(session_service.soap_requests) == 1
    assert len(script_service.soap_requests) == 1

    script_service.add_soap_response('ExecuteServerScript', get_xml_file('AddFileFailure.xml'))

    file_payload = {
        'demand_number': 'xxx',
        'file': {'filename': 'test.txt', 'content': 'VEVTVCBFTlZPSSBQSi4='},
        'file_type': 'AUTRE_JUSTIF',
    }
    url = generic_endpoint_url('selligent', 'add-demand-file', slug=connector.slug)
    resp = app.post_json('%s?scriptNrid=32071193360350' % url, file_payload)
    assert len(script_service.soap_requests) == 3
    soap_request = script_service.soap_requests[2]
    assert soap_request.scriptNrid == '32071193360350'
    err_json = resp.json
    assert err_json['err']
    assert "s_WS_AttachPJbyRef' not found." in err_json['err_desc']

    file_payload['demand_number'] = demand_number
    script_service.replace_soap_response('ExecuteServerScript', get_xml_file('AddFile.xml'))
    resp = app.post_json('%s?scriptNrid=32071193360350' % url, file_payload)
    assert len(script_service.soap_requests) == 4
    assert resp.json == {'data': 'OK', 'err': 0}
