# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import json
import urllib
from unittest.mock import Mock

import httmock
import pytest
from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from requests import ConnectionError, Timeout

from passerelle.apps.citeos.models import CiteosConnector
from passerelle.base.models import AccessRight, ApiUser
from passerelle.base.signature import check_url
from tests.utils import generic_endpoint_url, setup_access_rights


@pytest.fixture()
def connector(db):
    return setup_access_rights(
        CiteosConnector.objects.create(
            slug='test',
            add_alert_url='http://example.org/add-alert',
            oauth2_token_url='http://example.org/auth',
            oauth2_username='client-id',
            oauth2_password='client-secret',
        )
    )


@httmock.urlmatch(netloc='^example.org$', path='^/auth')
def mock_auth(url, request):
    auth_data = urllib.parse.parse_qs(request.body)
    assert auth_data['grant_type'][0] == 'client_credentials'
    return httmock.response(200, {'access_token': 'ACCESS_TOKEN'})


@httmock.urlmatch()
def error_handler(url, request):
    assert False, 'should not be reached'


def test_add_alert(app, connector):
    endpoint = generic_endpoint_url('citeos', 'add-alert', slug=connector.slug)

    @httmock.urlmatch(netloc='^example.org$', path='^/add-alert')
    def mock_add_alert(url, request):
        post_data = json.loads(request.body)
        assert len(post_data) == 1

        alert = post_data[0]
        assert alert['name'] == 'alert-name'
        assert alert['signaldate'] == '2023-01-01T00:00:00.000Z'
        assert alert['address'] == 'alert-address'
        assert alert['coordonnees'] == 'POINT (10.0 20.0)'
        assert alert['field'] == 'EP'
        assert alert['priority'] == 1
        assert alert['sourcesystem'] == 'publik'
        assert alert['comment'] == 'test comment'
        assert alert['documents'] == []
        return httmock.response(200, {'insterted': '1'})

    with httmock.HTTMock(mock_add_alert, mock_auth, error_handler):
        response = app.post_json(
            endpoint,
            params={
                'name': 'alert-name',
                'date': '2023-01-01T00:00:00.000Z',
                'address': 'alert-address',
                'coordinates': {'lat': 10.0, 'lon': 20.0},
                'comment': 'test comment',
            },
        )
        assert response.status_code == 200


def test_add_alert_with_documents(app, connector):
    endpoint = generic_endpoint_url('citeos', 'add-alert', slug=connector.slug)
    document_url = None

    @httmock.urlmatch(netloc='^example.org$', path='^/add-alert')
    def mock_add_alert(url, request):
        nonlocal document_url
        post_data = json.loads(request.body)
        alert = post_data[0]
        document = alert['documents'][0]
        assert document['libelle'] == 'foo'
        assert document['extension'] == '.jpg'
        assert document['source'] == 'publik'
        document_url = document['url']
        return httmock.response(200, {'insterted': '1'})

    with httmock.HTTMock(mock_add_alert, mock_auth, error_handler):
        response = app.post_json(
            endpoint,
            params={
                'name': 'alert-name',
                'date': '2023-01-01T00:00:00.000Z',
                'address': 'alert-address',
                'coordinates': {'lat': 10.0, 'lon': 20.0},
                'comment': 'test comment',
                'documents': [{'content': 'Rm9vIGJhcg==', 'content_type': 'raw/text', 'filename': 'foo.jpg'}],
            },
        )
        assert response.status_code == 200

    assert document_url is not None

    document_response = app.get(document_url)
    assert document_response.headers['Content-Type'] == 'raw/text'
    assert document_response.body == b'Foo bar'


@pytest.mark.parametrize('error', [ConnectionError('error message'), Timeout('error message')])
def test_citeos_connection_error(app, connector, error):
    endpoint = generic_endpoint_url('citeos', 'add-alert', slug=connector.slug)

    @httmock.urlmatch(netloc='^example.org$', path='^/add-alert')
    def mock_add_alert(url, request):
        raise error

    with httmock.HTTMock(mock_add_alert, mock_auth, error_handler):
        response = app.post_json(
            endpoint,
            params={
                'name': 'alert-name',
                'date': '2023-01-01T00:00:00.000Z',
                'address': 'alert-address',
                'coordinates': {'lat': 10.0, 'lon': 20.0},
                'comment': 'test comment',
            },
        )

        assert response.status_code == 200
        assert response.json['err'] == 1
        assert response.json['err_desc'] == 'HTTP request failed: error message'

    # Transaction should've reverted created alerts
    assert connector.alerts.count() == 0


def test_citeos_api_error(app, connector):
    endpoint = generic_endpoint_url('citeos', 'add-alert', slug=connector.slug)

    @httmock.urlmatch(netloc='^example.org$', path='^/add-alert')
    def mock_add_alert(url, request):
        return httmock.response(400, {'message': 'Error message'})

    with httmock.HTTMock(mock_add_alert, mock_auth, error_handler):
        response = app.post_json(
            endpoint,
            params={
                'name': 'alert-name',
                'date': '2023-01-01T00:00:00.000Z',
                'address': 'alert-address',
                'coordinates': {'lat': 10.0, 'lon': 20.0},
                'comment': 'test comment',
            },
        )

        assert response.status_code == 200
        assert response.json['err'] == 1
        assert response.json['err_desc'] == 'Citeos error : Error message'

    # Transaction should've reverted created alerts
    assert connector.alerts.count() == 0


@pytest.fixture
def mock_wcs_settings():
    settings.KNOWN_SERVICES = {
        'wcs': {
            'default': {
                'title': 'test',
                'url': 'https://wcs.com',
                'secret': 'xxx',
                'orig': 'passerelle',
            },
            'other': {
                'title': 'test',
                'url': 'https://other-wcs.com',
                'secret': 'blah',
                'orig': 'passerelle',
            },
        }
    }


def test_update_status(app, connector):
    endpoint = generic_endpoint_url('citeos', 'add-alert', slug=connector.slug)

    @httmock.urlmatch(netloc='^example.org$', path='^/add-alert')
    def mock_add_alert(url, request):
        return httmock.response(200, {'insterted': '1'})

    with httmock.HTTMock(mock_add_alert, mock_auth, error_handler):
        response = app.post_json(
            endpoint,
            params={
                'name': 'alert-name',
                'date': '2023-01-01T00:00:00.000Z',
                'address': 'alert-address',
                'coordinates': {'lat': 10.0, 'lon': 20.0},
                'comment': 'test comment',
                'status_update_trigger_url': 'http://example.org/update-status',
            },
        )
        assert response.status_code == 200
        alert_uuid = response.json['data']['alert_uuid']

    external_user = ApiUser.objects.create(username='external', keytype='API', key='external-key')
    AccessRight.objects.create(
        codename='can_update_alert_status',
        apiuser=external_user,
        resource_type=ContentType.objects.get_for_model(connector),
        resource_pk=connector.pk,
    )

    update_status_endpoint = generic_endpoint_url('citeos', 'update-alert-status', slug=connector.slug)
    response = app.post_json(
        update_status_endpoint + '?apikey=external-key',
        params={
            'uuid': alert_uuid,
            'status': 'new status',
        },
    )
    assert response.status_code == 200

    settings.KNOWN_SERVICES = {
        'wcs': {
            'default': {
                'title': 'test',
                'url': 'http://example.org',
                'secret': 'xxx',
                'orig': 'passerelle',
            },
        }
    }

    @httmock.urlmatch(netloc='^example.org$', path='^/update-status')
    def mock_wcs(url, request):
        post_data = json.loads(request.body)
        assert post_data['status'] == 'new status'
        assert check_url(request.url, settings.KNOWN_SERVICES['wcs']['default']['secret'])
        return httmock.response(200, {})

    spy_wcs = Mock(wraps=mock_wcs)

    with httmock.HTTMock(spy_wcs, error_handler):
        connector.jobs()
        spy_wcs.assert_called_once()
