/* global clients */
const _EVENTS = 'events'
const _CERTIFICATES = 'certificates'

let db = null

// We could store that in an indexed db, so that the whole event list doesn't
// get refreshed each time we relaunch the browser.
let lastUpdateTimestamp = 0
const tallyUrls = new Set()

self.addEventListener('activate', async () => {
  try {
    console.log('Activating QR code service worker')
    db = await openIndexedDb()
    clients.claim()
  } catch (error) {
    console.log(error)
  }
})

self.addEventListener('message', async (event) => {
  try {
    const tallyUrl = event.data.refreshTally
    if (tallyUrl) {
      tallyUrls.add(tallyUrl)
      refreshTally(tallyUrl)
    }
  } catch (error) {
    console.log(error)
  }
})

self.addEventListener('fetch', (event) => {
  try {
    if (tallyUrls.has(event.request.url)) {
      event.respondWith(tally(event.request))
    }
  } catch (error) {
    console.log(error)
  }
})

function openIndexedDb () {
  return new Promise((resolve, reject) => {
    try {
      const openDbRequest = indexedDB.open('QrCodeReader', 1)

      openDbRequest.onupgradeneeded = (event) => {
        console.log('Upgrading database')
        const db = event.target.result
        const eventStore = db.createObjectStore(_EVENTS, { keyPath: 'uuid'})
        eventStore.createIndex('certificate', 'certificate', { unique: false })

        const certificateStore = db.createObjectStore(_CERTIFICATES, { keyPath: 'uuid' })
        certificateStore.createIndex('uuid', 'uuid', { unique: true })
      }

      openDbRequest.onsuccess = (event) => {
        resolve(event.target.result)
      }

      openDbRequest.onerror = (error) => {
        reject(error)
      }
    } catch (error) {
      reject(error)
    }
  })
}

async function tally (request) {
  try {
    const json = await request.json()
    const certificates = {}
    for (const tallyEvent of json.events) {
      certificates[tallyEvent.certificate] = await addEvent(tallyEvent)
    }

    return new Response(JSON.stringify({
      err: 0,
      data: {
        timestamp: lastUpdateTimestamp,
        certificates,
      },
    }))
  } catch (error) {
    console.log(error)
  }
}

async function refreshTally (url) {
  try {
    console.info(`Refreshing tally from ${url}`)
    try {
      const tallyEvents = await getPendingEvents()
      const response = await fetch(
        url,
        {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            since: lastUpdateTimestamp,
            asynchronous: true,
            events: tallyEvents,
          }),
        })

      if (!response.ok) {
        console.log(`Error while refreshing tally : ${response.status} (${response.statusText})`)
      }

      const eventsToFlush = [...tallyEvents.map(evt => evt.uuid)]
      const json = await response.json()
      const data = json.data

      for (const [uuid, certificate] of Object.entries(data.certificates)) {
        await saveCertificate({uuid, ...certificate}, eventsToFlush)
      }

      lastUpdateTimestamp = data.timestamp
    } catch (error) {
      if (error.name !== 'NetworkError') {
        throw error
      }
    }
  } catch (error) {
    console.log(error)
  }
}

function addEvent (tallyEvent) {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction([_EVENTS, _CERTIFICATES], 'readwrite')
    const tallyEventStore = transaction.objectStore(_EVENTS)
    const tallyEventIndex = tallyEventStore.index('certificate')
    const certificateStore = transaction.objectStore(_CERTIFICATES)

    const certificateUUID = tallyEvent.certificate
    const pendingEventsRequest = tallyEventIndex.count(certificateUUID)

    pendingEventsRequest.onerror = () => reject(pendingEventsRequest.error)
    pendingEventsRequest.onsuccess = () => {
      const certificateRequest = certificateStore.get(certificateUUID)

      certificateRequest.onerror = () => reject(certificateRequest.error)
      certificateRequest.onsuccess = () => {
        const certificate = certificateRequest.result || { credit: 0 }
        const credit = certificate.credit - pendingEventsRequest.result - 1
        if (credit < 0) {
          resolve({...certificate, credit})
        } else {
          const addEventRequest = tallyEventStore.add(tallyEvent)

          addEventRequest.onerror = () => reject(addEventRequest.error)
          addEventRequest.onsuccess = () => {
            resolve({...certificate, credit})
          }
        }
      }
    }
  })
}

function getPendingEvents () {
  return new Promise((resolve, reject) => {
    // It is important to create the transaction inside the promise : if
    // execution is given back to the event loop and a transaction has no
    // pending request, the transaction is closed, making following calls fail
    // with InactiveTransactionError.
    const transaction = db.transaction([_EVENTS], 'readonly')
    const tallyEventStore = transaction.objectStore(_EVENTS)

    // If a stored event was received from the API, it will not have a
    // "pending" field. Even if it was added locally with a "pending" field set
    // to 1, refreshTally will overwrite it with a new event object without
    // that field.
    //
    // IndexedDB indices only return objects that have a value for the field
    // they index, so all that we have to do here is to get all events in the
    // "pending" index : they are those added locally and not yet synchronized
    // with the backend.
    const request = tallyEventStore.getAll()
    request.onsuccess = () => resolve(request.result)
    request.onerror = () => reject(request.error)
  })
}

function saveCertificate (certificate, eventsToFlush) {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction([_CERTIFICATES, _EVENTS], 'readwrite')
    const certificateStore = transaction.objectStore(_CERTIFICATES)
    const tallyEventStore = transaction.objectStore(_EVENTS)
    const tallyEventIndex = tallyEventStore.index('certificate')
    const saveCertificateRequest = certificateStore.put(certificate)

    saveCertificateRequest.onerror = () => reject(saveCertificateRequest.error)
    saveCertificateRequest.onsuccess = () => {
      const flushEventsRequest = tallyEventIndex.openCursor(certificate.uuid)

      flushEventsRequest.onerror = () => reject(flushEventsRequest.error)
      flushEventsRequest.onsuccess = (event) => {
        const cursor = event.target.result
        if (!cursor) {
          resolve()
          return
        }

        if (eventsToFlush.includes(cursor.value.uuid)) {
          cursor.delete()
        }

        cursor.continue()
      }
    }
  })
}
