import binascii
import os
import uuid
from collections import defaultdict
from datetime import datetime, timedelta, timezone
from io import BytesIO
from urllib.parse import urlparse

from django.conf import settings
from django.core.exceptions import PermissionDenied
from django.core.validators import RegexValidator
from django.db import models, transaction
from django.db.models import Q
from django.http import HttpResponse
from django.shortcuts import get_object_or_404
from django.template.response import TemplateResponse
from django.urls import reverse
from django.utils.dateparse import parse_datetime
from django.utils.http import urlencode
from django.utils.translation import gettext_lazy as _
from nacl.signing import SigningKey
from qrcode import ERROR_CORRECT_Q, QRCode
from qrcode.image.pil import PilImage
from requests.exceptions import RequestException

from passerelle.base.models import BaseResource
from passerelle.base.signature import sign_url
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError

UUID_PATTERN = '(?P<uuid>[0-9|a-f]{8}-[0-9|a-f]{4}-[0-9|a-f]{4}-[0-9|a-f]{4}-[0-9a-f]{12})'

CAMPAIGN_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-06/schema#',
    'type': 'object',
    'unflatten': True,
    'additionalProperties': False,
    'properties': {
        'metadata': {
            'type': 'object',
            'title': _('Metadata associated to the campaign'),
            'additionalProperties': {'type': 'string'},
        },
    },
}

CERTIFICATE_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-06/schema#',
    'type': 'object',
    'unflatten': True,
    'additionalProperties': False,
    'properties': {
        'campaign': {
            'type': 'string',
            'title': _('Campaign'),
            'pattern': UUID_PATTERN,
        },
        'data': {
            'type': 'object',
            'title': _('Data to encode in the certificate'),
            'additionalProperties': {'type': 'string'},
        },
        'metadata': {
            'type': 'object',
            'title': _('Metadata associated to the certificate'),
            'additionalProperties': {'type': 'string'},
        },
        'validity_start': {
            'any': [{'type': 'null'}, {'const': ''}, {'type': 'string', 'format': 'date-time'}],
        },
        'validity_end': {
            'any': [{'type': 'null'}, {'const': ''}, {'type': 'string', 'format': 'date-time'}],
        },
    },
}

READER_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-06/schema#',
    'type': 'object',
    'additionalProperties': False,
    'properties': {
        'campaign': {
            'type': 'string',
            'title': _('Campaign'),
            'pattern': UUID_PATTERN,
        },
        'readable_metadatas': {
            'title': _('Comma-separated list of metadata keys that this reader is allowed to read.'),
            'any': [{'type': 'null'}, {'const': ''}, {'type': 'string'}],
        },
        'validity_start': {
            'any': [{'type': 'null'}, {'const': ''}, {'type': 'string', 'format': 'date-time'}],
        },
        'validity_end': {
            'any': [{'type': 'null'}, {'const': ''}, {'type': 'string', 'format': 'date-time'}],
        },
        'metadata': {
            'type': 'object',
            'title': _('Metadata associated with this reader'),
            'additionalProperties': {'type': 'string'},
        },
    },
}

EVENT_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-06/schema#',
    'type': 'object',
    'additionalProperties': False,
    'properties': {
        'certificate': {
            'type': 'string',
            'title': _('Certificate'),
            'pattern': UUID_PATTERN,
        },
        'metadata': {
            'type': 'object',
            'title': _('Metadata associated with the event'),
            'additionalProperties': True,
        },
    },
}


def generate_key():
    key = os.urandom(32)
    return ''.join(format(x, '02x') for x in key)


TALLY_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-06/schema#',
    'type': 'object',
    'unflatten': True,
    'additionalProperties': False,
    'properties': {
        'since': {
            'type': 'integer',
            'title': _('Get events since this timestamp'),
        },
        'asynchronous': {
            'type': 'boolean',
            'title': _('Asynchronous tallying (force save events if credit is insufficient)'),
        },
        'events': {
            'type': 'array',
            'title': _('Events to tally'),
            'items': {
                'type': 'object',
                'properties': {
                    'timestamp': {
                        'type': 'integer',
                        'title': _('Timestamp when the QRCode was read'),
                    },
                    'certificate': {
                        'type': 'string',
                        'title': _('Read certificate'),
                        'pattern': UUID_PATTERN,
                    },
                },
            },
        },
    },
}


class QRCodeConnector(BaseResource):
    category = _('Misc')

    class Meta:
        verbose_name = _('QR Code')

    @property
    def certificates(self):
        return Certificate.objects.filter(campaign__resource=self)

    @property
    def readers(self):
        return Reader.objects.filter(campaign__resource=self)

    @endpoint(
        name='save-campaign',
        pattern=f'^{UUID_PATTERN}?$',
        example_pattern='{uuid}',
        description=_('Create or update a campaign'),
        post={'request_body': {'schema': {'application/json': CAMPAIGN_SCHEMA}}},
        parameters={
            'uuid': {
                'description': _('Campaign identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def save_campaign(self, request, uuid=None, post_data=None):
        metadata = post_data.get('metadata') or {}
        if not uuid:
            campaign = self.campaigns.create(metadata=metadata)
        else:
            campaign = get_object_or_404(self.campaigns, uuid=uuid)
            campaign.metadata = metadata
            campaign.save()

        return {
            'data': {
                'uuid': campaign.uuid,
            }
        }

    @endpoint(
        name='save-certificate',
        pattern=f'^{UUID_PATTERN}?$',
        example_pattern='{uuid}',
        description=_('Create or update a certificate'),
        post={'request_body': {'schema': {'application/json': CERTIFICATE_SCHEMA}}},
        parameters={
            'uuid': {
                'description': _('Certificate identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def save_certificate(self, request, uuid=None, post_data=None):
        if post_data.get('validity_start'):
            validity_start = parse_datetime(post_data['validity_start'])
        else:
            validity_start = None
        if post_data.get('validity_end'):
            validity_end = parse_datetime(post_data['validity_end'])
        else:
            validity_end = None
        data = post_data.get('data') or {}
        metadata = post_data.get('metadata') or {}

        if not uuid:
            campaign = self.campaigns.get(uuid=post_data['campaign'])
            certificate = campaign.certificates.create(
                campaign=campaign,
                data=data,
                metadata=metadata,
                validity_start=validity_start,
                validity_end=validity_end,
            )
        else:
            certificate = get_object_or_404(self.certificates, uuid=uuid)
            if 'campaign' in post_data and post_data['campaign'] != str(certificate.campaign.uuid):
                raise APIError("You can't change the campaign of an existing certificate", http_status=400)
            certificate.validity_start = validity_start
            certificate.validity_end = validity_end
            certificate.data = data
            certificate.metadata = metadata
            certificate.save()

        return {
            'data': {
                'uuid': certificate.uuid,
                'qrcode_url': certificate.get_qrcode_url(request),
            }
        }

    @endpoint(
        name='get-certificate',
        description=_('Retrieve an existing certificate'),
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('Certificate identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def get_certificate(self, request, uuid):
        certificate = get_object_or_404(self.certificates, uuid=uuid)
        return {
            'err': 0,
            'data': {
                'uuid': certificate.uuid,
                'data': certificate.data,
                'validity_start': certificate.validity_start and certificate.validity_start.isoformat(),
                'validity_end': certificate.validity_end and certificate.validity_end.isoformat(),
                'qrcode_url': certificate.get_qrcode_url(request),
            },
        }

    @endpoint(
        name='get-qrcode',
        description=_('Get QR Code'),
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('Certificate identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def get_qrcode(self, request, uuid):
        certificate = get_object_or_404(self.certificates, uuid=uuid)
        qr_code = certificate.generate_qr_code()
        return HttpResponse(qr_code, content_type='image/png')

    @endpoint(
        name='save-reader',
        pattern=f'^{UUID_PATTERN}?$',
        example_pattern='{uuid}',
        description=_('Create or update a qrcode reader'),
        post={'request_body': {'schema': {'application/json': READER_SCHEMA}}},
        parameters={
            'uuid': {
                'description': _('QRCode reader identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def save_reader(self, request, uuid=None, post_data=None):
        if post_data.get('validity_start'):
            validity_start = parse_datetime(post_data['validity_start'])
        else:
            validity_start = None
        if post_data.get('validity_end'):
            validity_end = parse_datetime(post_data['validity_end'])
        else:
            validity_end = None

        readable_metadatas = post_data.get('readable_metadatas', '')
        metadata = post_data.get('metadata', {})

        if not uuid:
            campaign = get_object_or_404(self.campaigns, uuid=post_data['campaign'])
            reader = campaign.readers.create(
                campaign=campaign,
                validity_start=validity_start,
                validity_end=validity_end,
                readable_metadatas=readable_metadatas,
                metadata=metadata,
            )
        else:
            reader = get_object_or_404(self.readers, uuid=uuid)
            if 'campaign' in post_data:
                reader.campaign = get_object_or_404(self.campaigns, uuid=post_data['campaign'])
            reader.validity_start = validity_start
            reader.validity_end = validity_end
            reader.readable_metadatas = readable_metadatas
            reader.metadata = metadata
            reader.save()

        return {
            'data': {
                'uuid': reader.uuid,
                'url': reader.get_url(request),
            }
        }

    @endpoint(
        name='get-reader',
        description=_('Get informations about a QRCode reader'),
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('QRCode reader identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def get_reader(self, request, uuid):
        reader = get_object_or_404(self.readers, uuid=uuid)
        return {
            'err': 0,
            'data': {
                'uuid': reader.uuid,
                'validity_start': reader.validity_start and reader.validity_start.isoformat(),
                'validity_end': reader.validity_end and reader.validity_end.isoformat(),
                'url': reader.get_url(request),
            },
        }

    @endpoint(
        name='open-reader',
        perm='OPEN',
        description=_('Open a QRCode reader page.'),
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('QRCode reader identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def open_reader(self, request, uuid):
        reader = get_object_or_404(self.readers, uuid=uuid)
        now = datetime.now(timezone.utc)

        return TemplateResponse(
            request,
            'qrcode/qrcode-reader.html',
            context={
                'started': now >= reader.validity_start if reader.validity_start is not None else True,
                'expired': now >= reader.validity_end if reader.validity_end is not None else False,
                'verify_key': reader.campaign.hex_verify_key,
                'reader': reader,
                'metadata_url': reader.get_metadata_url(request),
                'tally_url': reader.get_tally_url(request) if reader.merged_metadata.get('tally') else None,
                'service_worker_scope': reverse(
                    'generic-endpoint',
                    kwargs={
                        'slug': self.slug,
                        'connector': self.get_connector_slug(),
                        'endpoint': 'open-reader',
                    },
                ),
            },
        )

    @endpoint(
        name='read-metadata',
        perm='OPEN',
        description=_('Read certificate metadata'),
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('QRCode reader identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            },
            'certificate': {
                'description': _('Certificate identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            },
        },
    )
    def read_metadata(self, request, uuid, certificate):
        reader = get_object_or_404(self.readers, uuid=uuid)
        certificate = get_object_or_404(self.certificates, uuid=certificate)
        now = datetime.now(timezone.utc)
        if reader.validity_start is not None and now < reader.validity_start:
            return {'err': 1, 'err_desc': _("Reader isn't usable yet.")}

        if reader.validity_end is not None and now > reader.validity_end:
            return {'err': 1, 'err_desc': _('Reader has expired.')}

        readable_metatadas = reader.readable_metadatas.split(',')
        return {'err': 0, 'data': {k: v for k, v in certificate.metadata.items() if k in readable_metatadas}}

    @endpoint(
        name='tally',
        perm='OPEN',
        pattern=f'^{UUID_PATTERN}?$',
        description=_('Tally and get tallied events'),
        post={'request_body': {'schema': {'application/json': TALLY_SCHEMA}}},
    )
    def tally(self, request, uuid=None, post_data=None):
        reader = get_object_or_404(self.readers, uuid=uuid)

        if not reader.merged_metadata.get('tally'):
            raise PermissionDenied('Tallying is not enabled for this reader')

        pending_events = defaultdict(list)
        for event in post_data.get('events', []):
            pending_events[event['certificate']].append(event)

        certificates = {}

        now = datetime.now(timezone.utc)
        certificates_queryset = Certificate.objects.filter(
            Q(validity_end__gte=datetime.now(timezone.utc)) | Q(validity_end__isnull=True),
            campaign=reader.campaign,
        )

        since_timestamp = post_data.get('since', 0)
        if since_timestamp != 0:
            since = datetime.fromtimestamp(since_timestamp, timezone.utc) - timedelta(minutes=1)
            updated_events_uuids = [
                cert.uuid
                for cert in Certificate.objects.filter(campaign=reader.campaign)
                .annotate(last_event_received=models.Max('events__received'))
                .filter(last_event_received__gte=since)
            ]

            certificates_queryset = certificates_queryset.filter(
                Q(uuid__in=pending_events.keys()) | Q(uuid__in=updated_events_uuids)
            )

        with transaction.atomic():
            asynchronous = post_data.get('asynchronous', False)
            for certificate in certificates_queryset.select_for_update():
                certificate_uuid = str(certificate.uuid)
                certificate_pending_events = pending_events.get(certificate_uuid, [])
                certificates[str(certificate_uuid)] = certificate.sync_events(
                    certificate_pending_events, reader, asynchronous
                )

        return {
            'data': {
                'timestamp': int(datetime.timestamp(now)),
                'certificates': certificates,
            }
        }

    @endpoint(
        name='add-event',
        description=_('Add an event to a certificate'),
        post={'request_body': {'schema': {'application/json': EVENT_SCHEMA}}},
    )
    def add_event(self, request, post_data):
        certificate_uuid = post_data['certificate']
        try:
            certificate = Certificate.objects.get(uuid=certificate_uuid)
        except Certificate.DoesNotExist:
            raise APIError('Unknown certificate {certificate_uuid}', http_status=400)
        now = datetime.now()

        metadata = post_data.get('metadata', {})
        if 'credit' in metadata:
            try:
                metadata['credit'] = int(metadata['credit'])
            except ValueError:
                raise APIError('Invalid credit value: %s' % metadata['credit'], http_status=400)

        certificate.events.create(metadata=metadata, happened=now)

    @endpoint(
        name='events',
        description=_('List events'),
        parameters={
            'campaign': {
                'description': _('List only events related to this campaign.'),
                'example_value': '12345678-1234-1234-1234-123456789012',
                'pattern': UUID_PATTERN,
            },
            'certificate': {
                'description': _('List only events related to this certificate.'),
                'example_value': '12345678-1234-1234-1234-123456789012',
                'pattern': UUID_PATTERN,
            },
            'reader': {
                'description': _('List only events related to this reader.'),
                'example_value': '12345678-1234-1234-1234-123456789012',
                'pattern': UUID_PATTERN,
            },
            'since': {
                'description': _('List only event received after this date.'),
                'example_value': '2024-02-01T20:00:00+00:00',
            },
            'until': {
                'description': _('List only event received before this date.'),
                'example_value': '2024-02-01T20:00:00+00:00',
            },
            'tally': {
                'description': _(
                    'Only list event generated by readers (default: list events coming from both readers and direct API calls)'
                ),
                'example_value': 'True',
                'type': 'boolean',
            },
        },
    )
    def events(
        self,
        request,
        campaign=None,
        certificate=None,
        reader=None,
        since=None,
        until=None,
        tally=None,
    ):
        result = []
        for event in self._list_events(
            campaign=campaign, certificate=certificate, reader=reader, since=since, until=until, tally=tally
        ):
            result.append(
                {
                    'uuid': str(event.uuid),
                    'certificate': str(event.certificate.uuid),
                    'reader': str(event.reader.uuid) if event.reader else None,
                    'received': event.received.isoformat(),
                    'happened': event.happened.isoformat(),
                    'metadata': event.metadata,
                }
            )
        return {'data': result}

    def _list_events(
        self,
        campaign=None,
        certificate=None,
        reader=None,
        since=None,
        until=None,
        tally=None,
        summary=False,
    ):
        query = Event.objects.filter(certificate__campaign__resource=self)
        if campaign:
            query = query.filter(certificate__campaign__uuid=campaign)
        if certificate:
            query = query.filter(certificate__uuid=certificate)
        if reader:
            query = query.filter(reader__uuid=reader)
        if since:
            query = query.filter(received__gte=datetime.fromisoformat(since))
        if until:
            query = query.filter(received__lte=datetime.fromisoformat(until))
        if tally is not None:
            query = query.filter(reader__isnull=not tally)

        return query


def encode_mime_like(data):
    msg = ''
    for key, value in data.items():
        msg += '%s: %s\n' % (key, value.replace('\n', '\n '))
    return msg.encode()


BASE45_CHARSET = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ $%*+-./:'
BASE45_DICT = {v: i for i, v in enumerate(BASE45_CHARSET)}


def b45encode(buf: bytes) -> bytes:
    """Convert bytes to base45-encoded string"""
    res = ''
    buflen = len(buf)
    for i in range(0, buflen & ~1, 2):
        x = (buf[i] << 8) + buf[i + 1]
        e, x = divmod(x, 45 * 45)
        d, c = divmod(x, 45)
        res += BASE45_CHARSET[c] + BASE45_CHARSET[d] + BASE45_CHARSET[e]
    if buflen & 1:
        d, c = divmod(buf[-1], 45)
        res += BASE45_CHARSET[c] + BASE45_CHARSET[d]
    return res.encode()


class Campaign(models.Model):
    uuid = models.UUIDField(verbose_name=_('UUID'), unique=True, default=uuid.uuid4)
    resource = models.ForeignKey(QRCodeConnector, on_delete=models.CASCADE, related_name='campaigns')
    created = models.DateTimeField(_('Created'), auto_now_add=True)
    modified = models.DateTimeField(verbose_name=_('Last modification'), auto_now=True)
    metadata = models.JSONField(null=True, verbose_name=_('Campaign meta data'))

    key = models.CharField(
        _('Private Key'),
        max_length=64,
        default=generate_key,
        validators=[RegexValidator(r'[a-z|0-9]{64}', 'Key should be a 32 bytes hexadecimal string')],
    )

    @property
    def signing_key(self):
        binary_key = binascii.unhexlify(self.key)
        return SigningKey(seed=binary_key)

    @property
    def hex_verify_key(self):
        verify_key = self.signing_key.verify_key.encode()
        return binascii.hexlify(verify_key).decode('utf-8')


class Certificate(models.Model):
    uuid = models.UUIDField(verbose_name=_('UUID'), unique=True, default=uuid.uuid4)
    campaign = models.ForeignKey(Campaign, on_delete=models.CASCADE, related_name='certificates')
    created = models.DateTimeField(_('Created'), auto_now_add=True)
    modified = models.DateTimeField(verbose_name=_('Last modification'), auto_now=True)
    validity_start = models.DateTimeField(verbose_name=_('Validity Start Date'), null=True)
    validity_end = models.DateTimeField(verbose_name=_('Validity End Date'), null=True)
    data = models.JSONField(null=True, verbose_name='Certificate Data')
    metadata = models.JSONField(null=True, verbose_name='Certificate meta data')

    def to_json(self):
        data = {'uuid': str(self.uuid)}
        if self.validity_start:
            data['validity_start'] = str(self.validity_start.timestamp())
        if self.validity_end:
            data['validity_end'] = str(self.validity_end.timestamp())
        data |= self.data
        return data

    def generate_b45_data(self):
        data = self.to_json()
        msg = encode_mime_like(data)
        signed = self.campaign.signing_key.sign(msg)
        return b45encode(signed).decode()

    def generate_qr_code(self):
        qr_code = QRCode(image_factory=PilImage, error_correction=ERROR_CORRECT_Q)
        data = self.generate_b45_data()
        qr_code.add_data(data)
        qr_code.make(fit=True)
        image = qr_code.make_image(fill_color='black', back_color='white')
        fd = BytesIO()
        image.save(fd)
        return fd.getvalue()

    def get_qrcode_url(self, request):
        qrcode_relative_url = reverse(
            'generic-endpoint',
            kwargs={
                'slug': self.campaign.resource.slug,
                'connector': self.campaign.resource.get_connector_slug(),
                'endpoint': 'get-qrcode',
                'rest': str(self.uuid),
            },
        )
        return request.build_absolute_uri(qrcode_relative_url)

    def sync_events(self, pending_events, pending_events_reader, asynchronous):
        events_data = self.events.values_list('uuid', 'metadata__credit')

        # discard duplicated events, the DB is the source of truth
        existing_uuids = {uuid for uuid, _ in events_data}
        pending_events = [evt for evt in pending_events if evt['uuid'] not in existing_uuids]

        credit = sum(credit for _, credit in events_data if credit)
        credit -= len(pending_events)

        # The asynchronous parameter is injected by the QR code reader's
        # service worker when it sends events that were already validated
        # according to it's cache state (c.f qrcode-service-worker.js). When we
        # receive such events, we want to save them regardless the credit left
        # on the certificate, because we want the DB to reflect what happened
        # in reality.
        #
        # Here the reality is that the person who used this certificate is
        # already playing in the swimming pool even if they didn't have credit
        # left on their card because the QR code reader a the entrance was out
        # of sync when they checked in. Make them pay.
        if credit >= 0 or asynchronous:
            for event in pending_events:
                self.events.create(
                    uuid=event.get('uuid'),
                    reader=pending_events_reader,
                    happened=datetime.fromtimestamp(event['timestamp'], timezone.utc),
                    metadata={'credit': -1},
                )
            self.trigger_wcs()

        return {'credit': credit}

    def trigger_wcs(self):
        if self.metadata is None or 'trigger_url' not in self.metadata:
            return

        trigger_url = self.metadata['trigger_url']
        signed_url = self._sign_wcs_url(trigger_url)
        if signed_url is None:
            return

        try:
            response = self.campaign.resource.requests.post(signed_url)
            response.raise_for_status()
        except RequestException as exception:
            self.campaign.resource.logger.error(
                'Error while triggering w.c.s. for certificate %s : %s', self.uuid, exception
            )

    def _sign_wcs_url(self, trigger_url):
        scheme, netloc, _, _, _, _ = urlparse(trigger_url)
        services = settings.KNOWN_SERVICES.get('wcs', {})
        for service in services.values():
            remote_url = service.get('url')
            service_scheme, service_netloc, _, _, _, _ = urlparse(remote_url)
            if service_scheme == scheme and service_netloc == netloc:
                orig = (service.get('orig'),)
                if orig:
                    trigger_url = (
                        trigger_url + ('&' if '?' in trigger_url else '?') + urlencode({'orig': orig})
                    )
                secret = service.get('secret')
                return sign_url(trigger_url, secret)

        self.campaign.resource.logger.error(
            "Can't find a suitable configured WCS service for url %s", trigger_url
        )

        return None


class Reader(models.Model):
    campaign = models.ForeignKey(Campaign, on_delete=models.CASCADE, related_name='readers')
    uuid = models.UUIDField(verbose_name=_('UUID'), unique=True, default=uuid.uuid4)
    created = models.DateTimeField(_('Created'), auto_now_add=True)
    modified = models.DateTimeField(verbose_name=_('Last modification'), auto_now=True)
    validity_start = models.DateTimeField(verbose_name=_('Validity Start Date'), null=True)
    validity_end = models.DateTimeField(verbose_name=_('Validity End Date'), null=True)
    readable_metadatas = models.CharField(max_length=128, verbose_name=_('Readable metadata keys'), null=True)
    metadata = models.JSONField(null=True, verbose_name=_('Reader meta data'))

    @property
    def merged_metadata(self):
        return (self.campaign.metadata or {}) | (self.metadata or {})

    def get_url(self, request):
        return self._get_endpoint_url(request, 'open-reader')

    def get_metadata_url(self, request):
        if not self.readable_metadatas:
            return None
        return self._get_endpoint_url(request, 'read-metadata')

    def get_tally_url(self, request):
        return self._get_endpoint_url(request, 'tally')

    def _get_endpoint_url(self, request, endpoint):
        relative_url = reverse(
            'generic-endpoint',
            kwargs={
                'slug': self.campaign.resource.slug,
                'connector': self.campaign.resource.get_connector_slug(),
                'endpoint': endpoint,
                'rest': str(self.uuid),
            },
        )
        return request.build_absolute_uri(relative_url)


class Event(models.Model):
    uuid = models.UUIDField(verbose_name=_('UUID'), unique=True, default=uuid.uuid4)
    certificate = models.ForeignKey(Certificate, on_delete=models.CASCADE, related_name='events')
    reader = models.ForeignKey(Reader, on_delete=models.CASCADE, related_name='events', null=True)
    happened = models.DateTimeField()
    received = models.DateTimeField(auto_now_add=True)
    metadata = models.JSONField(null=True, verbose_name='Event metadata')
