import binascii
import os
import uuid
from datetime import datetime, timedelta, timezone
from io import BytesIO

from django.core.exceptions import PermissionDenied
from django.core.validators import RegexValidator
from django.db import models
from django.http import HttpResponse
from django.shortcuts import get_object_or_404
from django.template.response import TemplateResponse
from django.urls import reverse
from django.utils.dateparse import parse_datetime
from django.utils.translation import gettext_lazy as _
from nacl.signing import SigningKey
from qrcode import ERROR_CORRECT_Q, QRCode
from qrcode.image.pil import PilImage

from passerelle.base.models import BaseResource
from passerelle.utils.api import endpoint

CERTIFICATE_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-06/schema#',
    'type': 'object',
    'unflatten': True,
    'additionalProperties': False,
    'properties': {
        'data': {
            'type': 'object',
            'title': _('Data to encode in the certificate'),
            'additionalProperties': {'type': 'string'},
        },
        'metadata': {
            'type': 'object',
            'title': _('Metadata associated to the certificate'),
            'additionalProperties': {'type': 'string'},
        },
        'validity_start': {
            'any': [{'type': 'null'}, {'const': ''}, {'type': 'string', 'format': 'date-time'}],
        },
        'validity_end': {
            'any': [{'type': 'null'}, {'const': ''}, {'type': 'string', 'format': 'date-time'}],
        },
    },
}

READER_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-06/schema#',
    'type': 'object',
    'additionalProperties': False,
    'properties': {
        'readable_metadatas': {
            'title': _('Comma-separated list of metadata keys that this reader is allowed to read.'),
            'any': [{'type': 'null'}, {'const': ''}, {'type': 'string'}],
        },
        'enable_tallying': {'title': _('Enable tallying on this reader.'), 'type': 'boolean'},
        'validity_start': {
            'any': [{'type': 'null'}, {'const': ''}, {'type': 'string', 'format': 'date-time'}],
        },
        'validity_end': {
            'any': [{'type': 'null'}, {'const': ''}, {'type': 'string', 'format': 'date-time'}],
        },
    },
}


def generate_key():
    key = os.urandom(32)
    return ''.join(format(x, '02x') for x in key)


UUID_PATTERN = '(?P<uuid>[0-9|a-f]{8}-[0-9|a-f]{4}-[0-9|a-f]{4}-[0-9|a-f]{4}-[0-9a-f]{12})'

TALLY_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-06/schema#',
    'type': 'object',
    'unflatten': True,
    'additionalProperties': False,
    'properties': {
        'since': {
            'type': 'integer',
            'title': _('Get events since this timestamp'),
        },
        'events': {
            'type': 'array',
            'title': _('Events to tally'),
            'items': {
                'type': 'object',
                'properties': {
                    'timestamp': {
                        'type': 'integer',
                        'title': _('Timestamp when the QRCode was read'),
                    },
                    'certificate': {
                        'type': 'string',
                        'title': _('Read certificate'),
                        'pattern': UUID_PATTERN,
                    },
                },
            },
        },
    },
}


class QRCodeConnector(BaseResource):
    category = _('Misc')

    key = models.CharField(
        _('Private Key'),
        max_length=64,
        default=generate_key,
        validators=[RegexValidator(r'[a-z|0-9]{64}', 'Key should be a 32 bytes hexadecimal string')],
    )

    class Meta:
        verbose_name = _('QR Code')

    @property
    def signing_key(self):
        binary_key = binascii.unhexlify(self.key)
        return SigningKey(seed=binary_key)

    @property
    def hex_verify_key(self):
        verify_key = self.signing_key.verify_key.encode()
        return binascii.hexlify(verify_key).decode('utf-8')

    @endpoint(
        name='save-certificate',
        pattern=f'^{UUID_PATTERN}?$',
        example_pattern='{uuid}',
        description=_('Create or update a certificate'),
        post={'request_body': {'schema': {'application/json': CERTIFICATE_SCHEMA}}},
        parameters={
            'uuid': {
                'description': _('Certificate identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def save_certificate(self, request, uuid=None, post_data=None):
        if post_data.get('validity_start'):
            validity_start = parse_datetime(post_data['validity_start'])
        else:
            validity_start = None
        if post_data.get('validity_end'):
            validity_end = parse_datetime(post_data['validity_end'])
        else:
            validity_end = None
        data = post_data.get('data') or {}
        metadata = post_data.get('metadata') or {}

        if not uuid:
            certificate = self.certificates.create(
                data=data,
                metadata=metadata,
                validity_start=validity_start,
                validity_end=validity_end,
            )
        else:
            certificate = get_object_or_404(self.certificates, uuid=uuid)
            certificate.validity_start = validity_start
            certificate.validity_end = validity_end
            certificate.data = data
            certificate.metadata = metadata
            certificate.save()

        return {
            'data': {
                'uuid': certificate.uuid,
                'qrcode_url': certificate.get_qrcode_url(request),
            }
        }

    @endpoint(
        name='get-certificate',
        description=_('Retrieve an existing certificate'),
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('Certificate identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def get_certificate(self, request, uuid):
        certificate = get_object_or_404(self.certificates, uuid=uuid)
        return {
            'err': 0,
            'data': {
                'uuid': certificate.uuid,
                'data': certificate.data,
                'validity_start': certificate.validity_start and certificate.validity_start.isoformat(),
                'validity_end': certificate.validity_end and certificate.validity_end.isoformat(),
                'qrcode_url': certificate.get_qrcode_url(request),
            },
        }

    @endpoint(
        name='get-qrcode',
        description=_('Get QR Code'),
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('Certificate identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def get_qrcode(self, request, uuid):
        certificate = self.certificates.get(uuid=uuid)
        qr_code = certificate.generate_qr_code()
        return HttpResponse(qr_code, content_type='image/png')

    @endpoint(
        name='save-reader',
        pattern=f'^{UUID_PATTERN}?$',
        example_pattern='{uuid}',
        description=_('Create or update a qrcode reader'),
        post={'request_body': {'schema': {'application/json': READER_SCHEMA}}},
        parameters={
            'uuid': {
                'description': _('QRCode reader identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def save_reader(self, request, uuid=None, post_data=None):
        if post_data.get('validity_start'):
            validity_start = parse_datetime(post_data['validity_start'])
        else:
            validity_start = None
        if post_data.get('validity_end'):
            validity_end = parse_datetime(post_data['validity_end'])
        else:
            validity_end = None

        readable_metadatas = post_data.get('readable_metadatas', '')

        if not uuid:
            reader = self.readers.create(
                validity_start=validity_start,
                validity_end=validity_end,
                readable_metadatas=readable_metadatas,
                tally=post_data.get('enable_tallying') or False,
            )
        else:
            reader = get_object_or_404(self.readers, uuid=uuid)
            reader.validity_start = validity_start
            reader.validity_end = validity_end
            reader.readable_metadatas = readable_metadatas
            reader.tally = post_data.get('enable_tallying') or False
            reader.save()

        return {
            'data': {
                'uuid': reader.uuid,
                'url': reader.get_url(request),
            }
        }

    @endpoint(
        name='get-reader',
        description=_('Get informations about a QRCode reader'),
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('QRCode reader identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def get_reader(self, request, uuid):
        reader = get_object_or_404(self.readers, uuid=uuid)
        return {
            'err': 0,
            'data': {
                'uuid': reader.uuid,
                'validity_start': reader.validity_start and reader.validity_start.isoformat(),
                'validity_end': reader.validity_end and reader.validity_end.isoformat(),
                'url': reader.get_url(request),
            },
        }

    @endpoint(
        name='open-reader',
        perm='OPEN',
        description=_('Open a QRCode reader page.'),
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('QRCode reader identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            }
        },
    )
    def open_reader(self, request, uuid):
        reader = get_object_or_404(self.readers, uuid=uuid)
        now = datetime.now(timezone.utc)

        return TemplateResponse(
            request,
            'qrcode/qrcode-reader.html',
            context={
                'started': now >= reader.validity_start if reader.validity_start is not None else True,
                'expired': now >= reader.validity_end if reader.validity_end is not None else False,
                'verify_key': self.hex_verify_key,
                'reader': reader,
                'metadata_url': reader.get_metadata_url(request),
                'tally_url': reader.get_tally_url(request) if reader.tally else None,
                'service_worker_scope': reverse(
                    'generic-endpoint',
                    kwargs={
                        'slug': self.slug,
                        'connector': self.get_connector_slug(),
                        'endpoint': 'open-reader',
                    },
                ),
            },
        )

    @endpoint(
        name='read-metadata',
        perm='OPEN',
        description=_('Read certificate metadata'),
        pattern=f'^{UUID_PATTERN}$',
        example_pattern='{uuid}',
        parameters={
            'uuid': {
                'description': _('QRCode reader identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            },
            'certificate': {
                'description': _('Certificate identifier'),
                'example_value': '12345678-1234-1234-1234-123456789012',
            },
        },
    )
    def read_metadata(self, request, uuid, certificate):
        reader = get_object_or_404(self.readers, uuid=uuid)
        certificate = get_object_or_404(self.certificates, uuid=certificate)
        now = datetime.now(timezone.utc)
        if reader.validity_start is not None and now < reader.validity_start:
            return {'err': 1, 'err_desc': _("Reader isn't usable yet.")}

        if reader.validity_end is not None and now > reader.validity_end:
            return {'err': 1, 'err_desc': _('Reader has expired.')}

        readable_metatadas = reader.readable_metadatas.split(',')
        return {'err': 0, 'data': {k: v for k, v in certificate.metadata.items() if k in readable_metatadas}}

    @endpoint(
        name='tally',
        perm='OPEN',
        pattern=f'^{UUID_PATTERN}?$',
        description=_('Tally and get tallied events'),
        post={'request_body': {'schema': {'application/json': TALLY_SCHEMA}}},
    )
    def tally(self, request, uuid=None, post_data=None):
        reader = get_object_or_404(self.readers, uuid=uuid)

        if not reader.tally:
            raise PermissionDenied('Tallying is not enabled for this reader')

        now = datetime.now(timezone.utc)
        since_timestamp = post_data.get('since')
        if since_timestamp == 0:
            since = now - timedelta(days=1)
        elif since_timestamp is not None:
            since = datetime.fromtimestamp(since_timestamp, timezone.utc) - timedelta(minutes=1)
        else:
            since = now

        stamps = {}

        for event in Event.objects.filter(received__gte=since):
            stamps[str(event.certificate.uuid)] = 'ok'

        for event in post_data.get('events', []):
            try:
                certificate = self.certificates.get(uuid=event['certificate'])
            except Certificate.DoesNotExist:
                continue

            _, created = Event.objects.get_or_create(
                certificate=certificate,
                defaults={
                    'reader': reader,
                    'happened': datetime.fromtimestamp(event['timestamp'], timezone.utc),
                },
            )
            stamps[str(certificate.uuid)] = 'ok' if created else 'duplicate'

        return {
            'data': {
                'timestamp': int(datetime.timestamp(now)),
                'stamps': stamps,
            }
        }


def encode_mime_like(data):
    msg = ''
    for key, value in data.items():
        msg += '%s: %s\n' % (key, value.replace('\n', '\n '))
    return msg.encode()


BASE45_CHARSET = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ $%*+-./:'
BASE45_DICT = {v: i for i, v in enumerate(BASE45_CHARSET)}


def b45encode(buf: bytes) -> bytes:
    """Convert bytes to base45-encoded string"""
    res = ''
    buflen = len(buf)
    for i in range(0, buflen & ~1, 2):
        x = (buf[i] << 8) + buf[i + 1]
        e, x = divmod(x, 45 * 45)
        d, c = divmod(x, 45)
        res += BASE45_CHARSET[c] + BASE45_CHARSET[d] + BASE45_CHARSET[e]
    if buflen & 1:
        d, c = divmod(buf[-1], 45)
        res += BASE45_CHARSET[c] + BASE45_CHARSET[d]
    return res.encode()


class Certificate(models.Model):
    uuid = models.UUIDField(verbose_name=_('UUID'), unique=True, default=uuid.uuid4)
    created = models.DateTimeField(_('Created'), auto_now_add=True)
    modified = models.DateTimeField(verbose_name=_('Last modification'), auto_now=True)
    validity_start = models.DateTimeField(verbose_name=_('Validity Start Date'), null=True)
    validity_end = models.DateTimeField(verbose_name=_('Validity End Date'), null=True)
    data = models.JSONField(null=True, verbose_name='Certificate Data')
    metadata = models.JSONField(null=True, verbose_name='Certificate meta data')
    resource = models.ForeignKey(QRCodeConnector, on_delete=models.CASCADE, related_name='certificates')

    def to_json(self):
        data = {'uuid': str(self.uuid)}
        if self.validity_start:
            data['validity_start'] = str(self.validity_start.timestamp())
        if self.validity_end:
            data['validity_end'] = str(self.validity_end.timestamp())
        data |= self.data
        return data

    def generate_b45_data(self):
        data = self.to_json()
        msg = encode_mime_like(data)
        signed = self.resource.signing_key.sign(msg)
        return b45encode(signed).decode()

    def generate_qr_code(self):
        qr_code = QRCode(image_factory=PilImage, error_correction=ERROR_CORRECT_Q)
        data = self.generate_b45_data()
        qr_code.add_data(data)
        qr_code.make(fit=True)
        image = qr_code.make_image(fill_color='black', back_color='white')
        fd = BytesIO()
        image.save(fd)
        return fd.getvalue()

    def get_qrcode_url(self, request):
        qrcode_relative_url = reverse(
            'generic-endpoint',
            kwargs={
                'slug': self.resource.slug,
                'connector': self.resource.get_connector_slug(),
                'endpoint': 'get-qrcode',
                'rest': str(self.uuid),
            },
        )
        return request.build_absolute_uri(qrcode_relative_url)


class Reader(models.Model):
    uuid = models.UUIDField(verbose_name=_('UUID'), unique=True, default=uuid.uuid4)
    created = models.DateTimeField(_('Created'), auto_now_add=True)
    modified = models.DateTimeField(verbose_name=_('Last modification'), auto_now=True)
    validity_start = models.DateTimeField(verbose_name=_('Validity Start Date'), null=True)
    validity_end = models.DateTimeField(verbose_name=_('Validity End Date'), null=True)
    readable_metadatas = models.CharField(max_length=128, verbose_name=_('Readable metadata keys'), null=True)
    resource = models.ForeignKey(QRCodeConnector, on_delete=models.CASCADE, related_name='readers')
    tally = models.BooleanField(verbose_name=_('Enable tally for this reader'), default=False)

    def get_url(self, request):
        return self._get_endpoint_url(request, 'open-reader')

    def get_metadata_url(self, request):
        if not self.readable_metadatas:
            return None
        return self._get_endpoint_url(request, 'read-metadata')

    def get_tally_url(self, request):
        return self._get_endpoint_url(request, 'tally')

    def _get_endpoint_url(self, request, endpoint):
        relative_url = reverse(
            'generic-endpoint',
            kwargs={
                'slug': self.resource.slug,
                'connector': self.resource.get_connector_slug(),
                'endpoint': endpoint,
                'rest': str(self.uuid),
            },
        )
        return request.build_absolute_uri(relative_url)


class Event(models.Model):
    uuid = models.UUIDField(verbose_name=_('UUID'), unique=True, default=uuid.uuid4)
    certificate = models.ForeignKey(Certificate, on_delete=models.CASCADE, related_name='events')
    reader = models.ForeignKey(Reader, on_delete=models.CASCADE, related_name='events')
    happened = models.DateTimeField()
    received = models.DateTimeField(auto_now_add=True)
