# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from urllib.parse import urljoin
from xml.etree import ElementTree as etree

from django.core.cache import cache
from django.db import models
from django.utils.functional import cached_property
from django.utils.translation import gettext_lazy as _

from passerelle.base.models import BaseResource
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError

PARAMS_SEPARATOR = '|$|'


class Selligent(BaseResource):
    base_api_url = models.URLField(max_length=400, verbose_name=_('Base API URL'))

    user = models.CharField(max_length=128, verbose_name=_('User'))
    password = models.CharField(max_length=128, verbose_name=_('Password'))
    database = models.CharField(max_length=128, verbose_name=_('Database'))
    domain = models.CharField(max_length=128, verbose_name=_('Domain'), blank=True)

    session_token_duration = models.IntegerField(
        default=120, verbose_name=_('Session token duration'), help_text=_('in minutes')
    )

    category = _('Business Process Connectors')

    class Meta:
        verbose_name = _('Selligent')

    @property
    def session_wsdl_url(self):
        return urljoin(self.base_api_url, 'SessionServices.asmx?WSDL')

    @property
    def script_wsdl_url(self):
        return urljoin(self.base_api_url, 'ScriptServices.asmx?WSDL')

    @cached_property
    def session_client(self):
        return self.soap_client(wsdl_url=self.session_wsdl_url, api_error=True)

    @cached_property
    def script_client(self):
        return self.soap_client(wsdl_url=self.script_wsdl_url, api_error=True)

    @property
    def token_cache_key(self):
        return 'selligent-%s-token' % self.id

    def call(self, client, method, **kwargs):
        service = getattr(self, client).service
        result = getattr(service, method)(**kwargs)
        if result.startswith('ERREUR'):
            raise APIError(result)
        if result.startswith('<'):
            t = etree.fromstring(result)
            warn = t.find('Warn')
            if warn is not None:
                raise APIError(warn.get('Message'))
        return result

    @property
    def session_token(self):
        token_cached_value = cache.get(self.token_cache_key)
        if token_cached_value:
            return token_cached_value
        result = self.call(
            'session_client',
            'Login',
            user=self.user,
            password=self.password,
            database=self.database,
            domain=self.domain,
        )
        cache.set(self.token_cache_key, result, self.session_token_duration * 60)
        return result

    def check_status(self):
        cache.delete(self.token_cache_key)
        assert self.session_token

    def execute_server_script(self, scriptNrid, data):
        result = self.call(
            'script_client',
            'ExecuteServerScript',
            sessionParams=self.session_token,
            scriptNrid=scriptNrid,
            paramsValue=data,
        )
        return {'data': result}

    def build_params_from_payload(self, payload):
        params = ['']
        if payload:
            for key, value in payload.items():
                value = value.replace('\n', 'ret-char1ot')
                value = value.replace('&', 'esperluett')
                value = value.replace(';', 'po1-virgul')
                params.append('%s=%s' % (key, value))
            params.append('')
        return PARAMS_SEPARATOR.join(params)

    @endpoint(
        methods=['post'],
        name='create-demand',
        description_post=_('Creates a demand'),
        post_json_schema={'type': 'object'},
        parameters={
            'scriptNrid': {'description': _('Script number'), 'example_value': '27821677706740'},
            'code_prestation': {'description': _('Code de prestation'), 'example_value': 'xyz'},
        },
    )
    def create_demand(self, request, scriptNrid, code_prestation, post_data, **kwargs):
        return self.execute_server_script(
            scriptNrid, '%s;%s' % (code_prestation, self.build_params_from_payload(post_data))
        )

    @endpoint(
        methods=['post'],
        name='add-demand-file',
        description_post=_('Add a file to demand'),
        post_json_schema={
            'type': 'object',
            'properties': {
                'file': {
                    'type': 'object',
                    'title': _('File object'),
                    'properties': {
                        'filename': {
                            'type': 'string',
                            'description': _('Filename'),
                        },
                        'content': {
                            'type': 'string',
                            'description': _('Content'),
                        },
                        'content_type': {
                            'type': 'string',
                            'description': _('Content type'),
                        },
                    },
                    'required': ['filename', 'content'],
                },
                'demand_number': {'type': 'string', 'title': _('Demand number')},
                'file_type': {'type': 'string', 'title': _('File type'), 'description': 'Ex: AUTRE_JUSTIF'},
            },
        },
        parameters={
            'scriptNrid': {'description': _('Script number'), 'example_value': '32071193360350'},
            'demand_number': {'description': _('Demand number'), 'example_value': 'D-20211123-0001'},
        },
    )
    def add_demand_file(self, request, scriptNrid, post_data, **kwargs):
        params = '%s;%s;%s;%s' % (
            post_data['demand_number'],
            post_data['file']['filename'],
            post_data['file']['content'],
            post_data['file_type'],
        )
        return self.execute_server_script(scriptNrid, params)
