# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2017  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import resource
import time
import traceback

from django.conf import settings
from django.core.management.base import BaseCommand, CommandError

from passerelle.base.models import BaseResource, ResourceLog
from passerelle.views import get_all_apps


class Command(BaseCommand):
    help = 'Execute scheduled commands'

    def add_arguments(self, parser):
        parser.add_argument(
            'frequency',
            metavar='FREQUENCY',
            type=str,
            help='every5min/hourly/daily/weekly/monthly/availability/jobs',
        )
        parser.add_argument(
            '--connector',
            dest='connector',
            metavar='CONNECTOR',
            type=str,
            help='limit updates to given connector type',
        )
        parser.add_argument(
            '--connector-slug',
            dest='slug',
            metavar='SLUG',
            type=str,
            help='limit updates to given connector slug',
        )

    def handle(self, frequency, **options):
        if frequency not in ('every5min', 'hourly', 'daily', 'weekly', 'monthly', 'availability', 'jobs'):
            raise CommandError('unknown frequency')
        errors = []

        if frequency == 'daily':
            ResourceLog.daily()

        for app in get_all_apps():
            for connector in app.objects.all():
                if options.get('connector') and connector.get_connector_slug() != options.get('connector'):
                    continue
                if options.get('slug') and connector.slug != options.get('slug'):
                    continue
                start_rusage = resource.getrusage(resource.RUSAGE_SELF)
                start_time = time.time()

                cron_method = getattr(connector, frequency)

                # method <frequency>() was not overloaded
                if cron_method.__func__ is getattr(BaseResource, frequency) and frequency != 'jobs':
                    continue

                log_function = connector.logger.info
                log_string = 'job "%s" of "%s.%s" finished without error (took %.1f real seconds, %.1f user+sys seconds)'
                try:
                    try:
                        cron_method()
                    finally:
                        real_time = time.time() - start_time
                        end_rusage = resource.getrusage(resource.RUSAGE_SELF)
                        user_sys_time = (
                            end_rusage.ru_utime
                            + end_rusage.ru_stime
                            - (start_rusage.ru_utime + start_rusage.ru_stime)
                        )
                    # warns if a cron job takes more than 5 minutes
                    if real_time > (5 * 60):
                        log_function = connector.logger.warning
                except Exception as e:
                    log_function = connector.logger.exception
                    log_string = 'job "%s" of "%s.%s" failed (took %.1f real seconds, %.1f user+sys seconds)'
                    errors.append(
                        {'connector': connector, 'exception': e, 'traceback': traceback.format_exc()}
                    )
                log_function(
                    log_string,
                    frequency,
                    connector.get_connector_slug(),
                    connector.slug,
                    real_time,
                    user_sys_time,
                )
        if errors:
            for error in errors:
                if options['verbosity'] >= 1:
                    print(
                        repr(error['connector']),
                    )
                    print(
                        '  url:',
                        getattr(settings, 'SITE_BASE_URL', '') + error['connector'].get_absolute_url(),
                    )
                    print('  error:', error['exception'])
                if options['verbosity'] >= 2:
                    print('  traceback:')
                    print(error['traceback'])
                    print()
            raise CommandError('error running jobs')
