# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2022  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
import datetime
import uuid
from datetime import timezone

import pytest

from passerelle.apps.qrcode.models import Certificate, QRCodeConnector, Reader
from tests.utils import generic_endpoint_url, setup_access_rights


@pytest.fixture()
def connector(db):
    return setup_access_rights(
        QRCodeConnector.objects.create(
            slug='test',
            key='5e8176e50d45b67e9db875d6006edf3ba805ff4ef4d945327012db4c797be1be',
        )
    )


def test_save_certificate(app, connector):
    endpoint = generic_endpoint_url('qrcode', 'save-certificate', slug=connector.slug)

    result = app.post_json(
        endpoint,
        params={
            'data': {
                'first_name': 'Georges',
                'last_name': 'Abitbol',
            },
            'validity_start': '2022-01-01 10:00:00+00:00',
            'validity_end': '2023-01-01 10:00:00+00:00',
        },
    )

    assert result.json['err'] == 0

    certificate_uuid = result.json['data']['uuid']
    assert result.json['data']['qrcode_url'] == f'http://testserver/qrcode/test/get-qrcode/{certificate_uuid}'
    certificate = connector.certificates.get(uuid=certificate_uuid)

    assert certificate.data['first_name'] == 'Georges'
    assert certificate.data['last_name'] == 'Abitbol'
    assert certificate.validity_start == datetime.datetime(2022, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc)
    assert certificate.validity_end == datetime.datetime(2023, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc)

    result = app.post_json(
        f'{endpoint}/{certificate_uuid}',
        params={
            'data': {
                'first_name': 'Robert',
                'last_name': 'Redford',
            },
            'validity_start': '2024-01-01T10:00:00+00:00',
            'validity_end': '2025-01-01T10:00:00+00:00',
        },
    )

    certificate.refresh_from_db()
    assert certificate.data['first_name'] == 'Robert'
    assert certificate.data['last_name'] == 'Redford'
    assert certificate.validity_start == datetime.datetime(2024, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc)
    assert certificate.validity_end == datetime.datetime(2025, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc)


def test_get_certificate(app, connector):
    certificate = connector.certificates.create(
        data={
            'first_name': 'Georges',
            'last_name': 'Abitbol',
        },
        validity_start=datetime.datetime(2022, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc),
        validity_end=datetime.datetime(2023, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc),
    )

    endpoint = generic_endpoint_url('qrcode', 'get-certificate', slug=connector.slug)
    result = app.get(f'{endpoint}/{certificate.uuid}')

    assert result.json == {
        'err': 0,
        'data': {
            'uuid': str(certificate.uuid),
            'data': {'first_name': 'Georges', 'last_name': 'Abitbol'},
            'validity_start': '2022-01-01T10:00:00+00:00',
            'validity_end': '2023-01-01T10:00:00+00:00',
            'qrcode_url': f'http://testserver/qrcode/test/get-qrcode/{certificate.uuid}',
        },
    }


def test_get_qrcode(app, connector):
    certificate = connector.certificates.create(
        uuid=uuid.UUID('12345678-1234-5678-1234-567812345678'),
        data={
            'first_name': 'Georges',
            'last_name': 'Abitbol',
        },
        validity_start=datetime.datetime(2022, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc),
        validity_end=datetime.datetime(2023, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc),
    )
    endpoint = generic_endpoint_url('qrcode', 'get-qrcode', slug=connector.slug)

    response = app.get(f'{endpoint}/{certificate.uuid}')
    assert response.headers['Content-Type'] == 'image/png'
    with open('tests/data/qrcode/test-qrcode.png', 'rb') as expected_qrcode:
        # just check images are the same. Decoded content is tested javascript-side.
        assert response.body == expected_qrcode.read()


def test_save_reader(app, connector):
    endpoint = generic_endpoint_url('qrcode', 'save-reader', slug=connector.slug)

    result = app.post_json(
        endpoint,
        params={
            'validity_start': '2022-01-01 10:00:00+00:00',
            'validity_end': '2023-01-01 10:00:00+00:00',
        },
    )

    assert result.json['err'] == 0

    reader_uuid = result.json['data']['uuid']
    assert result.json['data']['url'] == f'http://testserver/qrcode/test/open-reader/{reader_uuid}'
    reader = connector.readers.get(uuid=reader_uuid)

    assert reader.validity_start == datetime.datetime(2022, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc)
    assert reader.validity_end == datetime.datetime(2023, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc)

    result = app.post_json(
        f'{endpoint}/{reader_uuid}',
        params={
            'validity_start': '2024-01-01T10:00:00+00:00',
            'validity_end': '2025-01-01T10:00:00+00:00',
        },
    )

    reader.refresh_from_db()
    assert reader.validity_start == datetime.datetime(2024, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc)
    assert reader.validity_end == datetime.datetime(2025, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc)


def test_get_reader(app, connector):
    reader = connector.readers.create(
        validity_start=datetime.datetime(2022, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc),
        validity_end=datetime.datetime(2023, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc),
    )

    endpoint = generic_endpoint_url('qrcode', 'get-reader', slug=connector.slug)
    result = app.get(f'{endpoint}/{reader.uuid}')

    assert result.json == {
        'err': 0,
        'data': {
            'uuid': str(reader.uuid),
            'validity_start': '2022-01-01T10:00:00+00:00',
            'validity_end': '2023-01-01T10:00:00+00:00',
            'url': f'http://testserver/qrcode/test/open-reader/{reader.uuid}',
        },
    }


def test_open_reader(app, connector, freezer):
    reader = connector.readers.create(
        validity_start=datetime.datetime(2022, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc),
        validity_end=datetime.datetime(2023, 1, 1, 10, 0, 0, 0, tzinfo=timezone.utc),
    )

    endpoint = generic_endpoint_url('qrcode', 'open-reader', slug=connector.slug)
    freezer.move_to('2022-01-01T09:59:59')
    result = app.get(f'{endpoint}/{reader.uuid}')

    assert 'Reader isn\'t usable yet' in result.body.decode('utf-8')

    freezer.move_to('2022-01-01T10:00:00')
    result = app.get(f'{endpoint}/{reader.uuid}')

    assert result.pyquery(f'qrcode-reader[verify-key="{connector.hex_verify_key}"]')

    freezer.move_to('2023-01-01T10:00:01')
    result = app.get(f'{endpoint}/{reader.uuid}')

    assert 'Reader has expired.' in result.body.decode('utf-8')


MISSING = object()


@pytest.mark.parametrize('value', [MISSING, None, ''], ids=['missing', 'null', 'empty string'])
class TestOptional:
    def test_certificate_validity_start(self, value, app, connector):
        params = {
            'data': {
                'first_name': 'Georges',
                'last_name': 'Abitbol',
            },
            'validity_end': '2023-01-01 10:00:00+00:00',
        }
        if value is not MISSING:
            params['validity_start'] = value

        app.post_json('/qrcode/test/save-certificate/', params=params)
        assert Certificate.objects.get().validity_start is None

    def test_certificate_validity_end(self, value, app, connector):
        params = {
            'data': {
                'first_name': 'Georges',
                'last_name': 'Abitbol',
            },
            'validity_start': '2023-01-01 10:00:00+00:00',
        }
        if value is not MISSING:
            params['validity_end'] = value

        app.post_json('/qrcode/test/save-certificate/', params=params)
        assert Certificate.objects.get().validity_end is None

    def test_reader_validity_start(self, value, app, connector):
        params = {
            'validity_end': '2023-01-01 10:00:00+00:00',
        }
        if value is not MISSING:
            params['validity_start'] = value

        app.post_json('/qrcode/test/save-reader/', params=params)
        assert Reader.objects.get().validity_start is None

    def test_reader_validity_end(self, value, app, connector):
        params = {
            'validity_start': '2023-01-01 10:00:00+00:00',
        }
        if value is not MISSING:
            params['validity_end'] = value

        app.post_json('/qrcode/test/save-reader/', params=params)
        assert Reader.objects.get().validity_end is None
