import base64
import json

import pytest
import responses
from django.contrib.contenttypes.models import ContentType

from passerelle.apps.atal_rest.models import AtalREST
from passerelle.base.models import AccessRight, ApiUser


@pytest.fixture()
def connector(db):
    api = ApiUser.objects.create(username='all', keytype='', key='')
    connector = AtalREST.objects.create(base_url='https://atal.invalid', slug='test', api_key='secret')
    obj_type = ContentType.objects.get_for_model(connector)
    AccessRight.objects.create(
        codename='can_access', apiuser=api, resource_type=obj_type, resource_pk=connector.pk
    )
    return connector


def test_check_status(connector):
    with responses.RequestsMock() as rsps:
        rsps.get(
            'https://atal.invalid/api/Test',
            status=200,
            body=b'ok',
        )
        connector.check_status()
        assert len(rsps.calls) == 1
        assert rsps.calls[0].request.headers['X-API-Key'] == 'secret'


def test_worksrequest(app, connector):
    with responses.RequestsMock() as rsps:
        rsps.post(
            'https://atal.invalid/api/WorksRequests',
            status=200,
            json={'id': '1'},
        )
        params = {
            'desired_date': '2023-06-28',
            'object': "test entr'ouvert Publik",
            'recipient_id': '56',
            'requester_id': '12',
            'requesting_department_id': '17',
        }
        resp = app.post_json('/atal-rest/test/worksrequests', params=params)
        json_resp = resp.json
        assert json_resp['err'] == 0
        assert json_resp['data']['id'] == '1'
        request_data = json.loads(rsps.calls[0].request.body)
        assert request_data['DesiredDate'] == '2023-06-28'


def test_worksrequest_status(app, connector):
    with responses.RequestsMock() as rsps:
        rsps.get(
            'https://atal.invalid/api/WorksRequests/1',
            status=200,
            json={'status': 'foo'},
        )
        resp = app.get('/atal-rest/test/worksrequest-status?worksrequests_id=1')
        json_resp = resp.json
        assert json_resp['err'] == 0
        assert json_resp['data']['status'] == 'foo'


def test_worksrequests_single_attachment(app, connector):
    with responses.RequestsMock() as rsps:
        rsps.post('https://atal.invalid/api/WorksRequests/1/Attachments', status=200, body=b'')
        params = {'file': {'filename': 'bla', 'content': base64.b64encode(b'bla').decode('utf-8')}}
        resp = app.post_json(
            '/atal-rest/test/worksrequests-single-attachment?worksrequests_id=1', params=params
        )
        json_resp = resp.json
        assert json_resp['err'] == 0


def test_worksrequests_attachments(app, connector):
    with responses.RequestsMock() as rsps:
        rsps.post('https://atal.invalid/api/WorksRequests/Attachments', status=200, body=b'')
        params = {
            'files': [
                {'filename': 'bla', 'content': base64.b64encode(b'bla').decode('utf-8')},
                {'filename': 'blo', 'content': base64.b64encode(b'blo').decode('utf-8')},
            ],
            'worksrequests_ids': ['0', '1'],
        }
        resp = app.post_json('/atal-rest/test/worksrequests-attachments', params=params)
        json_resp = resp.json
        assert json_resp['err'] == 0


def test_worksrequests_attachments_error(app, connector):
    with responses.RequestsMock() as rsps:
        rsps.post(
            'https://atal.invalid/api/WorksRequests/Attachments',
            status=400,
            json={
                'type': 'https://tools.ietf.org/html/rfc7231#section-6.5.1',
                'title': 'Bad Request',
                'status': 400,
                '"detail': 'No content","traceId":"00-1034a23a6cfbb7c508aa7e125a8e9a52-4570fc75745b7d1d-00',
            },
        )
        params = {
            'files': [
                {'filename': 'bla', 'content': base64.b64encode(b'bla').decode('utf-8')},
                {'filename': 'blo', 'content': base64.b64encode(b'blo').decode('utf-8')},
            ],
            'worksrequests_ids': ['0', '1'],
        }
        resp = app.post_json('/atal-rest/test/worksrequests-attachments', params=params)
        json_resp = resp.json
        assert json_resp['err'] == 1
        assert json_resp['data']['title'] == 'Bad Request'


def test_thirdparties_requesting_departments(app, connector):
    with responses.RequestsMock() as rsps:
        query_params = {'RequestType': '1001'}
        rsps.get(
            'https://atal.invalid/api/ThirdParties/RequestingDepartments',
            match=[responses.matchers.query_param_matcher(query_params)],
            status=200,
            json=[
                {
                    'Name': 'foo',
                    'Id': 1,
                },
                {
                    'Name': 'bar',
                    'Id': 2,
                },
            ],
        )
        resp = app.get(
            '/atal-rest/test/thirdparties-requesting-departments?request_type=1001',
        )
        json_resp = resp.json
        assert json_resp['err'] == 0
        assert json_resp['data'] == [
            {'Name': 'foo', 'Id': 1, 'id': 1, 'text': 'foo'},
            {'Name': 'bar', 'Id': 2, 'id': 2, 'text': 'bar'},
        ]


def test_users(app, connector):
    with responses.RequestsMock() as rsps:
        rsps.get(
            'https://atal.invalid/api/Users',
            status=200,
            json=[
                {
                    'Name': 'foo',
                    'Id': 1,
                },
                {
                    'Name': 'bar',
                    'Id': 2,
                },
            ],
        )
        resp = app.get(
            '/atal-rest/test/users',
        )
        json_resp = resp.json
        assert json_resp['err'] == 0
        assert json_resp['data'] == [
            {'Name': 'foo', 'Id': 1, 'id': 1, 'text': 'foo'},
            {'Name': 'bar', 'Id': 2, 'id': 2, 'text': 'bar'},
        ]
