# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2023 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import base64
from urllib import parse as urlparse

import requests
from django.core.exceptions import ValidationError
from django.db import models
from django.http import HttpResponse
from django.utils.translation import gettext_lazy as _
from requests.auth import HTTPBasicAuth

from passerelle.base.models import BaseResource
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError

DOCUMENT_CREATION_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'type': 'object',
    'required': ['type'],
    'additionalProperties': True,
    'properties': {'type': {'type': 'string'}},
}

DOCUMENT_FILE_UPLOAD_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'type': 'object',
    'required': ['file', 'field_name'],
    'additionalProperties': False,
    'properties': {
        'field_name': {
            'type': 'string',
            'description': _('Document file\'s field name'),
        },
        'file': {
            'title': _('File object'),
            'type': 'object',
            'properties': {
                'name': {
                    'type': 'string',
                    'description': _('Filename'),
                },
                'content': {
                    'type': 'string',
                    'description': _('Content'),
                },
                'content_type': {
                    'type': 'string',
                    'description': _('Content type'),
                },
            },
            'required': ['name', 'content'],
        },
    },
}


class Pastell(BaseResource):
    api_base_url = models.URLField(
        max_length=128,
        verbose_name=_('API base URL'),
    )
    token = models.CharField(max_length=128, blank=True, verbose_name=_('API token'))
    username = models.CharField(max_length=128, blank=True, verbose_name=_('API username'))
    password = models.CharField(max_length=128, blank=True, verbose_name=_('API password'))

    category = _('Business Process Connectors')

    class Meta:
        verbose_name = _('Pastell')

    def clean(self, *args, **kwargs):
        if not self.token and not self.username:
            raise ValidationError(_('API token or API username and API password should be defined.'))
        return super().clean(*args, **kwargs)

    @property
    def requests_options(self):
        options = {}
        if self.token:
            options['headers'] = {'Authorization': 'Bearer: %s' % self.token}
        elif self.username:
            options['auth'] = HTTPBasicAuth(self.username, self.password)
        return options

    def call(self, path, method='get', params=None, **kwargs):
        url = urlparse.urljoin(self.api_base_url, path)
        try:
            response = self.requests.request(
                url=url, method=method, params=params, **kwargs, **self.requests_options
            )
            response.raise_for_status()
        except (requests.Timeout, requests.RequestException) as e:
            raise APIError(str(e))
        return response

    def check_status(self):
        try:
            response = self.call('version')
        except APIError as e:
            raise Exception('Pastell server is down: %s' % e)
        return {'data': response.json()}

    @endpoint(description=_('List entities'))
    def entities(self, request):
        data = []
        response = self.call('entite')
        for item in response.json():
            item['id'] = item['id_e']
            item['text'] = item['denomination']
            data.append(item)
        return {'data': data}

    @endpoint(
        description=_('List entity documents'),
        parameters={'entity_id': {'description': _('Entity ID'), 'example_value': '42'}},
    )
    def documents(self, request, entity_id):
        data = []
        response = self.call('entite/%s/document' % entity_id)
        for item in response.json():
            item['id'] = item['id_d']
            item['text'] = item['titre']
            data.append(item)
        return {'data': data}

    @endpoint(
        description=_('Get document details'),
        parameters={
            'entity_id': {'description': _('Entity ID'), 'example_value': '42'},
            'document_id': {'description': _('Document ID'), 'example_value': 'hDWtdSC'},
        },
    )
    def document(self, request, entity_id, document_id):
        response = self.call('entite/%s/document/%s' % (entity_id, document_id))
        return {'data': response.json()}

    @endpoint(
        post={
            'description': _('Create a document for an entity'),
            'request_body': {'schema': {'application/json': DOCUMENT_CREATION_SCHEMA}},
        },
        parameters={
            'entity_id': {'description': _('Entity ID'), 'example_value': '42'},
        },
    )
    def create_document(self, request, entity_id, post_data):
        # create document
        response = self.call('entite/%s/document' % entity_id, 'post', params=post_data)
        document_id = response.json()['id_d']

        # update it with other attributes
        response = self.call('entite/%s/document/%s' % (entity_id, document_id), 'patch', params=post_data)

        return {'data': response.json()}

    @endpoint(
        post={
            'description': _('Upload a file to a document'),
            'request_body': {'schema': {'application/json': DOCUMENT_FILE_UPLOAD_SCHEMA}},
        },
        parameters={
            'entity_id': {'description': _('Entity ID'), 'example_value': '42'},
            'document_id': {'description': _('Document ID'), 'example_value': 'hDWtdSC'},
        },
    )
    def upload_document_file(self, request, entity_id, document_id, post_data):
        filename = post_data['file']['name']
        file_data = {
            'file_content': (
                filename,
                base64.b64decode(post_data['file']['content']),
                post_data['file'].get('content_type'),
            )
        }

        response = self.call(
            'entite/%s/document/%s/file/%s' % (entity_id, document_id, post_data['field_name']),
            'post',
            files=file_data,
            data={'file_name': filename},
        )
        return {'data': response.json()}

    @endpoint(
        description=_('Get document\'s file'),
        parameters={
            'entity_id': {'description': _('Entity ID'), 'example_value': '42'},
            'document_id': {'description': _('Document ID'), 'example_value': 'hDWtdSC'},
            'field_name': {
                'description': _('Document file\'s field name'),
                'example_value': 'document',
            },
        },
    )
    def get_document_file(self, request, entity_id, document_id, field_name):
        document = self.call('entite/%s/document/%s/file/%s' % (entity_id, document_id, field_name))
        response = HttpResponse(document.content, content_type=document.headers['Content-Type'])
        response['Content-Disposition'] = document.headers['Content-disposition']
        return response
