import base64

import pytest
import responses
from django.contrib.contenttypes.models import ContentType

from passerelle.apps.esup_signature.models import EsupSignature
from passerelle.base.models import AccessRight, ApiUser


@pytest.fixture()
def connector(db):
    api = ApiUser.objects.create(username='all', keytype='', key='')
    connector = EsupSignature.objects.create(
        base_url='https://esup-signature.invalid/',
        slug='esup-signature',
    )
    obj_type = ContentType.objects.get_for_model(connector)
    AccessRight.objects.create(
        codename='can_access', apiuser=api, resource_type=obj_type, resource_pk=connector.pk
    )
    return connector


def test_new(app, connector):
    params = {
        'file': {
            'filename': 'bla',
            'content': base64.b64encode(b'who what').decode(),
            'content_type': 'text/plain',
        },
        'recipients_emails/0': 'foo@invalid',
        'recipients_emails/1': 'bar@invalid',
        'eppn': 'baz@invalid',
        'title': 'a title',
    }
    with responses.RequestsMock() as rsps:
        rsps.post('https://esup-signature.invalid/ws/signrequests/new', status=200, json=9)
        resp = app.post_json('/esup-signature/esup-signature/new', params=params)
        assert len(rsps.calls) == 1
        assert rsps.calls[0].request.headers['Content-Type'].startswith('multipart/form-data')
        json_resp = resp.json
        assert json_resp['err'] == 0
        assert json_resp['data'] == 9


def test_status(app, connector):
    with responses.RequestsMock() as rsps:
        rsps.get('https://esup-signature.invalid/ws/signrequests/1', status=200, json={'status': 'completed'})
        resp = app.get('/esup-signature/esup-signature/status?signrequests_id=1')
        assert resp.json['err'] == 0
        assert resp.json['data']['status'] == 'completed'


def test_audit_trail(app, connector):
    with responses.RequestsMock() as rsps:
        rsps.get(
            'https://esup-signature.invalid/ws/signrequests/audit-trail/1',
            status=200,
            json={'id': 1, 'documentCheckSum': 'abc'},
        )
        resp = app.get('/esup-signature/esup-signature/audit-trail?signrequests_id=1')
        assert resp.json['err'] == 0
        assert resp.json['data']['id'] == 1
        assert resp.json['data']['documentCheckSum'] == 'abc'


def test_get_last_file(app, connector):
    with responses.RequestsMock() as rsps:
        headers = {'Content-Type': 'text/plain', 'Content-Disposition': 'attachment; filename=foo.txt'}
        rsps.get(
            'https://esup-signature.invalid/ws/signrequests/get-last-file/1',
            status=200,
            body='who hwat',
            headers=headers,
        )
        resp = app.get('/esup-signature/esup-signature/get-last-file?signrequests_id=1')
        assert resp.text == 'who hwat'
