# Copyright (C) 2022 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
import os
from unittest import mock

import pytest
from lxml import etree
from requests.exceptions import ConnectionError

from passerelle.contrib.toulouse_maelis.models import Link, ToulouseMaelis
from passerelle.utils.jsonresponse import APIError
from passerelle.utils.soap import SOAPError
from tests.utils import FakedResponse, generic_endpoint_url, setup_access_rights

TEST_BASE_DIR = os.path.join(os.path.dirname(__file__), 'data', 'toulouse_maelis')


def get_xml_file(filename):
    with open(os.path.join(TEST_BASE_DIR, filename), 'rb') as desc:
        return desc.read()


CONNECTION_ERROR = ConnectionError('No address associated with hostname')
FAMILY_SERVICE_WSDL = FakedResponse(content=get_xml_file('FamilyService.wsdl'), status_code=200)
ACTIVITY_SERVICE_WSDL = FakedResponse(content=get_xml_file('ActivityService.wsdl'), status_code=200)
INVOICE_SERVICE_WSDL = FakedResponse(content=get_xml_file('InvoiceService.wsdl'), status_code=200)
FAILED_AUTH = FakedResponse(content=get_xml_file('R_failed_authentication.xml'), status_code=500)
ISWSRUNNING_TRUE = FakedResponse(content=get_xml_file('R_is_ws_running.xml') % b'true', status_code=200)
ISWSRUNNING_FALSE = FakedResponse(content=get_xml_file('R_is_ws_running.xml') % b'false', status_code=200)
READ_FAMILY = FakedResponse(content=get_xml_file('R_read_family.xml'), status_code=200)
READ_RL1_FAMILY = FakedResponse(content=get_xml_file('R_read_family_with_only_rl1.xml'), status_code=200)
READ_CATEGORIES = FakedResponse(content=get_xml_file('R_read_category_list.xml'), status_code=200)
READ_CIVILITIES = FakedResponse(content=get_xml_file('R_read_civility_list.xml'), status_code=200)
READ_CSP = FakedResponse(content=get_xml_file('R_read_csp_list.xml'), status_code=200)
READ_DIETCODE = FakedResponse(content=get_xml_file('R_read_dietcode_list.xml'), status_code=200)
READ_ORGAN = FakedResponse(content=get_xml_file('R_read_organ_list.xml'), status_code=200)
READ_PAI = FakedResponse(content=get_xml_file('R_read_pai_list.xml'), status_code=200)
READ_QUALITIES = FakedResponse(content=get_xml_file('R_read_quality_list.xml'), status_code=200)
READ_QUOTIENT = FakedResponse(content=get_xml_file('R_read_quotient_list.xml'), status_code=200)
READ_SITUATIONS = FakedResponse(content=get_xml_file('R_read_situation_list.xml'), status_code=200)
READ_VACCIN = FakedResponse(content=get_xml_file('R_read_vaccin_list.xml'), status_code=200)
IS_RL_EXISTS_TRUE = FakedResponse(content=get_xml_file('R_is_rl_exists.xml') % b'true', status_code=200)
IS_RL_EXISTS_FALSE = FakedResponse(content=get_xml_file('R_is_rl_exists.xml') % b'false', status_code=200)
IS_CHILD_EXISTS_TRUE = FakedResponse(content=get_xml_file('R_is_child_exists.xml') % b'true', status_code=200)
IS_CHILD_EXISTS_FALSE = FakedResponse(
    content=get_xml_file('R_is_child_exists.xml') % b'false', status_code=200
)
CREATE_FAMILY = FakedResponse(content=get_xml_file('R_create_family.xml'), status_code=200)
CREATE_FAMILY_ERR = FakedResponse(content=get_xml_file('R_create_family_error.xml'), status_code=200)
CREATE_CHILD = FakedResponse(content=get_xml_file('R_create_child.xml'), status_code=200)
CREATE_CHILD_ERR = FakedResponse(content=get_xml_file('R_create_child_error.xml'), status_code=200)
UPDATE_FAMILY = FakedResponse(content=get_xml_file('R_update_family.xml'), status_code=200)
UPDATE_FAMILY_ERR = FakedResponse(content=get_xml_file('R_update_family_error.xml'), status_code=200)
UPDATE_FAMILY_500 = FakedResponse(content=get_xml_file('R_update_family_soap_error.xml'), status_code=500)
UPDATE_DIETCODE = FakedResponse(content=get_xml_file('R_update_child_dietcode.xml'), status_code=200)
UPDATE_PAI = FakedResponse(content=get_xml_file('R_update_child_pai.xml'), status_code=200)
UPDATE_PAI_500 = FakedResponse(content=get_xml_file('R_update_child_pai_soap_error.xml'), status_code=500)
UPDATE_FSL = FakedResponse(content=get_xml_file('R_update_child_fsl.xml'), status_code=200)
UPDATE_FSL_500 = FakedResponse(content=get_xml_file('R_update_child_fsl_soap_error.xml'), status_code=500)
UPDATE_MEDICAL = FakedResponse(content=get_xml_file('R_update_child_fsl.xml'), status_code=200)
UPDATE_MEDICAL_500 = FakedResponse(
    content=get_xml_file('R_update_child_medical_record_soap_error.xml'), status_code=500
)


def assert_sent_payload(mocked_post, query_file):
    soap_sent = etree.tostring(etree.fromstring(mocked_post.call_args.kwargs['data']), pretty_print=True)
    expected = etree.tostring(etree.fromstring(get_xml_file(query_file)), pretty_print=True)
    assert soap_sent.decode() == expected.decode()


def get_endpoint(name):
    url = generic_endpoint_url('toulouse-maelis', name)
    assert url == '/toulouse-maelis/test/%s' % name
    return url


@pytest.fixture
def con(db):
    return setup_access_rights(
        ToulouseMaelis.objects.create(
            slug='test',
            zeep_wsse_username='maelis-webservice',
            zeep_wsse_password='maelis-password',
        )
    )


@mock.patch('passerelle.utils.Request.get')
def test_call_with_wrong_wsdl_url(mocked_get, con):
    mocked_get.side_effect = CONNECTION_ERROR
    with pytest.raises(
        SOAPError,
        match='cannot be loaded: No address associated with hostname',
    ) as e:
        con.call('Family', 'isWSRunning')
    assert e.value.err == 1
    assert e.value.http_status == 200
    assert e.value.url == 'https://demo-toulouse.sigec.fr/maelisws-toulouse/services/FamilyService?wsdl'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_call_with_wrong_credentials(mocked_post, mocked_get, con):
    mocked_get.side_effect = [FAMILY_SERVICE_WSDL]
    mocked_post.side_effect = [FAILED_AUTH]
    with pytest.raises(
        APIError,
        match='The security token could not be authenticated or authorized',
    ) as e:
        con.call('Family', 'isWSRunning')
    assert e.value.__dict__ == {
        'err': 1,
        'log_error': False,
        'http_status': 200,
        'err_code': 'Family-isWSRunning-ns1:FailedAuthentication',
    }


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_call(mocked_post, mocked_get, con):
    mocked_get.side_effect = [FAMILY_SERVICE_WSDL]
    mocked_post.side_effect = [ISWSRUNNING_TRUE]
    resp = con.call('Family', 'isWSRunning')
    assert resp is True


@pytest.mark.parametrize(
    'get_responses, post_responses, exception',
    [
        (
            [FAMILY_SERVICE_WSDL, ACTIVITY_SERVICE_WSDL, INVOICE_SERVICE_WSDL],
            [ISWSRUNNING_TRUE, ISWSRUNNING_TRUE, ISWSRUNNING_TRUE],
            None,
        ),
        (
            [FAMILY_SERVICE_WSDL, ACTIVITY_SERVICE_WSDL, CONNECTION_ERROR],
            [ISWSRUNNING_TRUE, ISWSRUNNING_TRUE, ISWSRUNNING_TRUE],
            SOAPError,
        ),
        (
            [FAMILY_SERVICE_WSDL, ACTIVITY_SERVICE_WSDL, INVOICE_SERVICE_WSDL],
            [ISWSRUNNING_TRUE, ISWSRUNNING_TRUE, FAILED_AUTH],
            APIError,
        ),
        (
            [FAMILY_SERVICE_WSDL, ACTIVITY_SERVICE_WSDL, INVOICE_SERVICE_WSDL],
            [ISWSRUNNING_TRUE, ISWSRUNNING_TRUE, ISWSRUNNING_FALSE],
            AssertionError,
        ),
    ],
)
@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_check_status(mocked_post, mocked_get, get_responses, post_responses, exception, con):
    mocked_get.side_effect = get_responses
    mocked_post.side_effect = post_responses
    if exception:
        with pytest.raises(Exception):
            con.check_status()
    else:
        con.check_status()


def test_replace_null_values(con):
    payload = {
        'adresse': {
            'num': '169',
            'numComp': None,
            'street1': 'Chateau',
            'street2': None,
            'town': 'Paris',
            'zipcode': '75014',
        }
    }
    con.replace_null_values(payload)
    assert payload == {
        'adresse': {
            'num': '169',
            'numComp': '',
            'street1': 'Chateau',
            'street2': '',
            'town': 'Paris',
            'zipcode': '75014',
        }
    }


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_link(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_FAMILY
    url = get_endpoint('link')
    assert Link.objects.count() == 0

    params = {
        'family_id': '1312',
        'firstname': 'Jhon',
        'lastname': 'Doe',
        'dateBirth': '1938-07-26',
    }
    resp = app.post_json(url + '?NameID=local', params=params)
    assert Link.objects.count() == 1
    assert resp.json['err'] == 0
    assert resp.json['data'] == 'ok'

    params['lastname'] = 'John'
    resp = app.post_json(url + '?NameID=local', params=params)
    assert Link.objects.count() == 1
    assert resp.json['err'] == 'not-found'
    assert resp.json['err_desc'] == "RL1 does not match '1312' family"


def test_unlink(con, app):
    url = get_endpoint('unlink')
    Link.objects.create(resource=con, family_id='1312', name_id='local')

    resp = app.post_json(url + '?NameID=local')
    assert Link.objects.count() == 0
    assert resp.json['err'] == 0
    assert resp.json['data'] == 'ok'

    resp = app.post_json(url + '?NameID=local')
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_get_referential_using_cache(mocked_post, mocked_get, con, freezer):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_CATEGORIES

    freezer.move_to('2021-11-10 00:00')
    con.get_referential('Category')
    assert mocked_post.call_count == 1

    con.get_referential('Category')
    assert mocked_post.call_count == 1

    freezer.move_to('2021-11-10 02:00')
    con.get_referential('Category')
    assert mocked_post.call_count == 2


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_get_referential_value(mocked_post, mocked_get, con):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_CSP

    assert con.get_referential_value('CSP', '1') == 'AGRICULTEUR'
    assert con.get_referential_value('CSP', 'AGR') == 'AGRICULTEUR'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_get_referential_value_not_found(mocked_post, mocked_get, con, caplog):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_CIVILITIES

    assert con.get_referential_value('Civility', 'MR') == 'MR'
    assert len(caplog.records) == 1
    assert caplog.records[0].levelno == logging.WARNING
    assert caplog.records[0].message == "No 'MR' key into Maelis 'Civility' referential"


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_category_list(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_CATEGORIES
    url = get_endpoint('read-category-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert resp.json['data'] == [
        {'id': 'BI', 'text': 'BIPARENTALE'},
        {'id': 'ACCEUI', 'text': "FAMILLE D'ACCUEIL"},
        {'id': 'MONO', 'text': 'MONOPARENTALE'},
    ]


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_civility_list(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_CIVILITIES
    url = get_endpoint('read-civility-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert resp.json['data'] == [
        {"id": "MME", "text": "Madame"},
        {"id": "M.", "text": "Monsieur"},
    ]


def test_read_complement_list(con, app):
    url = get_endpoint('read-complement-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert resp.json['data'] == [
        {'id': 'B', 'text': 'bis'},
        {'id': 'T', 'text': 'ter'},
        {'id': 'Q', 'text': 'quater'},
    ]


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_csp_list(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_CSP
    url = get_endpoint('read-csp-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert resp.json['data'] == [
        {'id': '14', 'text': 'AGENT DE MAITRISE'},
        {'id': '1', 'text': 'AGRICULTEUR'},
        {'id': 'ARTI', 'text': 'ARTISAN'},
        {'id': '2', 'text': 'ARTISAN-COMMERCANT'},
        {'id': '15', 'text': 'AUTRES'},
        {'id': 'CADR', 'text': 'CADRE'},
        {'id': '13', 'text': 'CADRE SUPERIEUR'},
        {'id': '3', 'text': "CHEF D'ENTREPRISE"},
        {'id': 'CHOM', 'text': 'CHOMEUR'},
        {'id': 'COM', 'text': 'COMMERCANT'},
        {'id': '10', 'text': "DEMANDEUR D'EMPLOI"},
        {'id': 'DIV', 'text': 'DIVERS'},
        {'id': 'EMP', 'text': 'EMPLOYE'},
        {'id': 'ENS', 'text': 'ENSEIGNANT'},
        {'id': 'ETU', 'text': 'ETUDIANT'},
        {'id': '11', 'text': 'FONCTIONNAIRE'},
        {'id': 'MAIR', 'text': 'MAIRIE DE NICE'},
        {'id': 'OUV', 'text': 'OUVRIER'},
        {'id': 'PERENS', 'text': 'PERISCO ENSEIGNANT'},
        {'id': 'PEREXT', 'text': 'PERISCO EXTERNE'},
        {'id': 'PERMAI', 'text': 'PERISCO MAIRIE DE NICE'},
        {'id': 'PERANI', 'text': 'PERISCO S.ANIMATION'},
        {'id': '9', 'text': 'PROFESSION LIBERALE'},
        {'id': '12', 'text': 'RETRAITE'},
        {'id': 'RMI', 'text': "REVENU MINIMUM D'INSERTION"},
        {'id': '16', 'text': 'SANS PROFESSION'},
        {'id': 'SEC', 'text': 'SECRETAIRE'},
    ]


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_dietcode_list(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_DIETCODE
    url = get_endpoint('read-dietcode-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert resp.json['data'] == [
        {'id': 'BB', 'text': 'REPAS BEBE'},
        {'id': 'MSP', 'text': 'REPAS MOYEN SANS PORC'},
        {'id': 'MSV', 'text': 'REPAS MOYEN SANS VIANDE'},
        {'id': 'MST', 'text': 'REPAS MOYEN STANDARD'},
        {'id': 'STD', 'text': '1- REPAS STANDARD'},
        {'id': 'RSP', 'text': '2- RÉGIME SANS PORC'},
        {'id': 'RSV', 'text': '3- RÉGIME SANS VIANDE'},
        {'id': 'PAI', 'text': "4- PROTOCOLE D'ACCUEIL INDIVIDUALISÉ"},
    ]


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_organ_list(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_ORGAN
    url = get_endpoint('read-organ-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert len(resp.json['data']) == 92
    assert resp.json['data'][:5] == [
        {'id': 'A10004460232', 'text': 'LEVENS'},
        {'id': 'A10007752822', 'text': 'LA COLLE SUR LOUP'},
        {'id': 'A10001133770', 'text': 'ASSIM'},
        {'id': 'A10007751483', 'text': 'EZE'},
        {'id': 'A10008152785', 'text': 'LUCERAM'},
    ]


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_pai_list(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_PAI
    url = get_endpoint('read-pai-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert resp.json['data'] == [
        {'id': 'PAIALI', 'text': 'ALIMENTAIRE'},
        {'id': 'PAIO', 'text': 'AUTRE'},
        {'id': 'PAIMED', 'text': 'MEDICAL'},
        {'id': 'PAI2', 'text': 'MEDICAL ET ALIMENTAIRE'},
    ]


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_quality_list(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_QUALITIES
    url = get_endpoint('read-quality-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert resp.json['data'] == [
        {'id': 'AU', 'text': 'AUTRE'},
        {'id': 'BP', 'text': 'BEAU PERE'},
        {'id': 'BM', 'text': 'BELLE MERE'},
        {'id': 'CONSO', 'text': 'CONSOMMATEUR'},
        {'id': 'EN', 'text': 'ENFANT'},
        {'id': 'FS', 'text': 'FRERES ET SOEURS'},
        {'id': 'GM', 'text': 'GRAND MERE MATERNELLE'},
        {'id': 'GP', 'text': 'GRAND PERE MATERNEL'},
        {'id': 'GMP', 'text': 'GRAND-MERE PATERNELLE'},
        {'id': 'GPP', 'text': 'GRAND-PERE PATERNEL'},
        {'id': 'MAIRIE', 'text': 'MAIRIE'},
        {'id': 'MERE', 'text': 'MERE'},
        {'id': 'O', 'text': 'ONCLE'},
        {'id': 'OS', 'text': 'ORGANISME SOCIAL'},
        {'id': 'PERE', 'text': 'PERE'},
        {'id': 'T', 'text': 'TANTE'},
        {'id': 'TUTEUR', 'text': 'TUTEUR'},
    ]


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_quotient_list(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_QUOTIENT
    url = get_endpoint('read-quotient-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert resp.json['data'] == [
        {'id': 'QF', 'text': 'GRILLE DE QUOTIENT'},
        {'id': 'QCLSH', 'text': 'QUOTIENT CENTRE AERE'},
        {'id': 'QJEU', 'text': 'QUOTIENT JEUNESSE'},
        {'id': 'QS', 'text': 'QUOTIENT SCOLAIRE'},
        {'id': 'QSPORT', 'text': 'QUOTIENT SPORT'},
        {'id': 'MOY ECO', 'text': 'REVENU MOYEN ( MENSUEL OU ANNUEL)'},
    ]


def test_read_sex_list(con, app):
    url = get_endpoint('read-sex-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert resp.json['data'] == [
        {'id': 'M', 'text': 'Masculin'},
        {'id': 'F', 'text': 'Féminin'},
    ]


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_situation_list(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_SITUATIONS
    url = get_endpoint('read-situation-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert resp.json['data'] == [
        {'id': 'C', 'text': 'Célibataire'},
        {'id': 'D', 'text': 'Divorcé (e)'},
        {'id': 'CS', 'text': 'EN COURS DE SEPARATION'},
        {'id': 'M', 'text': 'Marié (e)'},
        {'id': 'P', 'text': 'Pacsé (e)'},
        {'id': 'S', 'text': 'Séparé (e)'},
        {'id': 'UL', 'text': 'UNION LIBRE'},
        {'id': 'V', 'text': 'Veuf (ve)'},
        {'id': 'VM', 'text': 'Vivant maritalement'},
    ]


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_vaccin_list(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_VACCIN
    url = get_endpoint('read-vaccin-list')

    resp = app.get(url)
    assert resp.json['err'] == 0
    assert len(resp.json['data']) == 20
    assert resp.json['data'][:5] == [
        {'id': 'BCG', 'text': 'BCG'},
        {'id': 'CIB', 'text': 'CONTRE-INDICATION  TEMPORAIRE AU BCG'},
        {'id': 'MONO', 'text': 'CONTROLE DU BCG (+)'},
        {'id': 'MONON', 'text': 'CONTROLE DU BCG (-)'},
        {'id': 'DTC', 'text': 'DIPHTERIE TETANOS COQUELUCHE'},
    ]


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_family(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [
        READ_FAMILY,
        READ_CATEGORIES,
        READ_SITUATIONS,
        READ_CIVILITIES,
        READ_QUALITIES,
        READ_CSP,
        READ_DIETCODE,
        READ_PAI,
    ]
    url = get_endpoint('read-family')
    Link.objects.create(resource=con, family_id='1312', name_id='local')

    resp = app.get(url + '?NameID=local')
    assert resp.json['err'] == 0
    data = resp.json['data']
    del data['RL1']
    del data['RL2']
    del data['childList']
    del data['authorizedPersonList']
    del data['emergencyPersonList']
    assert data == {
        'number': 1312,
        'category': 'BI',
        'situation': 'M',
        'flagCom': True,
        'nbChild': 2,
        'nbTotalChild': None,
        'nbAES': None,
        'quotientList': [],
        'indicatorList': [],
        'childErrorList': [],
        'category_text': 'BIPARENTALE',
        'situation_text': 'Marié (e)',
    }
    data = resp.json['data']['RL1']
    del data['profession']
    assert data == {
        'num': '613878',
        'lastname': 'DOE',
        'firstname': 'JHON',
        'quality': 'PERE',
        'civility': 'M.',
        'dateBirth': '1938-07-26T00:00:00+01:00',
        'adresse': {
            'idStreet': None,
            'num': 170,
            'numComp': None,
            'street1': "Chateau d'eau",
            'street2': None,
            'town': 'Paris',
            'zipcode': '75014',
        },
        'contact': {
            'phone': 'non',
            'mobile': '0777777777',
            'mail': 'djhon@example.org',
            'isContactMail': True,
            'isContactSms': True,
            'isInvoicePdf': True,
        },
        'CAFInfo': None,
        'civility_text': 'Monsieur',
        'quality_text': 'PERE',
    }
    data = resp.json['data']['childList'][0]
    del data['fsl']
    del data['medicalRecord']
    assert data == {
        'num': '613880',
        'lastname': 'DOE',
        'firstname': 'JANNIS',
        'sexe': 'F',
        'sexe_text': 'Féminin',
        'birth': {'dateBirth': '1943-01-19T00:00:00+01:00', 'place': None},
        'dietcode': 'RSV',
        'dietcode_text': '3- RÉGIME SANS VIANDE',
        'bPhoto': False,
        'bLeaveAlone': False,
        'authorizedPersonList': [],
        'indicatorList': [],
        'subscribeSchoolList': [],
        'mother': {'num': 613963, 'civility': 'MME', 'firstname': 'JANE', 'lastname': 'DOE'},
        'father': {'num': 613878, 'civility': 'M.', 'firstname': 'JHON', 'lastname': 'DOE'},
        'rl': None,
        'subscribeActivityList': [],
        'paiInfoBean': {
            'code': 'PAIALI',
            'code_text': 'ALIMENTAIRE',
            'dateDeb': '2022-01-01T00:00:00+01:00',
            'dateFin': '2022-12-31T00:00:00+01:00',
            'description': 'bla bla PAI',
        },
    }
    assert resp.json['data']['childList'][0]['fsl'] == {
        'dateDeb': '2022-01-01T00:00:00+01:00',
        'dateFin': '2022-12-31T00:00:00+01:00',
        'allergieAlimentaire': True,
        'allergieRespiratoire': True,
        'allergieAutre': True,
        'allergieMedicament': True,
        'asthme': False,
        'flPAI': False,
        'flImage': False,
        'dtcPrap1': '2022-02-22T00:00:00+01:00',
        'obsMed1': 'some obsMed1 text',
        'obsMed2': 'some obsMed2 text',
        'obsMed3': 'some obsMed3 text',
        'obsDir1': 'some obsDir1 text',
        'obsDir2': 'some obsDir2 text',
        'obsDir3': 'some obsDir3 text',
        'obsAssist1': 'some obsAssist1 text',
        'obsAssist2': 'some obsAssist2 text',
        'obsAssist3': 'some obsAssist3 text',
        'cons1Med': 'some cons1Med text',
        'cons2Med': 'some cons2Med text',
    }
    assert resp.json['data']['childList'][0]['medicalRecord'] == {
        "familyDoctor": {
            "name": "DRE",
            "phone": "0612341234",
            "address": {"street1": "Alameda", "zipcode": "90220", "town": "Compton"},
        },
        "vaccinList": [
            {
                "code": "ROR",
                "label": "ROUGEOLE-OREILLONS-RUBEOLE",
                "vaccinationDate": "2012-02-22T00:00:00+01:00",
            },
            {
                "code": "DTC",
                "label": "DIPHTERIE TETANOS COQUELUCHE",
                "vaccinationDate": "2011-01-11T00:00:00+01:00",
            },
            {
                "code": "ROR",
                "label": "ROUGEOLE-OREILLONS-RUBEOLE",
                "vaccinationDate": "1970-01-11T00:00:00+01:00",
            },
        ],
    }
    assert resp.json['data']['emergencyPersonList'][0] == {
        'firstname': 'KENY',
        'lastname': 'ARKANA',
        'dateBirth': '1982-12-20T00:00:00+01:00',
        'quality': 'T',
        'civility': 'MME',
        'contact': {'phone': '0123456789', 'mobile': None, 'mail': 'pueblo@example.org'},
        'numPerson': 614059,
        'civility_text': 'Madame',
        'quality_text': 'TANTE',
    }


def test_read_family_not_linked_error(con, app):
    url = get_endpoint('read-family')

    resp = app.get(url + '?NameID=')
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_rl1(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [
        READ_FAMILY,
        READ_CIVILITIES,
        READ_QUALITIES,
        READ_CSP,
    ]
    url = get_endpoint('read-rl')
    Link.objects.create(resource=con, family_id='1312', name_id='local')

    resp = app.get(url + '?NameID=local&rl_id=613878')
    assert resp.json['err'] == 0
    assert resp.json['data']['firstname'] == 'JHON'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_rl2(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [
        READ_FAMILY,
        READ_CIVILITIES,
        READ_QUALITIES,
        READ_CSP,
    ]
    url = get_endpoint('read-rl')
    Link.objects.create(resource=con, family_id='1312', name_id='local')

    resp = app.get(url + '?NameID=local&rl_id=613879')
    assert resp.json['err'] == 0
    assert resp.json['data'] == {
        'num': '613879',
        'lastname': 'DOE',
        'firstname': 'JANE',
        'quality': 'MERE',
        'civility': 'MME',
        'dateBirth': '1940-06-22T00:00:00+02:00',
        'adresse': {
            'idStreet': None,
            'num': 170,
            'numComp': None,
            'street1': "Chateau d'eau",
            'street2': None,
            'town': 'Paris',
            'zipcode': '75014',
        },
        'contact': {
            'phone': None,
            'mobile': None,
            'mail': None,
            'isContactMail': False,
            'isContactSms': False,
            'isInvoicePdf': False,
        },
        'profession': {
            'codeCSP': 'ART',
            'profession': 'informaticien',
            'employerName': 'EO',
            'phone': '0123456789',
            'addressPro': {'num': None, 'street': None, 'zipcode': None, 'town': 'Orléans'},
            'codeCSP_text': 'ARTISAN',
        },
        'CAFInfo': {'number': '789', 'organ': 'ma caf'},
        'civility_text': 'Madame',
        'quality_text': 'MERE',
    }


def test_read_rl_not_linked_error(con, app):
    url = get_endpoint('read-rl')

    resp = app.get(url + '?NameID=local&rl_id=613879')
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_rl_not_found(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_FAMILY]
    url = get_endpoint('read-rl')
    Link.objects.create(resource=con, family_id='1312', name_id='local')

    resp = app.get(url + '?NameID=local&rl_id=000000')
    assert resp.json['err'] == 'not-found'
    assert resp.json['err_desc'] == "no '000000' RL on '1312' family"


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_person(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [
        READ_FAMILY,
        READ_CIVILITIES,
        READ_QUALITIES,
    ]
    url = get_endpoint('read-person')
    Link.objects.create(resource=con, family_id='1312', name_id='local')

    resp = app.get(url + '?NameID=local&person_id=614059&kind=emergency')
    assert resp.json['err'] == 0
    assert resp.json['data']['firstname'] == 'KENY'


def test_read_person_not_linked_error(con, app):
    url = get_endpoint('read-person')

    resp = app.get(url + '?NameID=local&person_id=614059')
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


def test_read_person_wrong_parameter(con, app):
    url = get_endpoint('read-person')
    Link.objects.create(resource=con, family_id='1312', name_id='local')

    resp = app.get(url + '?NameID=local&person_id=000000&kind=baby-sitter')
    assert resp.json['err'] == 1
    assert resp.json['err_desc'] == "wrong 'baby-sitter' value for kind parameter"


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_person_not_found(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_FAMILY]
    url = get_endpoint('read-person')
    Link.objects.create(resource=con, family_id='1312', name_id='local')

    resp = app.get(url + '?NameID=local&person_id=000000&kind=emergency')
    assert resp.json['err'] == 'not-found'
    assert resp.json['err_desc'] == "no '000000' emergency person on '1312' family"


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_child(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [
        READ_FAMILY,
        READ_DIETCODE,
        READ_PAI,
    ]
    url = get_endpoint('read-child')
    Link.objects.create(resource=con, family_id='1312', name_id='local')

    resp = app.get(url + '?NameID=local&child_id=613880')
    assert resp.json['err'] == 0
    assert resp.json['data']['firstname'] == 'JANNIS'


def test_read_child_not_linked_error(con, app):
    url = get_endpoint('read-child')

    resp = app.get(url + '?NameID=local&child_id=613880')
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_read_child_not_found(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [
        READ_FAMILY,
        READ_DIETCODE,
        READ_PAI,
    ]
    url = get_endpoint('read-child')
    Link.objects.create(resource=con, family_id='1312', name_id='local')

    resp = app.get(url + '?NameID=local&child_id=000000')
    assert resp.json['err'] == 'not-found'
    assert resp.json['err_desc'] == "no '000000' child on '1312' family"


@pytest.mark.parametrize(
    'post_response, result',
    [
        (IS_RL_EXISTS_TRUE, True),
        (IS_RL_EXISTS_FALSE, False),
    ],
)
@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_is_rl_exists(mocked_post, mocked_get, post_response, result, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = post_response
    url = get_endpoint('is-rl-exists')

    params = {
        'firstname': 'Jhon',
        'lastname': 'Doe',
        'dateBirth': '1938-07-26',
    }
    resp = app.post_json(url, params=params)
    assert resp.json['err'] == 0
    assert resp.json['data'] == result


def test_is_rl_exists_schema_error(con, app):
    url = get_endpoint('is-rl-exists')

    params = {
        'firstname': 'Jhon',
        'lastname': 'Doe',
        'dateBirth': '1938-07-26 more text',
    }
    resp = app.post_json(url, params=params, status=400)
    assert resp.json['err'] == 1
    assert "does not match '^[0-9]{4}-[0-9]{2}-[0-9]{2}$'" in resp.json['err_desc']


@pytest.mark.parametrize(
    'post_response, result',
    [
        (IS_CHILD_EXISTS_TRUE, True),
        (IS_CHILD_EXISTS_FALSE, False),
    ],
)
@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_is_child_exists(mocked_post, mocked_get, post_response, result, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = post_response
    url = get_endpoint('is-child-exists')

    params = {
        'firstname': 'Doe',
        'lastname': 'Jhonny allen',
        'dateBirth': '1942-11-27',
    }
    resp = app.post_json(url, params=params)
    assert resp.json['err'] == 0
    assert resp.json['data'] == result


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_family(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = CREATE_FAMILY
    url = get_endpoint('create-family')
    params = {
        'categorie': 'ACCEUI',
        'situation': 'C',
        'rl1/civility': 'M.',
        'rl1/firstname': 'Jhon',
        'rl1/lastname': 'Doe',
        'rl1/quality': 'AU',
        'rl1/dateBirth': '1938-07-26',
        'rl1/adresse/street1': 'Chateau',
        'rl1/adresse/town': 'Paris',
        'rl1/adresse/zipcode': '75014',
    }

    resp = app.post_json(url + '?NameID=local', params=params)
    assert_sent_payload(mocked_post, 'Q_create_family.xml')
    assert resp.json['err'] == 0
    assert resp.json['data'] == {
        'number': 196545,
        'password': '394634V2',
        'rl1ErrorList': [],
        'childErrorList': [],
    }
    assert Link.objects.get(resource=con, family_id='196545', name_id='local')


def test_create_family_already_linked_error(con, app):
    url = get_endpoint('create-family')
    params = {
        'categorie': 'ACCEUI',
        'situation': 'C',
        'rl1/civility': 'M.',
        'rl1/firstname': 'Jhon',
        'rl1/lastname': 'Doe',
        'rl1/quality': 'AU',
        'rl1/dateBirth': '1938-07-26',
        'rl1/adresse/street1': 'Chateau',
        'rl1/adresse/town': 'Paris',
        'rl1/adresse/zipcode': '75014',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'already-linked'
    assert resp.json['err_desc'] == 'User already linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_family_maelis_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = CREATE_FAMILY_ERR
    url = get_endpoint('create-family')
    params = {
        'categorie': 'ACCEUI',
        'situation': 'C',
        'rl1/civility': 'M.',
        'rl1/firstname': 'Jhon',
        'rl1/lastname': 'Doe',
        'rl1/quality': 'AU',
        'rl1/dateBirth': '1938-07-26',
        'rl1/adresse/street1': 'Chateau',
        'rl1/adresse/town': 'Paris',
        'rl1/adresse/zipcode': '75014',
    }

    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'E54a, E54a'
    assert 'Il existe déjà' in resp.json['err_desc']


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_family(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_FAMILY
    url = get_endpoint('update-family')
    params = {
        'categorie': 'BI',
        'situation': 'C',
        'rl1/civility': 'M.',
        'rl1/firstname': 'Jhon',
        'rl1/lastname': 'Doe',
        'rl1/quality': 'AU',
        'rl1/dateBirth': '1938-07-26',
        'rl1/adresse/street1': 'Chateau',
        'rl1/adresse/town': 'Paris',
        'rl1/adresse/zipcode': '75014',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert_sent_payload(mocked_post, 'Q_update_family.xml')
    assert resp.json['err'] == 0


def test_update_family_already_not_linked_error(con, app):
    url = get_endpoint('update-family')
    params = {
        'categorie': 'BI',
        'situation': 'C',
        'rl1/civility': 'M.',
        'rl1/firstname': 'Jhon',
        'rl1/lastname': 'Doe',
        'rl1/quality': 'AU',
        'rl1/dateBirth': '1938-07-26',
        'rl1/adresse/street1': 'Chateau',
        'rl1/adresse/town': 'Paris',
        'rl1/adresse/zipcode': '75014',
    }

    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_family_maelis_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_FAMILY_ERR
    url = get_endpoint('update-family')
    params = {
        'categorie': 'ACCEUI',
        'situation': 'C',
        'childList/0/lastname': 'Zimmerman',
        'childList/0/firstname': 'Robert',
        'childList/0/sexe': 'M',
        'childList/0/birth/dateBirth': '1941-05-24',
        'childList/0/birth/place': 'Duluth',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'E65a'
    assert "E65a : Il existe déjà un enfant" in resp.json['err_desc']


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_family_soap_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_FAMILY_500
    url = get_endpoint('update-family')
    params = {
        'nbChild': '100',
        'categorie': 'BI',
        'situation': 'C',
        'rl1/civility': 'M.',
        'rl1/firstname': 'Jhon',
        'rl1/lastname': 'Doe',
        'rl1/quality': 'AU',
        'rl1/dateBirth': '1938-07-26',
        'rl1/adresse/street1': 'Chateau',
        'rl1/adresse/town': 'Paris',
        'rl1/adresse/zipcode': '75014',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'Family-updateFamily-soap:Server'
    assert 'Une erreur est survenue' in resp.json['err_desc']


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_rl1(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = CREATE_FAMILY
    url = get_endpoint('create-rl1')
    params = {
        'categorie': 'ACCEUI',
        'situation': 'C',
        'rl1/civility': 'M.',
        'rl1/firstname': 'Jhon',
        'rl1/lastname': 'Doe',
        'rl1/quality': 'AU',
        'rl1/dateBirth': '1938-07-26',
        'rl1/adresse/street1': 'Chateau',
        'rl1/adresse/town': 'Paris',
        'rl1/adresse/zipcode': '75014',
    }

    resp = app.post_json(url + '?NameID=local', params=params)
    assert_sent_payload(mocked_post, 'Q_create_family.xml')
    assert resp.json['err'] == 0
    assert resp.json['data'] == {'family_id': 196545}
    assert Link.objects.get(resource=con, family_id='196545', name_id='local')


def test_create_rl1_already_linked_error(con, app):
    url = get_endpoint('create-rl1')
    params = {
        'categorie': 'ACCEUI',
        'situation': 'C',
        'rl1/civility': 'M.',
        'rl1/firstname': 'Jhon',
        'rl1/lastname': 'Doe',
        'rl1/quality': 'AU',
        'rl1/dateBirth': '1938-07-26',
        'rl1/adresse/street1': 'Chateau',
        'rl1/adresse/town': 'Paris',
        'rl1/adresse/zipcode': '75014',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'already-linked'
    assert resp.json['err_desc'] == 'User already linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_rl1_maelis_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = CREATE_FAMILY_ERR
    url = get_endpoint('create-rl1')
    params = {
        'categorie': 'ACCEUI',
        'situation': 'C',
        'rl1/civility': 'M.',
        'rl1/firstname': 'Jhon',
        'rl1/lastname': 'Doe',
        'rl1/quality': 'AU',
        'rl1/dateBirth': '1938-07-26',
        'rl1/adresse/street1': 'Chateau',
        'rl1/adresse/town': 'Paris',
        'rl1/adresse/zipcode': '75014',
    }

    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'already-rl1'
    assert 'Il existe déjà' in resp.json['err_desc']


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_rl1(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_RL1_FAMILY, UPDATE_FAMILY]
    url = get_endpoint('update-rl1')
    params = {
        'civility': 'M.',
        'firstname': 'Jhonny',
        'lastname': 'Doe',
        'quality': 'PERE',
        'dateBirth': '1943-06-15',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert_sent_payload(mocked_post, 'Q_update_rl1.xml')
    assert resp.json['err'] == 0


def test_update_rl1_not_linked_error(con, app):
    url = get_endpoint('update-rl1')
    params = {
        'civility': 'M.',
        'firstname': 'Jhonny',
        'lastname': 'Doe',
        'quality': 'PERE',
        'dateBirth': '1943-06-15',
    }

    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_rl1_connection_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = CONNECTION_ERROR
    url = get_endpoint('update-rl1')
    params = {
        'civility': 'M.',
        'firstname': 'Jhonny',
        'lastname': 'Doe',
        'quality': 'PERE',
        'dateBirth': '1943-06-15',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params, status=500)
    assert resp.json['err']
    assert resp.json['err_desc'] == 'No address associated with hostname'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_rl2(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_RL1_FAMILY, UPDATE_FAMILY]
    url = get_endpoint('create-rl2')
    params = {
        'civility': 'MME',
        'firstname': 'JANE',
        'lastname': 'DOE',
        'quality': 'MERE',
        'dateBirth': '1940-06-22',
        'adresse/num': '170',
        'adresse/street1': "Chateau d'eau",
        'adresse/town': 'Paris',
        'adresse/zipcode': '75014',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert_sent_payload(mocked_post, 'Q_create_rl2.xml')
    assert resp.json['err'] == 0
    assert resp.json['data'] == 'ok'


def test_create_rl2_not_linked_error(con, app):
    url = get_endpoint('create-rl2')
    params = {
        'civility': 'MME',
        'firstname': 'JANE',
        'lastname': 'DOE',
        'quality': 'MERE',
        'dateBirth': '1940-06-22',
        'adresse/num': '170',
        'adresse/street1': "Chateau d'eau",
        'adresse/town': 'Paris',
        'adresse/zipcode': '75014',
    }

    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_rl2_connection_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = CONNECTION_ERROR
    url = get_endpoint('create-rl2')
    params = {
        'civility': 'MME',
        'firstname': 'JANE',
        'lastname': 'DOE',
        'quality': 'MERE',
        'dateBirth': '1940-06-22',
        'adresse/num': '170',
        'adresse/street1': "Chateau d'eau",
        'adresse/town': 'Paris',
        'adresse/zipcode': '75014',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params, status=500)
    assert resp.json['err']
    assert resp.json['err_desc'] == 'No address associated with hostname'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_rl2_already_exists_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_FAMILY
    url = get_endpoint('create-rl2')
    params = {
        'civility': 'MME',
        'firstname': 'JANE',
        'lastname': 'DOE',
        'quality': 'MERE',
        'dateBirth': '1940-06-22',
        'adresse/num': '170',
        'adresse/street1': "Chateau d'eau",
        'adresse/town': 'Paris',
        'adresse/zipcode': '75014',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'already-rl2'
    assert resp.json['err_desc'] == 'RL2 already defined on family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_rl2(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_FAMILY, UPDATE_FAMILY]
    url = get_endpoint('update-rl2')
    params = {
        'civility': 'MME',
        'firstname': 'JANE',
        'lastname': 'DOE',
        'quality': 'MERE',
        'dateBirth': '1940-06-22',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert_sent_payload(mocked_post, 'Q_update_rl2.xml')
    assert resp.json['err'] == 0


def test_update_rl2_not_linked_error(con, app):
    url = get_endpoint('update-rl2')
    params = {
        'civility': 'MME',
        'firstname': 'JANE',
        'lastname': 'DOE',
        'quality': 'MERE',
        'dateBirth': '1940-06-22',
    }

    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_rl2_connection_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = CONNECTION_ERROR
    url = get_endpoint('update-rl2')
    params = {
        'civility': 'MME',
        'firstname': 'JANE',
        'lastname': 'DOE',
        'quality': 'MERE',
        'dateBirth': '1940-06-22',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params, status=500)
    assert resp.json['err']
    assert resp.json['err_desc'] == 'No address associated with hostname'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_rl2_not_exists_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_RL1_FAMILY
    url = get_endpoint('update-rl2')
    params = {
        'civility': 'MME',
        'firstname': 'JANE',
        'lastname': 'DOE',
        'quality': 'MERE',
        'dateBirth': '1940-06-22',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'no-rl2'
    assert resp.json['err_desc'] == 'No RL2 to update on family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_child(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = CREATE_CHILD
    url = get_endpoint('create-child')
    params = {
        'lastname': 'DOE',
        'firstname': 'JANNIS',
        'sexe': 'F',
        'birth/dateBirth': '1943-01-19',
        'birth/place': 'Port Arthur',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert_sent_payload(mocked_post, 'Q_create_child.xml')
    assert resp.json['err'] == 0
    assert resp.json['data'] == {'child_id': 613880}


def test_create_child_not_linked_error(con, app):
    url = get_endpoint('create-child')
    params = {
        'lastname': 'DOE',
        'firstname': 'JANNIS',
        'sexe': 'F',
        'birth/dateBirth': '1943-01-19',
        'birth/place': 'Port Arthur',
    }

    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_child_connection_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = CONNECTION_ERROR
    url = get_endpoint('create-child')
    params = {
        'lastname': 'DOE',
        'firstname': 'JANNIS',
        'sexe': 'F',
        'birth/dateBirth': '1943-01-19',
        'birth/place': 'Port Arthur',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params, status=500)
    assert resp.json['err']
    assert resp.json['err_desc'] == 'No address associated with hostname'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_child_maelis_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = CREATE_CHILD_ERR
    url = get_endpoint('create-child')
    params = {
        'lastname': 'DOE',
        'firstname': 'JANNIS',
        'sexe': 'F',
        'birth/dateBirth': '1943-01-19',
        'birth/place': 'Port Arthur',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local', params=params)
    assert resp.json['err'] == 'already-child'
    assert 'Il existe déjà' in resp.json['err_desc']


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_child(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_FAMILY, UPDATE_FAMILY]
    url = get_endpoint('update-child')
    params = {
        'lastname': 'DOE',
        'firstname': 'JANNIS',
        'sexe': 'F',
        'birth/dateBirth': '1943-01-19',
        'birth/place': 'Port Arthur',
        'bPhoto': True,
        'bLeaveAlone': False,
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&child_id=613880', params=params)
    assert_sent_payload(mocked_post, 'Q_update_child.xml')
    assert resp.json['err'] == 0


def test_update_child_not_linked_error(con, app):
    url = get_endpoint('update-child')
    params = {
        'lastname': 'DOE',
        'firstname': 'JANNIS',
        'sexe': 'F',
        'birth/dateBirth': '1943-01-19',
        'birth/place': 'Port Arthur',
    }

    resp = app.post_json(url + '?NameID=local&child_id=613880', params=params)
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_child_connection_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = CONNECTION_ERROR
    url = get_endpoint('update-child')
    params = {
        'lastname': 'DOE',
        'firstname': 'JANNIS',
        'sexe': 'F',
        'birth/dateBirth': '1943-01-19',
        'birth/place': 'Port Arthur',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&child_id=613880', params=params, status=500)
    assert resp.json['err']
    assert resp.json['err_desc'] == 'No address associated with hostname'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_child_not_exists_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = READ_RL1_FAMILY
    url = get_endpoint('update-child')
    params = {
        'lastname': 'DOE',
        'firstname': 'JANNIS',
        'sexe': 'F',
        'birth/dateBirth': '1943-01-19',
        'birth/place': 'Port Arthur',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&child_id=42', params=params)
    assert resp.json['err'] == 'no-child'
    assert resp.json['err_desc'] == 'No child 42 to update on family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_coordinate(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_FAMILY
    url = get_endpoint('update-coordinate')
    params = {
        'adresse/num': '169',
        'adresse/numComp': None,
        'adresse/street1': 'Château',
        'adresse/street2': None,
        'adresse/town': 'Paris',
        'adresse/zipcode': '75014',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&rl_id=613878', params=params)
    assert_sent_payload(mocked_post, 'Q_update_coordinate.xml')
    assert resp.json['err'] == 0
    assert resp.json['data'] == 'ok'


def test_update_coordinate_schema_error(con, app):
    url = get_endpoint('update-coordinate')
    params = {
        'contact/isContactMail': 'true more text',
    }

    resp = app.post_json(url + '?NameID=local&rl_id=613878', params=params, status=400)
    assert resp.json['err'] == 1
    assert resp.json['err_desc'] == "'true more text' is not of type 'boolean'"


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_create_person(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_FAMILY, UPDATE_FAMILY]
    url = get_endpoint('create-person')
    params = {
        'civility': None,
        'firstname': 'Mathias',
        'lastname': 'Cassel',
        'quality': 'O',
        'dateBirth': '1972-01-01',
        'contact/phone': None,
        'contact/mobile': '0623456789',
        'contact/mail': None,
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&kind=emergency', params=params)
    assert_sent_payload(mocked_post, 'Q_create_person.xml')
    assert resp.json['err'] == 0


def test_create_person_wrong_parameter(con, app):
    url = get_endpoint('create-person')
    params = {
        'civility': None,
        'firstname': 'Mathias',
        'lastname': 'Cassel',
        'quality': 'O',
        'dateBirth': '1972-01-01',
        'contact/phone': None,
        'contact/mobile': '0623456789',
        'contact/mail': None,
    }

    resp = app.post_json(url + '?NameID=local&kind=baby-sitter', params=params)
    assert resp.json['err'] == 1
    assert resp.json['err_desc'] == "wrong 'baby-sitter' value for kind parameter"


def test_create_person_not_linked_error(con, app):
    url = get_endpoint('create-person')
    params = {
        'civility': None,
        'firstname': 'Mathias',
        'lastname': 'Cassel',
        'quality': 'O',
        'dateBirth': '1972-01-01',
        'contact/phone': None,
        'contact/mobile': '0623456789',
        'contact/mail': None,
    }

    resp = app.post_json(url + '?NameID=local&kind=emergency', params=params)
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_person(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_FAMILY, UPDATE_FAMILY]
    url = get_endpoint('update-person')
    params = {
        'civility': None,
        'firstname': 'Mathias',
        'lastname': 'Cassel',
        'quality': 'O',
        'dateBirth': '1972-01-01',
        'contact/phone': None,
        'contact/mobile': '0623456789',
        'contact/mail': None,
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&person_id=614059&kind=emergency', params=params)
    assert_sent_payload(mocked_post, 'Q_update_person.xml')
    assert resp.json['err'] == 0


def test_update_person_wrong_parameter(con, app):
    url = get_endpoint('update-person')
    params = {
        'civility': None,
        'firstname': 'Mathias',
        'lastname': 'Cassel',
        'quality': 'O',
        'dateBirth': '1972-01-01',
        'contact/phone': None,
        'contact/mobile': '0623456789',
        'contact/mail': None,
    }

    resp = app.post_json(url + '?NameID=local&person_id=614059&kind=baby-sitter', params=params)
    assert resp.json['err'] == 1
    assert resp.json['err_desc'] == "wrong 'baby-sitter' value for kind parameter"


def test_update_person_not_linked_error(con, app):
    url = get_endpoint('update-person')
    params = {
        'civility': None,
        'firstname': 'Mathias',
        'lastname': 'Cassel',
        'quality': 'O',
        'dateBirth': '1972-01-01',
        'contact/phone': None,
        'contact/mobile': '0623456789',
        'contact/mail': None,
    }

    resp = app.post_json(url + '?NameID=local&person_id=614059&kind=emergency', params=params)
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_person_not_found(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_FAMILY]
    url = get_endpoint('update-person')
    params = {
        'civility': None,
        'firstname': 'Mathias',
        'lastname': 'Cassel',
        'quality': 'O',
        'dateBirth': '1972-01-01',
        'contact/phone': None,
        'contact/mobile': '0623456789',
        'contact/mail': None,
    }
    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&person_id=000000&kind=emergency', params=params)
    assert resp.json['err'] == 'not-found'
    assert resp.json['err_desc'] == "no '000000' authorized person on '1312' family"


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_delete_person(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_FAMILY, UPDATE_FAMILY]
    url = get_endpoint('delete-person')

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&person_id=614059&kind=emergency')
    assert_sent_payload(mocked_post, 'Q_delete_person.xml')
    assert resp.json['err'] == 0


def test_delete_person_wrong_parameter(con, app):
    url = get_endpoint('delete-person')

    resp = app.post_json(url + '?NameID=local&person_id=614059&kind=baby-sitter')
    assert resp.json['err'] == 1
    assert resp.json['err_desc'] == "wrong 'baby-sitter' value for kind parameter"


def test_delete_person_not_linked_error(con, app):
    url = get_endpoint('delete-person')

    resp = app.post_json(url + '?NameID=local&person_id=614059&kind=emergency')
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_delete_person_not_found(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.side_effect = [READ_FAMILY]
    url = get_endpoint('delete-person')

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&person_id=000000&kind=emergency')
    assert resp.json['err'] == 'not-found'
    assert resp.json['err_desc'] == "no '000000' authorized person on '1312' family"


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_child_dietcode(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_DIETCODE
    url = get_endpoint('update-child-dietcode')

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&child_id=613878&dietcode=RSV')
    assert_sent_payload(mocked_post, 'Q_update_child_dietcode.xml')
    assert resp.json['err'] == 0
    assert resp.json['data'] == 'ok'


def test_update_child_dietcode_not_linked_error(con, app):
    url = get_endpoint('update-child-dietcode')

    resp = app.post_json(url + '?NameID=local&child_id=613878&dietcode=RVS')
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_child_pai(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_PAI
    url = get_endpoint('update-child-pai')
    params = {
        'code': 'PAIALI',
        'dateDeb': '2022-01-01',
        'dateFin': '',
        'description': 'some text',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&child_id=613878', params=params)
    assert_sent_payload(mocked_post, 'Q_update_child_pai.xml')
    assert resp.json['err'] == 0
    assert resp.json['data'] == 'ok'


def test_update_child_pai_not_linked_error(con, app):
    url = get_endpoint('update-child-pai')
    params = {
        'code': 'PAIALI',
        'dateDeb': '2022-01-01',
        'dateFin': '',
        'description': 'some text',
    }

    resp = app.post_json(url + '?NameID=local&child_id=613878', params=params)
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_child_pai_soap_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_PAI_500
    url = get_endpoint('update-child-pai')
    params = {
        'code': 'PAIALI',
        'dateDeb': '2022-01-01',
        'dateFin': '',
        'description': 'a' * 501,
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&child_id=613878', params=params)
    assert resp.json['err'] == 'Family-updateChildPAI-soap:Server'
    assert 'Une erreur est survenue' in resp.json['err_desc']
    assert 'valeur trop grande' in resp.json['err_desc']
    assert 'maximum : 500' in resp.json['err_desc']


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_child_fsl(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_FSL
    url = get_endpoint('update-child-fsl')
    params = {
        'dateDeb': '2022-01-01',
        'dateFin': '2022-12-31',
        'allergieAlimentaire': True,
        'allergieRespiratoire': True,
        'allergieAutre': True,
        'allergieMedicament': True,
        'asthme': False,
        'flPAI': False,
        'flImage': False,
        'dtcPrap1': '2022-02-22',
        'obsMed1': 'some obsMed1 text',
        'obsMed2': 'some obsMed2 text',
        'obsMed3': 'some obsMed3 text',
        'obsDir1': 'some obsDir1 text',
        'obsDir2': 'some obsDir2 text',
        'obsDir3': 'some obsDir3 text',
        'obsAssist1': 'some obsAssist1 text',
        'obsAssist2': 'some obsAssist2 text',
        'obsAssist3': 'some obsAssist3 text',
        'cons1Med': 'some cons1Med text',
        'cons2Med': 'some cons2Med text',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&child_id=613878', params=params)
    assert_sent_payload(mocked_post, 'Q_update_child_fsl.xml')
    assert resp.json['err'] == 0
    assert resp.json['data'] == 'ok'


def test_update_child_fsl_not_linked_error(con, app):
    url = get_endpoint('update-child-fsl')
    params = {
        'dateDeb': '2022-01-01',
        'dateFin': '2022-12-31',
    }

    resp = app.post_json(url + '?NameID=local&child_id=613878', params=params)
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_child_fsl_soap_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_FSL_500
    url = get_endpoint('update-child-fsl')
    params = {
        'dateDeb': '2022-01-01',
        'dateFin': '2022-12-31',
        'obsMed1': 'a' * 71,
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&child_id=613878', params=params)
    assert resp.json['err'] == 'Family-createOrUpdateFSL-soap:Server'
    assert 'Une erreur est survenue' in resp.json['err_desc']
    assert 'valeur trop grande' in resp.json['err_desc']
    assert 'maximum : 70' in resp.json['err_desc']


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_child_medical_record(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_MEDICAL
    url = get_endpoint('update-child-medical-record')
    params = {
        'familyDoctor/name': 'Dre',
        'familyDoctor/phone': '0612341234',
        'familyDoctor/address/street1': 'Alameda',
        'familyDoctor/address/zipcode': '90220',
        'familyDoctor/address/town': 'Compton',
        'vaccinList/0/code': 'DTC',
        'vaccinList/0/vaccinationDate': '2011-01-11',
        'vaccinList/1/code': 'ROR',
        'vaccinList/1/vaccinationDate': '2022-02-22',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&child_id=613878', params=params)
    assert_sent_payload(mocked_post, 'Q_update_child_medical_record.xml')
    assert resp.json['err'] == 0
    assert resp.json['data'] == 'ok'


def test_update_child_medical_record_not_linked_error(con, app):
    url = get_endpoint('update-child-medical-record')

    resp = app.post_json(url + '?NameID=local&child_id=613878', params={})
    assert resp.json['err'] == 'not-linked'
    assert resp.json['err_desc'] == 'User not linked to family'


@mock.patch('passerelle.utils.Request.get')
@mock.patch('passerelle.utils.Request.post')
def test_update_child_medical_record_soap_error(mocked_post, mocked_get, con, app):
    mocked_get.return_value = FAMILY_SERVICE_WSDL
    mocked_post.return_value = UPDATE_MEDICAL_500
    url = get_endpoint('update-child-medical-record')
    params = {
        'vaccinList/0/vaccinationDate': '2022-02-31',
    }

    Link.objects.create(resource=con, family_id='1312', name_id='local')
    resp = app.post_json(url + '?NameID=local&child_id=613878', params=params)
    assert resp.json['err'] == 'Family-updateChildMedicalRecord-soap:Client'
    assert 'Unmarshalling Error' in resp.json['err_desc']
    assert 'pas une valeur de calendrier grégorien' in resp.json['err_desc']
