# -*- coding: utf-8 -*-
# Copyright (C) 2022 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import copy

BOOLEAN_TYPES = [
    {'type': 'boolean'},
    {
        'type': 'string',
        'pattern': '^([Oo][Uu][Ii]|[Nn][Oo][Nn]|[Tt][Rr][Uu][Ee]|[Ff][Aa][Ll][Ss][Ee]|1|0)$',
        'pattern_description': 'Les valeurs "0", "1", "true", "false", "oui" ou "non" sont autorisées (insensibles à la casse).',
    },
]

ID_PROPERTIES = {
    'firstname': {
        'description': 'Prénom',
        'type': 'string',
    },
    'lastname': {
        'description': 'Nom',
        'type': 'string',
    },
    'dateBirth': {
        'description': 'Date de naissance',
        'type': 'string',
        'pattern': '^[0-9]{4}-[0-9]{2}-[0-9]{2}$',
    },
}

LINK_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Link',
    'description': "Appairage d'un usager Publik à une famille dans Maelis",
    'type': 'object',
    'required': ['family_id', 'firstname', 'lastname', 'dateBirth'],
    'properties': {
        'family_id': {
            'description': 'Numéro DUI',
            'type': 'string',
        },
    },
}
LINK_SCHEMA['properties'].update(ID_PROPERTIES)

ISEXISTS_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Exist',
    'description': "Recherche d'un responsable légal ou d'un enfant dans Maelis",
    'type': 'object',
    'required': ['firstname', 'lastname', 'dateBirth'],
    'properties': ID_PROPERTIES,
}

ADDRESS_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Address',
    'description': 'Informations sur une adresse',
    'type': 'object',
    'required': ['street1', 'town', 'zipcode'],
    'properties': {
        'num': {
            'description': 'numéro',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'numComp': {
            'description': 'Complément du numéro (B, T ou Q)',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'street1': {
            'description': 'Libellé de la voie',
            'type': 'string',
        },
        'street2': {
            'description': 'Complément de la voie',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'town': {
            'description': 'Ville',
            'type': 'string',
        },
        'zipcode': {
            'description': 'Code postal',
            'type': 'string',
        },
    },
}

CONTACT_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Contact',
    'description': 'Informations sur le contact',
    'oneOf': [
        {'type': 'null'},
        {
            'type': 'object',
            'properties': {
                'phone': {
                    'description': 'Téléphone',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'mobile': {
                    'description': 'Portable',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'mail': {
                    'description': 'Mail',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'isContactMail': {
                    'description': 'Accepte de recevoir des mails',
                    'oneOf': BOOLEAN_TYPES,
                },
                'isContactSms': {
                    'description': 'Accepte de recevoir des sms',
                    'oneOf': BOOLEAN_TYPES,
                },
                'isInvoicePdf': {
                    'description': 'Accepte de ne plus recevoir de facture papier',
                    'oneOf': BOOLEAN_TYPES,
                },
            },
        },
    ],
}

ADDRESSPROF_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Adresse pro',
    'description': "Informations sur l'adresse professionnelle",
    'oneOf': [
        {'type': 'null'},
        {
            'type': 'object',
            'properties': {
                'num': {
                    'description': "Numéro de l'adresse",
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'street': {
                    'description': 'Nom de la voie',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'town': {
                    'description': 'Ville',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'zipcode': {
                    'description': 'Code postal',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
            },
        },
    ],
}

PROFESSION_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Profession',
    'description': 'Informations sur la profession',
    'oneOf': [
        {'type': 'null'},
        {
            'type': 'object',
            'properties': {
                'codeCSP': {
                    'description': 'Catégorie socio-professionnelle',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'profession': {
                    'description': 'Profession',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'employerName': {
                    'description': "Nom de l'employeur",
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'phone': {
                    'description': 'Téléphone',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'addressPro': ADDRESSPROF_SCHEMA,
            },
        },
    ],
}

CAFINFO_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'CAF',
    'description': 'Informations sur la CAF',
    'oneOf': [
        {'type': 'null'},
        {
            'type': 'object',
            'properties': {
                'number': {
                    'description': "Numéro d'allocataire",
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'organ': {
                    'description': "Nom de l'organisme",
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
            },
        },
    ],
}

RLINFO_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'RL',
    'description': "Informations sur le responsable légal",
    'type': 'object',
    'required': ['firstname', 'lastname', 'civility', 'quality', 'dateBirth', 'adresse'],
    'properties': {
        'civility': {
            'description': 'civilité (depuis référenciel)',
            'type': 'string',
        },
        'quality': {
            'description': 'Qualité',
            'type': 'string',
        },
        'adresse': ADDRESS_SCHEMA,
        'contact': CONTACT_SCHEMA,
        'profession': PROFESSION_SCHEMA,
        'CAFInfo': CAFINFO_SCHEMA,
    },
}
RLINFO_SCHEMA['properties'].update(ID_PROPERTIES)

CHILDBIRTH_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Child birth',
    'description': "Informations sur la naissance d'un enfant",
    'type': 'object',
    'required': ['dateBirth'],
    'properties': {
        'dateBirth': {
            'description': 'Date de naissance',
            'type': 'string',
            'pattern': '^[0-9]{4}-[0-9]{2}-[0-9]{2}$',
        },
        'place': {
            'description': 'Lieu de naissance',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
    },
}

FSL_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'FSL',
    'description': 'Informations sur la fiche sanitaire',
    'oneOf': [
        {'type': 'null'},
        {
            'type': 'object',
            'properties': {
                'dateDeb': {
                    'description': 'Date de début',
                    'type': 'string',
                    'pattern': '^[0-9]{4}-[0-9]{2}-[0-9]{2}$',
                },
                'dateFin': {
                    'description': 'Date de fin',
                    'type': 'string',
                    'pattern': '^[0-9]{4}-[0-9]{2}-[0-9]{2}$',
                },
                'allergieAlimentaire': {
                    'description': 'Allergie alimentaire',
                    'oneOf': [{'type': 'null'}] + BOOLEAN_TYPES,
                },
                'allergieRespiratoire': {
                    'description': 'Allergie respiratoire',
                    'oneOf': [{'type': 'null'}] + BOOLEAN_TYPES,
                },
                'allergieAutre': {
                    'description': 'Allergie autre',
                    'oneOf': [{'type': 'null'}] + BOOLEAN_TYPES,
                },
                'allergieMedicament': {
                    'description': 'Allergie médicament',
                    'oneOf': [{'type': 'null'}] + BOOLEAN_TYPES,
                },
                'asthme': {
                    'description': 'Asthmatique',
                    'oneOf': [{'type': 'null'}] + BOOLEAN_TYPES,
                },
                'flPAI': {
                    'description': 'PAI',
                    'oneOf': [{'type': 'null'}] + BOOLEAN_TYPES,
                },
                'flImage': {
                    'description': 'Autorisation photo',
                    'oneOf': [{'type': 'null'}] + BOOLEAN_TYPES,
                },
                'dtcPrap1': {
                    'description': 'Date du dernier rappel DT Polio',
                    'type': 'string',
                    'pattern': '^([0-9]{4}-[0-9]{2}-[0-9]{2}){0,1}$',
                },
                'obsMed1': {
                    'description': 'Observation médecin 1',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'obsMed2': {
                    'description': 'Observation médecin 2',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'obsMed3': {
                    'description': 'Observation médecin 3',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'obsDir1': {
                    'description': 'Observation directeur 1',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'obsDir2': {
                    'description': 'Observation directeur 2',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'obsDir3': {
                    'description': 'Observation directeur 3',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'obsAssist1': {
                    'description': 'Observation assistant sanitaire 1',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'obsAssist2': {
                    'description': 'Observation assistant sanitaire 2',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'obsAssist3': {
                    'description': 'Observation assistant sanitaire 3',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'cons1Med': {
                    'description': 'Conseil médecin 1',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'cons2Med': {
                    'description': 'Conseil médecin 2',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
            },
        },
    ],
}

DOCTORADDRESS_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Doctor address',
    'description': "Informations sur l'adresse du docteur",
    'type': 'object',
    'properties': {
        'street1': {
            'description': 'Libellé de la voie',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'town': {
            'description': 'Ville',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'zipcode': {
            'description': 'Code postal',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
    },
}

FAMILYDOCTOR_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Family doctor',
    'description': "Informations sur le docteur",
    'type': 'object',
    'properties': {
        'name': {
            'description': 'Nom',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'phone': {
            'description': 'Téléphone',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'address': DOCTORADDRESS_SCHEMA,
    },
}

VACCIN_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Vaccin',
    'description': "Informations sur le vaccin",
    'type': 'object',
    'properties': {
        'code': {
            'description': 'Code du vaccin',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'label': {
            'description': 'Nom du vaccin',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'vaccinationDate': {
            'description': 'Date du vaccin',
            'type': 'string',
            'pattern': '^[0-9]{4}-[0-9]{2}-[0-9]{2}$',
        },
    },
}

MEDICALRECORD_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Medical record',
    'description': "Informations médicales",
    'oneOf': [
        {'type': 'null'},
        {
            'type': 'object',
            'properties': {
                'familyDoctor': FAMILYDOCTOR_SCHEMA,
                'vaccinList': {
                    'oneOf': [
                        {'type': 'null'},
                        {
                            'type': 'array',
                            'items': VACCIN_SCHEMA,
                        },
                    ],
                },
            },
        },
    ],
    'unflatten': True,
}

PAIINFO_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'PAI',
    'description': "Informations médicales",
    'oneOf': [
        {'type': 'null'},
        {
            'type': 'object',
            'required': ['code'],
            'properties': {
                'code': {
                    'description': 'Code',
                    'type': 'string',
                },
                'dateDeb': {
                    'description': 'Date de début',
                    'type': 'string',
                    'pattern': '^([0-9]{4}-[0-9]{2}-[0-9]{2}){0,1}$',
                },
                'dateFin': {
                    'description': 'Date de fin',
                    'type': 'string',
                    'pattern': '^([0-9]{4}-[0-9]{2}-[0-9]{2}){0,1}$',
                },
                'description': {
                    'description': 'Texte libre de description (max 500 caractères)',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
            },
        },
    ],
}

CHILD_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Child',
    'description': "Informations sur la création d'un enfant",
    'type': 'object',
    'required': ['sexe', 'firstname', 'lastname'],
    'oneOf': [
        {'required': ['dateBirth']},  # createFamily
        {'required': ['birth']},  # updateFamily
    ],
    'properties': {
        'sexe': {
            'description': 'Sexe',
            'type': 'string',
        },
        'birth': CHILDBIRTH_SCHEMA,
        'dietcode': {
            'description': 'Code de régime alimentaire',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'bPhoto': {
            'description': 'Autorisation photo',
            'oneOf': BOOLEAN_TYPES,
        },
        'bLeaveAlone': {
            'description': 'Autorisation à partir seul',
            'oneOf': BOOLEAN_TYPES,
        },
        'fsl': FSL_SCHEMA,
        'medicalRecord': MEDICALRECORD_SCHEMA,
        'paiInfoBean': PAIINFO_SCHEMA,
    },
}
CHILD_SCHEMA['properties'].update(ID_PROPERTIES)

CONTACTLIGHT_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Contact light',
    'description': "Informations de contact pour les personnes autorisées à récupérer les enfants ou à prévenir en cas d'urgence",
    'oneOf': [
        {'type': 'null'},
        {
            'type': 'object',
            'properties': {
                'phone': {
                    'description': 'Téléphone',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'mobile': {
                    'description': 'Portable',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
                'mail': {
                    'description': 'Mail',
                    'oneOf': [{'type': 'null'}, {'type': 'string'}],
                },
            },
        },
    ],
}

FAMILYPERSON_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Family person',
    'description': "Informations sur les personnes autorisées à venir chercher les enfants ou à prévenir en cas d'urgence",
    'type': 'object',
    'required': ['firstname', 'lastname', 'dateBirth'],
    'properties': {
        'civility': {
            'description': 'civilité (depuis référenciel)',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'quality': {
            'description': 'Qualité',
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'contact': CONTACTLIGHT_SCHEMA,
    },
    'unflatten': True,
}
FAMILYPERSON_SCHEMA['properties'].update(ID_PROPERTIES)

AUTHORIZEDPERSON_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Family persons',
    'description': "Informations sur les personnes autorisées à venir chercher les enfants ou à prévenir en cas d'urgence",
    'type': 'object',
    'properties': {
        'personList': {
            'oneOf': [
                {'type': 'null'},
                {
                    'type': 'array',
                    'items': FAMILYPERSON_SCHEMA,
                },
            ],
        },
    },
}

UPDATE_FAMILY_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Family',
    'description': 'Informations pour créer ou mettre à jour une famille',
    'type': 'object',
    'required': ['categorie', 'situation'],
    'properties': {
        'categorie': {
            'description': 'Categorie (depuis référenciel)',
            'type': 'string',
        },
        'situation': {
            'description': 'Situation familiale (depuis référenciel)',
            'type': 'string',
        },
        'flagCom': {
            'description': 'Hors commune',
            'oneOf': BOOLEAN_TYPES,
        },
        'nbChild': {
            'description': "Nombre d'enfants à charge",
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'nbTotalChild': {
            'description': "Nombre total d'enfants",
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'nbAES': {
            'description': "Nombre d'AES",
            'oneOf': [{'type': 'null'}, {'type': 'string'}],
        },
        'rl1': RLINFO_SCHEMA,
        'rl2': RLINFO_SCHEMA,
        'authorizedPersonList': {
            'oneOf': [
                {'type': 'null'},
                {
                    'type': 'array',
                    'items': AUTHORIZEDPERSON_SCHEMA,
                },
            ],
        },
        'emergencyPersonList': {
            'oneOf': [
                {'type': 'null'},
                {
                    'type': 'array',
                    'items': AUTHORIZEDPERSON_SCHEMA,
                },
            ],
        },
        'childList': {
            'oneOf': [
                {'type': 'null'},
                {
                    'type': 'array',
                    'items': CHILD_SCHEMA,
                },
            ],
        },
    },
    'unflatten': True,
}

CREATE_FAMILY_SCHEMA = copy.deepcopy(UPDATE_FAMILY_SCHEMA)
CREATE_FAMILY_SCHEMA['required'] = ['rl1', 'categorie', 'situation']
create_family_child_schema = CREATE_FAMILY_SCHEMA['properties']['childList']['oneOf'][1]['items']
del create_family_child_schema['properties']['fsl']
del create_family_child_schema['properties']['bPhoto']
del create_family_child_schema['properties']['bLeaveAlone']
del create_family_child_schema['properties']['paiInfoBean']

CREATE_RL1_SCHEMA = copy.deepcopy(CREATE_FAMILY_SCHEMA)
del CREATE_RL1_SCHEMA['properties']['rl2']
del CREATE_RL1_SCHEMA['properties']['authorizedPersonList']
del CREATE_RL1_SCHEMA['properties']['emergencyPersonList']
del CREATE_RL1_SCHEMA['properties']['childList']
del CREATE_RL1_SCHEMA['properties']['rl1']['properties']['contact']
del CREATE_RL1_SCHEMA['properties']['rl1']['properties']['profession']
del CREATE_RL1_SCHEMA['properties']['rl1']['properties']['CAFInfo']

UPDATE_RL1_SCHEMA = copy.deepcopy(RLINFO_SCHEMA)
UPDATE_RL1_SCHEMA['required'] = ['firstname', 'lastname', 'civility', 'quality', 'dateBirth']
del UPDATE_RL1_SCHEMA['properties']['adresse']
del UPDATE_RL1_SCHEMA['properties']['contact']
del UPDATE_RL1_SCHEMA['properties']['profession']
del UPDATE_RL1_SCHEMA['properties']['CAFInfo']

CREATE_RL2_SCHEMA = copy.deepcopy(RLINFO_SCHEMA)
CREATE_RL2_SCHEMA['unflatten'] = True
del CREATE_RL2_SCHEMA['properties']['contact']
del CREATE_RL2_SCHEMA['properties']['profession']
del CREATE_RL2_SCHEMA['properties']['CAFInfo']

UPDATE_RL2_SCHEMA = copy.deepcopy(UPDATE_RL1_SCHEMA)

CREATE_CHILD_SCHEMA = copy.deepcopy(CHILD_SCHEMA)
CREATE_CHILD_SCHEMA['unflatten'] = True
del CREATE_CHILD_SCHEMA['properties']['dietcode']
del CREATE_CHILD_SCHEMA['properties']['bPhoto']
del CREATE_CHILD_SCHEMA['properties']['bLeaveAlone']
del CREATE_CHILD_SCHEMA['properties']['fsl']
del CREATE_CHILD_SCHEMA['properties']['medicalRecord']
del CREATE_CHILD_SCHEMA['properties']['paiInfoBean']

UPDATE_CHILD_SCHEMA = copy.deepcopy(CREATE_CHILD_SCHEMA)

UPDATE_COORDINATE_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'title': 'Update coordinate',
    'description': "Mise à jour des coordonnées d'un responsable légal",
    'type': 'object',
    'properties': {
        'adresse': ADDRESS_SCHEMA,
        'contact': CONTACT_SCHEMA,
        'profession': PROFESSION_SCHEMA,
        'CAFInfo': CAFINFO_SCHEMA,
    },
    'unflatten': True,
}
