# -*- coding: utf-8 -*-
# Copyright (C) 2022 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from urllib.parse import urljoin

import zeep
from django.core.cache import cache
from django.db import models
from zeep.helpers import serialize_object
from zeep.wsse.username import UsernameToken

from passerelle.base.models import BaseResource, HTTPResource
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError

from . import schemas


class ToulouseMaelis(BaseResource, HTTPResource):
    base_wsdl_url = models.CharField(
        max_length=128,
        blank=False,
        verbose_name='URL de base des WSDL',
        default='https://demo-toulouse.sigec.fr/maelisws-toulouse/services/',
    )
    zeep_wsse_username = models.CharField(
        max_length=64, blank=True, default='', verbose_name='Identifiant utilisateur WSSE'
    )
    zeep_wsse_password = models.CharField(
        max_length=64, blank=True, default='', verbose_name='Mot de passe WSSE'
    )

    category = 'Connecteurs métiers'
    _category_ordering = ['Famille', 'Activités']

    class Meta:
        verbose_name = 'Toulouse Maelis'

    def get_client(self, wsdl_short_name):
        wsse = UsernameToken(self.zeep_wsse_username, self.zeep_wsse_password)
        wsdl_name = wsdl_short_name + 'Service?wsdl'
        wsdl_url = urljoin(self.base_wsdl_url, wsdl_name)
        return self.soap_client(wsdl_url=wsdl_url, wsse=wsse)

    def call(self, wsdl_short_name, service, **kwargs):
        client = self.get_client(wsdl_short_name)
        method = getattr(client.service, service)
        try:
            return method(**kwargs)
        except zeep.exceptions.Fault as e:
            raise APIError(e.message, err_code='%s-%s-%s' % (wsdl_short_name, service, e.code))

    def check_status(self):
        assert self.call('Family', 'isWSRunning')
        assert self.call('Activity', 'isWSRunning')
        assert self.call('Invoice', 'isWSRunning')

    def get_referential(self, referential_name):

        # local referentials
        if referential_name == 'Complement':
            response = [
                {'id': 'B', 'text': 'bis'},
                {'id': 'T', 'text': 'ter'},
                {'id': 'Q', 'text': 'quater'},
            ]
            return {'list': response, 'dict': {x['id']: x['text'] for x in response}}
        elif referential_name == 'Sex':
            response = [
                {'id': 'M', 'text': 'Masculin'},
                {'id': 'F', 'text': 'Féminin'},
            ]
            return {'list': response, 'dict': {x['id']: x['text'] for x in response}}

        # remote referentials
        cache_key = 'maelis-%s-%s' % (self.pk, referential_name)
        data = cache.get(cache_key)
        if data is None:
            response = self.call('Family', 'read' + referential_name + 'List')
            if referential_name == 'Organ':
                data = {
                    'list': [{'id': x.id, 'text': x.code} for x in response],
                    'dict': {x.id: x.code for x in response},
                }
            else:
                data = {
                    'list': [{'id': x.code, 'text': x.libelle} for x in response],
                    'dict': {x.code: x.libelle for x in response},
                }
            # put in cache for two hours
            cache.set(cache_key, data, 3600 * 2)
        return data

    def get_referential_value(self, referential_name, key):
        try:
            return self.get_referential(referential_name)['dict'][key]
        except KeyError:
            # Maelis DB not properly configurated
            self.logger.warning("No '%s' key into Maelis '%s' referential", key, referential_name)
            return key

    def get_link(self, NameID):
        try:
            return self.link_set.get(name_id=NameID)
        except Link.DoesNotExist:
            raise APIError('User not linked to family', err_code='not-linked')

    def get_family_raw(self, family_id):
        response = self.call('Family', 'readFamily', dossierNumber=family_id)
        data = serialize_object(response)

        # make emergencyPersonList keys egal to authorizedPersonList
        for person in data['emergencyPersonList']:
            person['numPerson'] = person['num']
            del person['num']
            del person['id']
        return data

    def get_family(self, family_id):
        data = self.get_family_raw(family_id)

        def add_text_value(referential_name, data, keys):
            last_key = keys.pop()
            for key in keys:
                if not isinstance(data, dict) or not key in data:
                    return
                data = data[key]
            if isinstance(data, dict) and last_key in data and data[last_key] is not None:
                data[last_key + '_text'] = self.get_referential_value(referential_name, data[last_key])

        # add text from referentials
        add_text_value('Category', data, ['category'])
        add_text_value('Situation', data, ['situation'])
        for rlg in 'RL1', 'RL2':
            add_text_value('Civility', data, [rlg, 'civility'])
            add_text_value('Quality', data, [rlg, 'quality'])
            add_text_value('Complement', data, [rlg, 'adresse', 'numComp'])
            add_text_value('CSP', data, [rlg, 'profession', 'codeCSP'])
        for child in data['childList']:
            add_text_value('Sex', child, ['sexe'])
            add_text_value('DietCode', child, ['dietcode'])
            add_text_value('PAI', child, ['paiInfoBean', 'code'])
        for kind in ('authorized', 'emergency'):
            for person in data[kind + 'PersonList']:
                add_text_value('Civility', person, ['civility'])
                add_text_value('Quality', person, ['quality'])

        # convert O/N string into boolean
        for child in data['childList']:
            if child.get('fsl'):
                for key in (
                    'allergieAlimentaire',
                    'allergieRespiratoire',
                    'allergieAutre',
                    'allergieMedicament',
                    'asthme',
                    'flPAI',
                    'flImage',
                ):
                    child['fsl'][key] = bool(child['fsl'][key] == 'O')
        return data

    def replace_null_values(self, dico):
        '''send null fields as empty SOAP tag to tell maelis to empty the value'''
        for key, value in dico.items():
            if isinstance(value, dict):
                self.replace_null_values(value)
            if value is None:
                dico[key] = ''

    @endpoint(
        display_category='Famille',
        description='Liste des catégories',
        name='read-category-list',
        perm='can_access',
    )
    def read_category_list(self, request):
        return {'data': self.get_referential('Category')['list']}

    @endpoint(
        display_category='Famille',
        description='Liste des civilités',
        name='read-civility-list',
        perm='can_access',
    )
    def read_civility_list(self, request):
        return {'data': self.get_referential('Civility')['list']}

    @endpoint(
        display_category='Famille',
        description='Liste des compléments du numéro de voie',
        name='read-complement-list',
        perm='can_access',
    )
    def read_complement_list(self, request):
        return {'data': self.get_referential('Complement')['list']}

    @endpoint(
        display_category='Famille',
        description='liste des catégories socio-professionnelles',
        name='read-csp-list',
        perm='can_access',
    )
    def read_csp_list(self, request):
        data = self.get_referential('CSP')['list']

        # remove redundant codes
        uniq_text = set()
        uniq_data = []
        for item in data:
            item['text'] = item['text'].strip()
            if item['text'] not in uniq_text:
                uniq_data.append(item)
                uniq_text.add(item['text'])
        return {'data': uniq_data}

    @endpoint(
        display_category='Famille',
        description='Liste des régimes alimentaires',
        name='read-dietcode-list',
        perm='can_access',
    )
    def read_dietcode_list(self, request):
        return {'data': self.get_referential('DietCode')['list']}

    @endpoint(
        display_category='Famille',
        description='Liste des organismes (CAF)',
        name='read-organ-list',
        perm='can_access',
    )
    def read_organ_list(self, request):
        return {'data': self.get_referential('Organ')['list']}

    @endpoint(
        display_category='Famille',
        description="Liste des projet d'accueil individualisés",
        name='read-pai-list',
        perm='can_access',
    )
    def read_pai_list(self, request):
        return {'data': self.get_referential('PAI')['list']}

    @endpoint(
        display_category='Famille',
        description='liste des qualités du référenciel',
        name='read-quality-list',
        perm='can_access',
    )
    def read_quality_list(self, request):
        return {'data': self.get_referential('Quality')['list']}

    @endpoint(
        display_category='Famille',
        description='Liste des quotients',
        name='read-quotient-list',
        perm='can_access',
    )
    def read_quotient_list(self, request):
        return {'data': self.get_referential('Quotient')['list']}

    @endpoint(
        display_category='Famille',
        description='Liste des sexes',
        name='read-sex-list',
        perm='can_access',
    )
    def read_sex_list(self, request):
        return {'data': self.get_referential('Sex')['list']}

    @endpoint(
        display_category='Famille',
        description='liste des situations',
        name='read-situation-list',
        perm='can_access',
    )
    def read_situation_list(self, request):
        return {'data': self.get_referential('Situation')['list']}

    @endpoint(
        display_category='Famille',
        description='Liste des vaccins',
        name='read-vaccin-list',
        perm='can_access',
    )
    def read_vaccin_list(self, request):
        return {'data': self.get_referential('Vaccin')['list']}

    @endpoint(
        display_category='Famille',
        description='Lier un compte usager à une famille',
        perm='can_access',
        parameters={'NameID': {'description': 'Publik NameID'}},
        post={'request_body': {'schema': {'application/json': schemas.LINK_SCHEMA}}},
    )
    def link(self, request, NameID, post_data):
        family_id = post_data['family_id']
        response = self.call('Family', 'readFamily', dossierNumber=family_id)
        if not (
            response['RL1']['firstname'] == post_data['firstname'].upper()
            and response['RL1']['lastname'] == post_data['lastname'].upper()
            and response['RL1']['dateBirth'].strftime('%Y-%m-%d') == post_data['dateBirth']
        ):
            raise APIError("RL1 does not match '%s' family" % family_id, err_code='not-found')
        Link.objects.update_or_create(resource=self, name_id=NameID, defaults={'family_id': family_id})
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description='Supprimer une liaison entre un compte usager et une famille',
        methods=['post'],
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
        },
    )
    def unlink(self, request, NameID):
        link = self.get_link(NameID)
        link.delete()
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description='Informations sur la famille',
        perm='can_access',
        name='read-family',
        parameters={'NameID': {'description': 'Publik NameID'}},
    )
    def read_family(self, request, NameID):
        family_id = self.get_link(NameID).family_id
        data = self.get_family(family_id)
        return {'data': data}

    @endpoint(
        display_category='Famille',
        description="Informations sur un responsable légal",
        perm='can_access',
        name='read-rl',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'rl_id': {'description': 'Numéro du représentant légal'},
        },
    )
    def read_rl(self, request, NameID, rl_id):
        family_id = self.get_link(NameID).family_id
        data = self.get_family(family_id)
        if data['RL1']['num'] == rl_id:
            data = data['RL1']
        elif data['RL2'] and data['RL2']['num'] == rl_id:
            data = data['RL2']
        else:
            raise APIError("no '%s' RL on '%s' family" % (rl_id, family_id), err_code='not-found')
        return {'data': data}

    @endpoint(
        display_category='Famille',
        description="Informations sur une personne autorisée à récupérer les enfants ou à prévenir en cas d'urgence",
        perm='can_access',
        name='read-person',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'person_id': {'description': 'Numéro de la personne'},
            'kind': {'description': "'authorized' (par defaut) ou 'emergency'"},
        },
    )
    def read_person(self, request, NameID, person_id, kind='authorized'):
        if kind not in ('authorized', 'emergency'):
            raise APIError("wrong '%s' value for kind parameter" % kind)
        family_id = self.get_link(NameID).family_id
        data = self.get_family(family_id)
        for person in data[kind + 'PersonList']:
            if str(person['numPerson']) == person_id:
                break
        else:
            raise APIError(
                "no '%s' %s person on '%s' family" % (person_id, kind, family_id), err_code='not-found'
            )
        return {'data': person}

    @endpoint(
        display_category='Famille',
        description="Informations sur un enfant",
        perm='can_access',
        name='read-child',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'child_id': {'description': "Numéro de l'enfant"},
        },
    )
    def read_child(self, request, NameID, child_id):
        family_id = self.get_link(NameID).family_id
        data = self.get_family(family_id)
        for child in data['childList']:
            if child['num'] == child_id:
                break
        else:
            raise APIError("no '%s' child on '%s' family" % (child_id, family_id), err_code='not-found')
        return {'data': child}

    @endpoint(
        display_category='Famille',
        description="Vérifier qu'un responsable légal existe en base",
        perm='can_access',
        name='is-rl-exists',
        post={'request_body': {'schema': {'application/json': schemas.ISEXISTS_SCHEMA}}},
    )
    def is_rl_exists(self, request, post_data):
        response = self.call('Family', 'isRLExists', **post_data)
        return {'data': response}

    @endpoint(
        display_category='Famille',
        description="Vérifier qu'un responsable légal existe en base",
        perm='can_access',
        name='is-child-exists',
        post={'request_body': {'schema': {'application/json': schemas.ISEXISTS_SCHEMA}}},
    )
    def is_child_exists(self, request, post_data):
        response = self.call('Family', 'isChildExists', **post_data)
        return {'data': response}

    @endpoint(
        display_category='Famille',
        description='Création de la famille',
        name='create-family',
        perm='can_access',
        parameters={'NameID': {'description': 'Publik NameID'}},
        post={'request_body': {'schema': {'application/json': schemas.CREATE_FAMILY_SCHEMA}}},
    )
    def create_family(self, request, NameID, post_data):
        if self.link_set.filter(name_id=NameID).exists():
            raise APIError('User already linked to family', err_code='already-linked')

        response = self.call('Family', 'createFamily', **post_data)
        data = serialize_object(response)
        family_id = data.get('number')
        if not family_id:
            errors = data.get('rl1ErrorList') + data.get('childErrorList')
            err_codes = [x.split(':')[0][:4] for x in errors]
            raise APIError(' ; '.join(errors), err_code=', '.join(err_codes))

        Link.objects.create(resource=self, name_id=NameID, family_id=family_id)
        return {'data': data}

    @endpoint(
        display_category='Famille',
        description='Modification de la famille',
        name='update-family',
        perm='can_access',
        parameters={'NameID': {'description': 'Publik NameID'}},
        post={'request_body': {'schema': {'application/json': schemas.UPDATE_FAMILY_SCHEMA}}},
    )
    def update_family(self, request, NameID, post_data):
        family_id = self.get_link(NameID).family_id
        self.replace_null_values(post_data)

        response = self.call('Family', 'updateFamily', dossierNumber=family_id, **post_data)
        data = serialize_object(response)
        family_id = data.get('number')
        errors = data.get('childErrorList')
        if errors:
            err_codes = [x.split(':')[0][:4] for x in errors]
            raise APIError(' ; '.join(errors), err_code=', '.join(err_codes))
        return {'data': data}

    @endpoint(
        display_category='Famille',
        description='Création du RL1',
        name='create-rl1',
        perm='can_access',
        parameters={'NameID': {'description': 'Publik NameID'}},
        post={'request_body': {'schema': {'application/json': schemas.CREATE_RL1_SCHEMA}}},
    )
    def create_rl1(self, request, NameID, post_data):
        if self.link_set.filter(name_id=NameID).exists():
            raise APIError('User already linked to family', err_code='already-linked')

        response = self.call('Family', 'createFamily', **post_data)
        data = serialize_object(response)
        family_id = data.get('number')
        if not family_id:
            errors = data.get('rl1ErrorList') or []
            raise APIError(' ; '.join(errors), err_code='already-rl1')

        Link.objects.create(resource=self, name_id=NameID, family_id=family_id)
        return {'data': {'family_id': family_id}}

    @endpoint(
        display_category='Famille',
        description='Modification du RL1',
        name='update-rl1',
        perm='can_access',
        parameters={'NameID': {'description': 'Publik NameID'}},
        post={'request_body': {'schema': {'application/json': schemas.UPDATE_RL1_SCHEMA}}},
    )
    def update_rl1(self, request, NameID, post_data):
        family_id = self.get_link(NameID).family_id
        family = self.get_family_raw(family_id)
        self.replace_null_values(post_data)

        rl1 = post_data
        rl1['num'] = family['RL1']['num']
        rl1['adresse'] = family['RL1']['adresse']
        payload = {
            'dossierNumber': family_id,
            'categorie': family['category'],
            'situation': family['situation'],
            'flagCom': family['flagCom'],
            'nbChild': family['nbChild'],
            'nbTotalChild': family['nbTotalChild'],
            'nbAES': family['nbAES'],
            'rl1': rl1,
        }
        self.call('Family', 'updateFamily', **payload)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description='Création du RL2',
        name='create-rl2',
        perm='can_access',
        parameters={'NameID': {'description': 'Publik NameID'}},
        post={'request_body': {'schema': {'application/json': schemas.CREATE_RL2_SCHEMA}}},
    )
    def create_rl2(self, request, NameID, post_data):
        family_id = self.get_link(NameID).family_id
        family = self.get_family_raw(family_id)
        if family['RL2']:
            raise APIError('RL2 already defined on family', err_code='already-rl2')

        payload = {
            'dossierNumber': family_id,
            'categorie': family['category'],
            'situation': family['situation'],
            'flagCom': family['flagCom'],
            'nbChild': family['nbChild'],
            'nbTotalChild': family['nbTotalChild'],
            'nbAES': family['nbAES'],
            'rl2': post_data,
        }
        self.call('Family', 'updateFamily', **payload)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description='Modification du RL2',
        name='update-rl2',
        perm='can_access',
        parameters={'NameID': {'description': 'Publik NameID'}},
        post={'request_body': {'schema': {'application/json': schemas.UPDATE_RL2_SCHEMA}}},
    )
    def update_rl2(self, request, NameID, post_data):
        family_id = self.get_link(NameID).family_id
        family = self.get_family_raw(family_id)
        if not family['RL2']:
            raise APIError('No RL2 to update on family', err_code='no-rl2')
        self.replace_null_values(post_data)

        rl2 = post_data
        rl2['num'] = family['RL2']['num']
        rl2['adresse'] = family['RL2']['adresse']
        payload = {
            'dossierNumber': family_id,
            'categorie': family['category'],
            'situation': family['situation'],
            'flagCom': family['flagCom'],
            'nbChild': family['nbChild'],
            'nbTotalChild': family['nbTotalChild'],
            'nbAES': family['nbAES'],
            'rl2': rl2,
        }
        self.call('Family', 'updateFamily', **payload)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description="Ajout d'un enfant",
        name='create-child',
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'force': {
                'description': 'boolean to bypass doublon error',
                'type': 'bool',
                'example_value': 'false',
            },
        },
        post={'request_body': {'schema': {'application/json': schemas.CREATE_CHILD_SCHEMA}}},
    )
    def create_child(self, request, NameID, post_data, force=False):
        family_id = self.get_link(NameID).family_id

        payload = {
            'numDossier': family_id,
            'isForceCreateChild': force,
            'child': post_data,
        }
        response = self.call('Family', 'createChild', **payload)
        data = serialize_object(response)
        child_id = data.get('number')
        if not child_id:
            errors = data.get('childErrorList') or []
            raise APIError(' ; '.join(errors), err_code='already-child')
        return {'data': {'child_id': child_id}}

    @endpoint(
        display_category='Famille',
        description="Modification d'un enfant",
        name='update-child',
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'child_id': {'description': "Numéro de l'enfant"},
        },
        post={'request_body': {'schema': {'application/json': schemas.UPDATE_CHILD_SCHEMA}}},
    )
    def update_child(self, request, NameID, child_id, post_data):
        family_id = self.get_link(NameID).family_id
        family = self.get_family_raw(family_id)
        if child_id not in [x['num'] for x in family['childList']]:
            raise APIError('No child %s to update on family' % child_id, err_code='no-child')
        self.replace_null_values(post_data)

        child = post_data
        child['num'] = child_id
        payload = {
            'dossierNumber': family_id,
            'categorie': family['category'],
            'situation': family['situation'],
            'flagCom': family['flagCom'],
            'nbChild': family['nbChild'],
            'nbTotalChild': family['nbTotalChild'],
            'nbAES': family['nbAES'],
            'childList': [child],
        }
        self.call('Family', 'updateFamily', **payload)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description="Mise à jour des coordonnées d'une personne",
        name='update-coordinate',
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'rl_id': {'description': 'Numéro du représentant légal'},
        },
        post={'request_body': {'schema': {'application/json': schemas.UPDATE_COORDINATE_SCHEMA}}},
    )
    def update_coordinate(self, request, NameID, rl_id, post_data):
        family_id = self.get_link(NameID).family_id
        self.replace_null_values(post_data)

        self.call('Family', 'updateCoordinate', numDossier=family_id, numPerson=rl_id, **post_data)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description="Création d'une personne autorisée à récupérer les enfants ou à prévenir en cas d'urgence",
        name='create-person',
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'kind': {'description': "'authorized' (par defaut) ou 'emergency'"},
        },
        post={'request_body': {'schema': {'application/json': schemas.FAMILYPERSON_SCHEMA}}},
    )
    def create_person(self, request, NameID, post_data, kind='authorized'):
        if kind not in ('authorized', 'emergency'):
            raise APIError("wrong '%s' value for kind parameter" % kind)
        family_id = self.get_link(NameID).family_id
        family = self.get_family_raw(family_id)
        self.replace_null_values(post_data)

        personList = family[kind + 'PersonList']
        personList.append(post_data)
        payload = {
            'dossierNumber': family_id,
            'categorie': family['category'],
            'situation': family['situation'],
            'flagCom': family['flagCom'],
            'nbChild': family['nbChild'],
            'nbTotalChild': family['nbTotalChild'],
            'nbAES': family['nbAES'],
            kind + 'PersonList': [{'personList': personList}],
        }
        self.call('Family', 'updateFamily', **payload)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description="Mise à jour d'une personne autorisée à récupérer les enfants ou à prévenir en cas d'urgence",
        name='update-person',
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'person_id': {'description': 'Numéro de la personne'},
            'kind': {'description': "'authorized' (par defaut) ou 'emergency'"},
        },
        post={'request_body': {'schema': {'application/json': schemas.FAMILYPERSON_SCHEMA}}},
    )
    def update_person(self, request, NameID, person_id, post_data, kind='authorized'):
        if kind not in ('authorized', 'emergency'):
            raise APIError("wrong '%s' value for kind parameter" % kind)
        family_id = self.get_link(NameID).family_id
        family = self.get_family_raw(family_id)
        self.replace_null_values(post_data)

        personList = family[kind + 'PersonList']
        for i, person in enumerate(personList):
            if str(person['numPerson']) == person_id:
                personList[i] = post_data
                personList[i]['numPerson'] = person_id
                break
        else:
            raise APIError(
                "no '%s' authorized person on '%s' family" % (person_id, family_id), err_code='not-found'
            )
        payload = {
            'dossierNumber': family_id,
            'categorie': family['category'],
            'situation': family['situation'],
            'flagCom': family['flagCom'],
            'nbChild': family['nbChild'],
            'nbTotalChild': family['nbTotalChild'],
            'nbAES': family['nbAES'],
            kind + 'PersonList': [{'personList': personList}],
        }
        self.call('Family', 'updateFamily', **payload)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description="Suppression d'une personne autorisée à récupérer les enfants ou à prévenir en cas d'urgence",
        name='delete-person',
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'person_id': {'description': 'Numéro de la personne'},
            'kind': {'description': "'authorized' (par defaut) ou 'emergency'"},
        },
        methods=['post'],
    )
    def delete_person(self, request, NameID, person_id, kind='authorized'):
        if kind not in ('authorized', 'emergency'):
            raise APIError("wrong '%s' value for kind parameter" % kind)
        family_id = self.get_link(NameID).family_id
        family = self.get_family_raw(family_id)

        personList = family[kind + 'PersonList']
        for i, person in enumerate(personList):
            if str(person['numPerson']) == person_id:
                del personList[i]
                break
        else:
            raise APIError(
                "no '%s' authorized person on '%s' family" % (person_id, family_id), err_code='not-found'
            )
        payload = {
            'dossierNumber': family_id,
            'categorie': family['category'],
            'situation': family['situation'],
            'flagCom': family['flagCom'],
            'nbChild': family['nbChild'],
            'nbTotalChild': family['nbTotalChild'],
            'nbAES': family['nbAES'],
            kind + 'PersonList': [{'personList': personList}],
        }
        self.call('Family', 'updateFamily', **payload)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description="Créer ou mettre à jour le régime alimentaire d'un enfant",
        name='update-child-dietcode',
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'child_id': {'description': "Numéro de l'enfant"},
            'dietcode': {'description': 'code du régime alimentaire'},
        },
        methods=['post'],
    )
    def update_child_dietcode(self, request, NameID, child_id, dietcode):
        self.get_link(NameID)

        self.call('Family', 'createOrUpdateChildDiet', personNumber=child_id, code=dietcode)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description="Créer ou mettre à jour les informations relatives au PAI d'un enfant",
        name='update-child-pai',
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'child_id': {'description': "Numéro de l'enfant"},
        },
        post={'request_body': {'schema': {'application/json': schemas.PAIINFO_SCHEMA}}},
    )
    def update_child_pai(self, request, NameID, child_id, post_data):
        self.get_link(NameID)

        # use None to empty date passed as an empty string by date filter
        for key in 'dateDeb', 'dateFin':
            if post_data[key] == '':
                post_data[key] = None

        self.call('Family', 'updateChildPAI', personNumber=child_id, **post_data)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description="Créer ou mettre à jour la fiche sanitaire d'un enfant",
        name='update-child-fsl',
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'child_id': {'description': "Numéro de l'enfant"},
        },
        post={'request_body': {'schema': {'application/json': schemas.FSL_SCHEMA}}},
    )
    def update_child_fsl(self, request, NameID, child_id, post_data):
        self.get_link(NameID)

        # maelis expect strings O/N
        for key in (
            'allergieAlimentaire',
            'allergieRespiratoire',
            'allergieAutre',
            'allergieMedicament',
            'asthme',
            'flPAI',
            'flImage',
        ):
            post_data[key] = 'O' if post_data.get(key) else 'N'

        # use None to empty optional date passed as an empty string by date filter
        if post_data.get('dtcPrap1') == '':
            post_data['dtcPrap1'] = None

        self.call('Family', 'createOrUpdateFSL', arg0=child_id, arg1=post_data)
        return {'data': 'ok'}

    @endpoint(
        display_category='Famille',
        description="Créer ou mettre à jour les données médicales d'un enfant",
        name='update-child-medical-record',
        perm='can_access',
        parameters={
            'NameID': {'description': 'Publik NameID'},
            'child_id': {'description': "Numéro de l'enfant"},
        },
        post={'request_body': {'schema': {'application/json': schemas.MEDICALRECORD_SCHEMA}}},
    )
    def update_child_medical_record(self, request, NameID, child_id, post_data):
        self.get_link(NameID)
        self.replace_null_values(post_data)

        payload = {
            'numPerson': child_id,
            'medicalRecord': post_data,
        }
        self.call('Family', 'updateChildMedicalRecord', updateChildMedicalRecordRequest=payload)
        return {'data': 'ok'}


class Link(models.Model):
    resource = models.ForeignKey(ToulouseMaelis, on_delete=models.CASCADE)
    name_id = models.CharField(blank=False, max_length=256)
    family_id = models.CharField(blank=False, max_length=128)
    created = models.DateTimeField(auto_now_add=True)
    updated = models.DateTimeField(auto_now=True)

    class Meta:
        unique_together = ('resource', 'name_id')
