# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2019 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.views.generic import CreateView, DeleteView, UpdateView

from passerelle.base.mixins import ResourceChildViewMixin

from . import forms, models


class StayIfChanged(object):
    has_changed = False

    def form_valid(self, form):
        if set(form.changed_data) & set(['procedure', 'formdef']):
            self.has_changed = True
        return super(StayIfChanged, self).form_valid(form)

    def get_success_url(self):
        if self.has_changed:
            return self.get_changed_url()
        return super(StayIfChanged, self).get_success_url()

    def get_changed_url(self):
        return ''


class MappingNew(StayIfChanged, ResourceChildViewMixin, CreateView):
    model = models.Mapping
    form_class = forms.MappingForm

    def form_valid(self, form):
        form.instance.resource = self.resource
        return super(MappingNew, self).form_valid(form)

    def get_changed_url(self):
        return self.object.get_absolute_url()


class MappingEdit(StayIfChanged, ResourceChildViewMixin, UpdateView):
    model = models.Mapping
    form_class = forms.MappingForm


class MappingDelete(ResourceChildViewMixin, DeleteView):
    model = models.Mapping


def run(request, connector, slug):
    resource = get_object_or_404(models.Resource, slug=slug)
    resource.run_loop()
    return HttpResponseRedirect(resource.get_absolute_url())
