# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2019 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django import forms

from . import fields, models


class MappingForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if self.instance.procedure and self.instance and self.instance.formdef:
            choices = [('', '--------')] + [(v, v) for v in self.instance.variables]
            for i, field in enumerate(self.schema_fields()):
                label = field.label
                label += ' (%s)' % (field.varname or 'NO VARNAME')
                base_name = str(field.varname or i)
                initial = self.instance.rules.get('fields', {}).get(base_name)
                self.fields['field_%s' % base_name] = fields.VariableAndExpressionField(
                    label=label, choices=choices, initial=initial, required=False
                )

    def table_fields(self):
        return [field for field in self if field.name.startswith('field_')]

    def schema_fields(self):
        if self.instance and self.instance.formdef:
            schema = self.instance.formdef.schema
            for i, field in enumerate(schema.fields):
                if field.type in ('page', 'comment', 'title', 'subtitle'):
                    continue
                yield field

    def save(self, commit=True):
        fields = {}
        for key in self.cleaned_data:
            if not key.startswith('field_'):
                continue
            if not self.cleaned_data[key]:
                continue
            real_key = key[len('field_') :]
            value = self.cleaned_data[key].copy()
            value['label'] = self.fields[key].label
            fields[real_key] = value
        self.instance.rules['fields'] = fields
        return super().save(commit=commit)

    class Meta:
        model = models.Mapping
        fields = [
            'procedure',
            'formdef',
        ]
