# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2019 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django import forms
from django.core.exceptions import ValidationError
from django.template import TemplateSyntaxError, engines


def validate_django_template(value):
    try:
        engines['django'].from_string(value)
    except TemplateSyntaxError as e:
        raise ValidationError('invalid template %s' % e)


class VariableAndExpressionWidget(forms.MultiWidget):
    template_name = 'passerelle/widgets/variable_and_expression_widget.html'

    def __init__(self, **kwargs):
        widgets = [
            forms.Select,
            forms.TextInput,
        ]
        super(VariableAndExpressionWidget, self).__init__(widgets=widgets, **kwargs)

    def decompress(self, value):
        if not value:
            return [None, None]
        return value['variable'], value['expression']

    # XXX: bug in Django https://code.djangoproject.com/ticket/29205
    # required_attribute is initialized from the parent.field required
    # attribute and not from each sub-field attribute
    def use_required_attribute(self, initial):
        return False


class VariableAndExpressionField(forms.MultiValueField):
    widget = VariableAndExpressionWidget

    def __init__(
        self, choices=(), required=True, widget=None, label=None, initial=None, help_text='', *args, **kwargs
    ):
        fields = [
            forms.ChoiceField(choices=choices, required=required),
            forms.CharField(required=False, validators=[validate_django_template]),
        ]
        super(VariableAndExpressionField, self).__init__(
            fields=fields,
            required=required,
            widget=widget,
            label=label,
            initial=initial,
            help_text=help_text,
            require_all_fields=False,
            *args,
            **kwargs,
        )
        self.choices = choices

    def _get_choices(self):
        return self._choices

    def _set_choices(self, value):
        # Setting choices also sets the choices on the widget.
        # choices can be any iterable, but we call list() on it because
        # it will be consumed more than once.
        if callable(value):
            value = forms.CallableChoiceIterator(value)
        else:
            value = list(value)
        self._choices = value
        self.widget.widgets[0].choices = value

    choices = property(_get_choices, _set_choices)

    def compress(self, data):
        try:
            variable, expression = data
        except (ValueError, TypeError):
            return None
        else:
            return {
                'variable': variable,
                'expression': expression,
            }
