# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2021 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import datetime
import json
from uuid import uuid4

import lxml.etree as ET
from django.conf import settings
from django.contrib.postgres.fields import JSONField
from django.db import models
from django.db.transaction import atomic
from django.urls import reverse
from django.utils.six.moves.urllib import parse as urlparse
from django.utils.text import slugify
from django.utils.timezone import now
from django.utils.translation import ugettext_lazy as _
from requests import RequestException

from passerelle.base.models import BaseResource, HTTPResource, SkipJob
from passerelle.utils import xml
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError
from passerelle.utils.wcs import WcsApi, WcsApiError

from . import schemas


class ToulouseSmartResource(BaseResource, HTTPResource):
    category = _('Business Process Connectors')

    webservice_base_url = models.URLField(_('Webservice Base URL'))

    log_requests_errors = False

    class Meta:
        verbose_name = _('Toulouse Smart')

    def get_intervention_types(self):
        try:
            return self.get('intervention-types', max_age=120)
        except KeyError:
            pass

        try:
            url = self.webservice_base_url + 'v1/type-intervention'
            response = self.requests.get(url)
            doc = ET.fromstring(response.content)
            intervention_types = []
            for xml_item in doc:
                item = xml.to_json(xml_item)
                for prop in item.get('properties', []):
                    prop['required'] = prop.get('required') == 'true'
                    if prop.get('restrictedValues'):
                        prop['type'] = 'item'
                intervention_types.append(item)
            intervention_types.sort(key=lambda x: x['name'])
            for i, intervention_type in enumerate(intervention_types):
                intervention_type['order'] = i + 1
        except Exception:
            try:
                return self.get('intervention-types')
            except KeyError:
                raise
        self.set('intervention-types', intervention_types)
        return intervention_types

    def get(self, key, max_age=None):
        cache_entries = self.cache_entries
        if max_age:
            cache_entries = cache_entries.filter(timestamp__gt=now() - datetime.timedelta(seconds=max_age))
        try:
            return cache_entries.get(key=key).value
        except Cache.DoesNotExist:
            raise KeyError(key)

    def set(self, key, value):
        self.cache_entries.update_or_create(key=key, defaults={'value': value})

    @endpoint(
        name='type-intervention',
        description=_('Get intervention types'),
        perm='can_access',
    )
    def type_intervention(self, request):
        try:
            return {
                'data': [
                    {
                        'id': slugify(intervention_type['name']),
                        'text': intervention_type['name'],
                        'uuid': intervention_type['id'],
                    }
                    for intervention_type in self.get_intervention_types()
                ]
            }
        except Exception:
            return {
                'data': [
                    {
                        'id': '',
                        'text': _('Service is unavailable'),
                        'disabled': True,
                    }
                ]
            }

    def request(self, url, json=None, timeout=None):
        headers = {'Accept': 'application/json'}
        try:
            if json:
                headers['Content-Type'] = 'application/json'
                response = self.requests.post(url, headers=headers, timeout=timeout, json=json)
            else:
                response = self.requests.get(url, headers=headers, timeout=timeout)
            response.raise_for_status()
        except RequestException as e:
            raise APIError('failed to %s %s: %s' % ('post' if json else 'get', url, e))
        return response

    @endpoint(
        name='get-intervention',
        methods=['get'],
        description=_('Retrieve an intervention'),
        perm='can_access',
        parameters={
            'id': {'description': _('Intervention identifier')},
        },
    )
    def get_intervention(self, request, id):
        url = self.webservice_base_url + 'v1/intervention/%s' % id
        response = self.request(url)
        return {'data': response.json()}

    @atomic
    @endpoint(
        name='create-intervention',
        methods=['post'],
        description=_('Create an intervention'),
        perm='can_access',
        post={'request_body': {'schema': {'application/json': schemas.CREATE_SCHEMA}}},
    )
    def create_intervention(self, request, post_data):
        slug = post_data['slug']
        wcs_form_number = post_data['external_number']
        try:
            types = [x for x in self.get_intervention_types() if slugify(x['name']) == slug]
        except KeyError:
            raise APIError('Service is unavailable')
        if len(types) == 0:
            raise APIError("unknown '%s' block slug" % slug, http_status=400)
        intervention_type = types[0]
        wcs_block_varname = slugify(intervention_type['name']).replace('-', '_')
        try:
            block = post_data['fields']['%s_raw' % wcs_block_varname][0]
        except:
            raise APIError("cannot find '%s' block field content" % slug, http_status=400)
        data = {}
        cast = {'string': str, 'int': int, 'boolean': bool, 'item': str}
        for prop in intervention_type['properties']:
            name = prop['name'].lower()
            if block.get(name):
                try:
                    data[prop['name']] = cast[prop['type']](block[name])
                except ValueError:
                    raise APIError(
                        "cannot cast '%s' field to %s : '%s'" % (name, cast[prop['type']], block[name]),
                        http_status=400,
                    )
            elif prop['required']:
                raise APIError("'%s' field is required on '%s' block" % (name, slug), http_status=400)

        if self.wcs_requests.filter(wcs_form_api_url=post_data['form_api_url']):
            raise APIError(
                "'%s' intervention already created" % post_data['external_number'],
                http_status=400,
            )
        wcs_request = self.wcs_requests.create(
            wcs_form_api_url=post_data['form_api_url'],
            wcs_form_number=post_data['external_number'],
        )
        update_intervention_endpoint_url = request.build_absolute_uri(
            reverse(
                'generic-endpoint',
                kwargs={'connector': 'toulouse-smart', 'endpoint': 'update-intervention', 'slug': self.slug},
            )
        )
        wcs_request.payload = {
            'description': post_data['description'],
            'cityId': post_data['cityId'],
            'interventionCreated': post_data['interventionCreated'] + 'Z',
            'interventionDesired': post_data['interventionDesired'] + 'Z',
            'submitterFirstName': post_data['submitterFirstName'],
            'submitterLastName': post_data['submitterLastName'],
            'submitterMail': post_data['submitterMail'],
            'submitterPhone': post_data['submitterPhone'],
            'submitterAddress': post_data['submitterAddress'],
            'submitterType': post_data['submitterType'],
            'external_number': post_data['external_number'],
            'external_status': post_data['external_status'],
            'address': post_data['address'],
            'interventionData': data,
            'geom': {
                'type': 'Point',
                'coordinates': [post_data['lon'], post_data['lat']],
                'crs': 'EPSG:4326',
            },
            'interventionTypeId': intervention_type['id'],
            'notificationUrl': '%s?uuid=%s' % (update_intervention_endpoint_url, wcs_request.uuid),
        }
        wcs_request.save()
        if not wcs_request.push():
            self.add_job(
                'create_intervention_job',
                pk=wcs_request.pk,
                natural_id='wcs-request-%s' % wcs_request.pk,
            )
        return {
            'data': {
                'wcs_form_api_url': wcs_request.wcs_form_api_url,
                'wcs_form_number': wcs_request.wcs_form_number,
                'uuid': wcs_request.uuid,
                'payload': wcs_request.payload,
                'result': wcs_request.result,
                'status': wcs_request.status,
            }
        }

    def create_intervention_job(self, *args, **kwargs):
        wcs_request = self.wcs_requests.get(pk=kwargs['pk'])
        if not wcs_request.push():
            raise SkipJob()

    @atomic
    @endpoint(
        name='update-intervention',
        methods=['post'],
        description=_('Update an intervention status'),
        parameters={
            'id': {'description': _('Intervention identifier')},
        },
        post={'request_body': {'schema': {'application/json': schemas.UPDATE_SCHEMA}}},
    )
    def update_intervention(self, request, uuid, post_data):
        try:
            wcs_request = self.wcs_requests.get(uuid=uuid)
        except WcsRequest.DoesNotExist:
            raise APIError("Cannot find intervention '%s'" % uuid, http_status=400)
        smart_request = wcs_request.smart_requests.create(payload=post_data)
        self.add_job(
            'update_intervention_job',
            id=smart_request.id,
            natural_id='smart-request-%s' % smart_request.id,
        )
        return {
            'data': {
                'wcs_form_api_url': wcs_request.wcs_form_api_url,
                'wcs_form_number': wcs_request.wcs_form_number,
                'uuid': wcs_request.uuid,
                'payload': smart_request.payload,
            }
        }

    def update_intervention_job(self, *args, **kwargs):
        smart_request = SmartRequest.objects.get(id=kwargs['id'])
        if not smart_request.push():
            raise SkipJob()


class Cache(models.Model):
    resource = models.ForeignKey(
        verbose_name=_('Resource'),
        to=ToulouseSmartResource,
        on_delete=models.CASCADE,
        related_name='cache_entries',
    )

    key = models.CharField(_('Key'), max_length=64)

    timestamp = models.DateTimeField(_('Timestamp'), auto_now=True)

    value = JSONField(_('Value'), default=dict)


class WcsRequest(models.Model):
    resource = models.ForeignKey(
        to=ToulouseSmartResource,
        on_delete=models.CASCADE,
        related_name='wcs_requests',
    )
    wcs_form_api_url = models.CharField(max_length=256, primary_key=True)
    wcs_form_number = models.CharField(max_length=16)
    uuid = models.UUIDField(default=uuid4, unique=True, editable=False)
    payload = JSONField(null=True)
    result = JSONField(null=True)
    status = models.CharField(
        max_length=20,
        default='registered',
        choices=(
            ('registered', _('Registered')),
            ('sent', _('Sent')),
        ),
    )

    def push(self):
        url = self.resource.webservice_base_url + 'v1/intervention'
        try:
            response = self.resource.request(url, json=self.payload)
        except APIError as e:
            self.result = str(e)
            self.save()
            return False
        try:
            self.result = response.json()
        except ValueError:
            err_desc = 'invalid json, got: %s' % response.text
            self.result = err_desc
            self.save()
            return False
        self.status = 'sent'
        self.save()
        return True


class SmartRequest(models.Model):
    resource = models.ForeignKey(
        to=WcsRequest,
        on_delete=models.CASCADE,
        related_name='smart_requests',
    )
    payload = JSONField()
    result = JSONField(null=True)

    def get_wcs_api(self, base_url):
        scheme, netloc, path, params, query, fragment = urlparse.urlparse(base_url)
        services = settings.KNOWN_SERVICES.get('wcs', {})
        service = None
        for service in services.values():
            remote_url = service.get('url')
            r_scheme, r_netloc, r_path, r_params, r_query, r_fragment = urlparse.urlparse(remote_url)
            if r_scheme == scheme and r_netloc == netloc:
                break
        else:
            return None
        return WcsApi(base_url, orig=service.get('orig'), key=service.get('secret'))

    def push(self):
        headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
        }
        base_url = '%shooks/update-intervention/' % (self.resource.wcs_form_api_url)
        wcs_api = self.get_wcs_api(base_url)
        if not wcs_api:
            err_desc = 'Cannot find wcs service for %s' % base_url
            self.result = err_desc
            self.save()
            return True
        try:
            result = wcs_api.post_json(self.payload, [], headers=headers)
        except WcsApiError as e:
            try:
                result = json.loads(e.args[3])
            except (ValueError):
                return False
        self.result = result
        self.save()
        return True
