# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2021 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import datetime

import lxml.etree as ET
from django.contrib.postgres.fields import JSONField
from django.db import models
from django.utils.timezone import now
from django.utils.translation import ugettext_lazy as _

from passerelle.base.models import BaseResource, HTTPResource
from passerelle.utils import xml
from passerelle.utils.api import endpoint


class ToulouseSmartResource(BaseResource, HTTPResource):
    category = _('Business Process Connectors')

    webservice_base_url = models.URLField(_('Webservice Base URL'))

    log_requests_errors = False

    class Meta:
        verbose_name = _('Toulouse Smart')

    def get_intervention_types(self):
        try:
            return self.get('intervention-types', max_age=120)
        except KeyError:
            pass

        try:
            url = self.webservice_base_url + 'v1/type-intervention'
            response = self.requests.get(url)
            doc = ET.fromstring(response.content)
            intervention_types = []
            for xml_item in doc:
                item = xml.to_json(xml_item)
                for prop in item.get('properties', []):
                    prop['required'] = prop.get('required') == 'true'
                intervention_types.append(item)
            intervention_types.sort(key=lambda x: x['name'])
            for i, intervention_type in enumerate(intervention_types):
                intervention_type['order'] = i + 1
        except Exception:
            try:
                return self.get('intervention-types')
            except KeyError:
                raise
        self.set('intervention-types', intervention_types)
        return intervention_types

    def get(self, key, max_age=None):
        cache_entries = self.cache_entries
        if max_age:
            cache_entries = cache_entries.filter(timestamp__gt=now() - datetime.timedelta(seconds=max_age))
        try:
            return cache_entries.get(key='intervention-types').value
        except Cache.DoesNotExist:
            raise KeyError(key)

    def set(self, key, value):
        self.cache_entries.update_or_create(key=key, defaults={'value': value})

    @endpoint(
        name='type-intervention',
        description=_('Get intervention types'),
        perm='can_access',
    )
    def type_intervention(self, request):
        try:
            return {
                'data': [
                    {
                        'id': intervention_type['id'],
                        'text': intervention_type['name'],
                    }
                    for intervention_type in self.get_intervention_types()
                ]
            }
        except Exception:
            return {
                'data': [
                    {
                        'id': '',
                        'text': _('Service is unavailable'),
                        'disabled': True,
                    }
                ]
            }


class Cache(models.Model):
    resource = models.ForeignKey(
        verbose_name=_('Resource'),
        to=ToulouseSmartResource,
        on_delete=models.CASCADE,
        related_name='cache_entries',
    )

    key = models.CharField(_('Key'), max_length=64)

    timestamp = models.DateTimeField(_('Timestamp'), auto_now=True)

    value = JSONField(_('Value'), default=dict)
