# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2021  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import copy
import os

import xmlschema

from passerelle.contrib.utils import axel

BASE_XSD_PATH = os.path.join(os.path.dirname(__file__), 'xsd')


boolean_type = {
    'oneOf': [
        {'type': 'boolean'},
        {
            'type': 'string',
            'pattern': '[Oo]|[Nn]|[Tt][Rr][Uu][Ee]|[Ff][Aa][Ll][Ss][Ee]|1|0',
        },
    ]
}


def encode_bool(obj):
    if obj is True or str(obj).lower() in ['true', 'o', '1']:
        return 'O'
    if obj is False or str(obj).lower() in ['false', 'n', '0']:
        return 'N'
    return obj


class CaluireAxelSchema(axel.AxelSchema):
    type_map = {
        '{urn:AllAxelTypes}DATEREQUIREDType': 'date',
        '{urn:AllAxelTypes}DATEType': 'date_optional',
        '{urn:AllAxelTypes}ONType': 'bool',
        '{urn:AllAxelTypes}ONEmptyType': 'bool_optional',
    }

    @classmethod
    def schema_bool(cls):
        return copy.deepcopy(boolean_type)

    def encode_bool(self, obj):
        return encode_bool(obj)

    def decode_bool(self, data):
        value = False
        if data.text.lower() == 'o':
            value = True
        return xmlschema.ElementData(
            tag=data.tag, text=value, content=data.content, attributes=data.attributes
        )


class Operation(axel.Operation):
    base_xsd_path = BASE_XSD_PATH
    axel_schema = CaluireAxelSchema


find_individus = Operation('FindIndividus')
get_famille_individus = Operation('GetFamilleIndividus')


LINK_SCHEMA = copy.deepcopy(
    find_individus.request_schema['properties']['PORTAIL']['properties']['FINDINDIVIDU']
)
for key in ['NAISSANCE', 'CODEPOSTAL', 'VILLE', 'TEL', 'MAIL']:
    LINK_SCHEMA['properties'].pop(key)
    LINK_SCHEMA['required'].remove(key)
LINK_SCHEMA['properties']['IDENTFAMILLE'] = {'type': 'string', 'maxLength': 8}
LINK_SCHEMA['required'].append('IDENTFAMILLE')
