# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2021  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django.db import models
from django.utils.translation import ugettext_lazy as _

from passerelle.base.models import BaseResource
from passerelle.contrib.utils import axel
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError

from . import schemas


class CaluireAxel(BaseResource):

    wsdl_url = models.CharField(
        max_length=128, blank=False, verbose_name=_('WSDL URL'), help_text=_('Caluire Axel WSDL URL')
    )

    category = _('Business Process Connectors')
    _category_ordering = [_('Family account')]

    class Meta:
        verbose_name = _('Caluire Axel')

    def check_status(self):
        response = self.requests.get(self.wsdl_url)
        response.raise_for_status()

    def check_individu(self, post_data):
        family_id = post_data.pop('IDENTFAMILLE')
        for key in ['NAISSANCE', 'CODEPOSTAL', 'VILLE', 'TEL', 'MAIL']:
            post_data[key] = None
        try:
            result = schemas.find_individus(self, {'PORTAIL': {'FINDINDIVIDU': post_data}})
        except axel.AxelError as e:
            raise APIError(
                'Axel error: %s' % e,
                err_code='error',
                data={'xml_request': e.xml_request, 'xml_response': e.xml_response},
            )
        data = result.json_response['DATA']['PORTAIL']['FINDINDIVIDUS']
        for individu in data.get('INDIVIDU') or []:
            for famille in individu['FAMILLE']:
                if famille['IDENTFAMILLE'] == family_id:
                    place = famille['PLACE']
                    if place not in ['1', '2']:
                        # not RL1 or RL2
                        raise APIError('Wrong place in family', err_code='family-place-error-%s' % place)
                    return individu, result

        raise APIError('Person not found', err_code='not-found')

    @endpoint(
        display_category=_('Family account'),
        display_order=1,
        description=_('Create link between user and Caluire Axel'),
        perm='can_access',
        parameters={
            'NameID': {'description': _('Publik ID')},
        },
        post={
            'request_body': {
                'schema': {
                    'application/json': schemas.LINK_SCHEMA,
                }
            }
        },
    )
    def link(self, request, NameID, post_data):
        if not NameID:
            raise APIError('NameID is empty', err_code='bad-request', http_status=400)

        family_id = post_data['IDENTFAMILLE']
        try:
            data, result = self.check_individu(post_data)
        except APIError as e:
            if not hasattr(e, 'err_code') or e.err_code == 'error':
                raise
            raise APIError('Person not found', err_code='not-found')

        link, created = self.link_set.get_or_create(
            name_id=NameID, defaults={'family_id': family_id, 'person_id': data['IDENT']}
        )
        if not created and (link.family_id != family_id or link.person_id != data['IDENT']):
            raise APIError('Data conflict', err_code='conflict')
        return {
            'link': link.pk,
            'created': created,
            'family_id': link.family_id,
            'data': {
                'xml_request': result.xml_request,
                'xml_response': result.xml_response,
            },
        }

    def get_link(self, name_id):
        try:
            return self.link_set.get(name_id=name_id)
        except Link.DoesNotExist:
            raise APIError('Person not found', err_code='not-found')

    @endpoint(
        display_category=_('Family account'),
        display_order=2,
        description=_('Delete link between user and Caluire Axel'),
        methods=['post'],
        perm='can_access',
        parameters={
            'NameID': {'description': _('Publik ID')},
        },
    )
    def unlink(self, request, NameID):
        link = self.get_link(NameID)
        link_id = link.pk
        link.delete()
        return {'link': link_id, 'deleted': True, 'family_id': link.family_id}

    def get_family_data(self, family_id):
        try:
            result = schemas.get_famille_individus(
                self, {'PORTAIL': {'GETFAMILLE': {'IDENTFAMILLE': family_id}}}
            )
        except axel.AxelError as e:
            raise APIError(
                'Axel error: %s' % e,
                err_code='error',
                data={'xml_request': e.xml_request, 'xml_response': e.xml_response},
            )

        family_data = result.json_response['DATA']['PORTAIL']['GETFAMILLE']

        for child in family_data.get('MEMBRE', []):
            child['id'] = child['IDENT']
            child['text'] = '{} {}'.format(child['PRENOM'].strip(), child['NOM'].strip()).strip()

        return family_data

    @endpoint(
        display_category=_('Family account'),
        display_order=3,
        description=_("Get information about user's family"),
        perm='can_access',
        parameters={
            'NameID': {'description': _('Publik ID')},
        },
    )
    def family_info(self, request, NameID):
        link = self.get_link(NameID)
        family_data = self.get_family_data(link.family_id)
        return {'data': family_data}

    @endpoint(
        display_category=_('Family account'),
        display_order=4,
        description=_("Get information about children"),
        perm='can_access',
        parameters={
            'NameID': {'description': _('Publik ID')},
        },
    )
    def children_info(self, request, NameID):
        link = self.get_link(NameID)
        family_data = self.get_family_data(link.family_id)
        return {'data': family_data.get('MEMBRE', [])}

    @endpoint(
        display_category=_('Family account'),
        display_order=5,
        description=_("Get information about a child"),
        perm='can_access',
        parameters={
            'NameID': {'description': _('Publik ID')},
            'idpersonne': {'description': _('Child ID')},
        },
    )
    def child_info(self, request, idpersonne, NameID):
        link = self.get_link(NameID)
        family_data = self.get_family_data(link.family_id)

        for child in family_data.get('MEMBRE', []):
            if child['IDENT'] == idpersonne:
                return {'data': child}

        raise APIError('Child not found', err_code='not-found')


class Link(models.Model):
    resource = models.ForeignKey(CaluireAxel, on_delete=models.CASCADE)
    name_id = models.CharField(blank=False, max_length=256)
    family_id = models.CharField(blank=False, max_length=128)
    person_id = models.CharField(blank=False, max_length=128)

    class Meta:
        unique_together = ('resource', 'name_id')
