# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2019  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from requests import RequestException

from zeep import Client
from zeep.cache import InMemoryCache
from zeep.transports import Transport

from django.utils.six.moves.urllib import parse as urlparse

from passerelle.utils.jsonresponse import APIError


class SOAPError(APIError):
    pass


class SOAPClient(Client):
    """Wrapper around zeep.Client

    resource muste have a wsdl_url and a requests attribute
    """
    def __init__(self, resource, **kwargs):
        wsdl_url = kwargs.pop('wsdl_url', None) or resource.wsdl_url
        transport_class = getattr(resource, 'soap_transport_class', SOAPTransport)
        transport = transport_class(resource, wsdl_url, session=resource.requests, cache=InMemoryCache())
        super(SOAPClient, self).__init__(wsdl_url, transport=transport, **kwargs)


class SOAPTransport(Transport):
    """Wrapper around zeep.Transport

    disable basic_authentication hosts unrelated to wsdl's endpoints
    """
    def __init__(self, resource, wsdl_url, **kwargs):
        self.resource = resource
        self.wsdl_host = urlparse.urlparse(wsdl_url).netloc
        super(SOAPTransport, self).__init__(**kwargs)

    def _load_remote_data(self, url):
        try:
            if urlparse.urlparse(url).netloc != self.wsdl_host:
                response = self.session.get(url, timeout=self.load_timeout, auth=None, cert=None)
                response.raise_for_status()
                return response.content
            return super(SOAPTransport, self)._load_remote_data(url)
        except RequestException as e:
            raise SOAPError('SOAP service is down, location %r cannot be loaded: %s' % (url, e), exception=e, url=url)
