# -*- coding: utf-8 -*-
# Copyright (C) 2020 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


from collections import defaultdict

from urllib.parse import urljoin

import zeep
from zeep.wsse.username import UsernameToken
from zeep.helpers import serialize_object

from django.db import models
from django.utils import timezone

from django.utils.translation import ugettext_lazy as _

from passerelle.base.models import BaseResource
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError

from . import utils


LINK_SCHEMA = {
    "$schema": "http://json-schema.org/draft-04/schema#",
    "title": "Maelis",
    "description": "",
    "type": "object",
    "required": ["family_id", "password"],
    "properties": {
        "family_id": {
            "description": "family_id",
            "type": "string",
        },
        "password": {
            "description": "family password",
            "type": "string",
        },
        "school_year": {
            "description": "school year",
            "type": "string",
        }
    }
}

COORDINATES_SCHEMA = {
    "$schema": "http://json-schema.org/draft-04/schema#",
    "title": "Maelis",
    "description": "Person Coordinates",
    "type": "object",
    "properties": {
        "num": {
            "description": "number",
            "type": "number",
        },
        "street": {
            "description": "street",
            "type": "string",
        },
        "zipcode": {
            "description": "zipcode",
            "type": "string",
        },
        "town": {
            "description": "town",
            "type": "string",
        },
        "phone": {
            "description": "phone",
            "type": "string",
        },
        "mobile": {
            "description": "mobile",
            "type": "string",
        },
        "mail": {
            "description": "mail",
            "type": "string",
        }
    }
}


class Maelis(BaseResource):
    base_url = models.URLField(_('Base API URL'),
                               default='http://www3.sigec.fr/entrouvertws/services/')
    login = models.CharField(_('API Login'), max_length=256)
    password = models.CharField(_('API Password'), max_length=256)

    category = _('Business Process Connectors')

    class Meta:
        verbose_name = u'Maélis'

    @classmethod
    def get_verbose_name(cls):
        return cls._meta.verbose_name

    def check_status(self):
        response = self.requests.get(self.base_url)
        response.raise_for_status()

    def get_client(self, wsdl_name):
        wsse = UsernameToken(self.login, self.password)
        wsdl_url = urljoin(self.base_url, wsdl_name)
        return self.soap_client(wsdl_url=wsdl_url, wsse=wsse)

    def call(self, wsdl_name, service, **kwargs):
        client = self.get_client(wsdl_name)
        method = getattr(client.service, service)
        try:
            return method(**kwargs)
        except zeep.exceptions.Fault as e:
            raise APIError(e)

    def get_link(self, name_id):
        try:
            return self.link_set.get(name_id=name_id)
        except Link.DoesNotExist:
            raise APIError('User not linked to family', err_code='not-found')

    def get_family_data(self, family_id, school_year=None):
        if not school_year:
            # fallback to current year if not provided
            school_year = utils.get_default_school_year()
        family_data = serialize_object(self.call('FamilyService?wsdl',
                                                 'readFamily',
                                                 dossierNumber=family_id,
                                                 schoolYear=school_year))
        for child in family_data['childInfoList']:
            utils.normalize_person(child)
        return family_data

    def get_invoices(self, regie_id, name_id):
        family_id = self.get_link(name_id).family_id
        return [utils.normalize_invoice(i) for i in self.call(
            'InvoiceService?wsdl', 'readInvoices',
            numDossier=family_id, codeRegie=regie_id)]

    @endpoint(
        display_category=_('Family'),
        display_order=1,
        description=_('Create link between user and family'),
        perm='can_access',
        parameters={
            'NameID': {'description': _('Publik ID')},
        },
        post={
            'request_body': {
                'schema': {
                    'application/json': LINK_SCHEMA
                }
            }
        })
    def link(self, request, NameID, post_data):
        if 'school_year' not in post_data:
            # fallback to default year if not provided
            post_data['school_year'] = utils.get_default_school_year()
        r = self.call('FamilyService?wsdl', 'readFamilyByPassword',
                      dossierNumber=post_data['family_id'],
                      password=post_data['password'],
                      schoolYear=post_data['school_year']
                      )
        if not r.number:
            raise APIError('Family not found', err_code='not-found')
        Link.objects.update_or_create(resource=self, name_id=NameID,
                                      defaults={'family_id': r.number})
        return {'data': serialize_object(r)}

    @endpoint(
        display_category=_('Family'),
        display_order=2,
        description=_('Delete link between user and family'),
        methods=['post'],
        perm='can_access',
        parameters={
            'NameID': {'description': _('Publik ID')},
        })
    def unlink(self, request, NameID):
        link = self.get_link(NameID)
        link_id = link.pk
        link.delete()
        return {'link': link_id, 'deleted': True, 'family_id': link.family_id}

    @endpoint(
        display_category=_('Family'),
        display_order=4,
        description=_("Get information about user's family"),
        name='family-info',
        perm='can_access',
        parameters={
            'NameID': {'description': _('Publik ID')},
        })
    def family_info(self, request, NameID):
        link = self.get_link(NameID)
        family_data = self.get_family_data(link.family_id)
        return {'data': family_data}

    @endpoint(
        display_category=_('Family'),
        display_order=6,
        description=_("Get information about children"),
        perm='can_access',
        name='children-info',
        parameters={
            'NameID': {'description': _('Publik ID')},
        })
    def children_info(self, request, NameID):
        link = self.get_link(NameID)
        family_data = self.get_family_data(link.family_id)
        return {'data': family_data['childInfoList']}

    @endpoint(
        display_category=_('Family'),
        display_order=7,
        description=_("Get information about adults"),
        perm='can_access',
        name='adults-info',
        parameters={
            'NameID': {'description': _('Publik ID')},
        })
    def adults_info(self, request, NameID):
        link = self.get_link(NameID)
        family_data = self.get_family_data(link.family_id)
        adults = []
        if family_data.get('rl1InfoBean'):
            adults.append(utils.normalize_person(family_data['rl1InfoBean']))
        if family_data.get('rl2InfoBean'):
            adults.append(utils.normalize_person(family_data['rl2InfoBean']))
        return {'data': adults}

    @endpoint(
        display_category=_('Family'),
        display_order=7,
        description=_("Get information about a child"),
        perm='can_access',
        name='child-info',
        parameters={
            'NameID': {'description': _('Publik ID')},
            'childID': {'description': _('Child ID')},
        })
    def child_info(self, request, NameID, childID):
        link = self.get_link(NameID)
        family_data = self.get_family_data(link.family_id)

        for child in family_data.get('childInfoList', []):
            if child['num'] == childID:
                return {'data': child}

        raise APIError('Child not found', err_code='not-found')

    @endpoint(
        display_category=_('Family'),
        display_order=7,
        description=_('Update coordinates'),
        perm='can_access',
        name='update-coordinates',
        parameters={
            'NameID': {'description': _('Publik ID')},
            'personID': {'description': _('Person ID')},
        },
        post={
            'request_body': {
                'schema': {
                    'application/json': COORDINATES_SCHEMA
                }
            }
        })
    def update_coordinates(self, request, NameID, personID, post_data):
        link = self.get_link(NameID)
        params = defaultdict(dict)
        for address_param in ('num', 'zipcode', 'town'):
            if address_param in post_data:
                params['adresse'][address_param] = post_data[address_param]
        if 'street' in post_data:
            params['adresse']['street1'] = post_data['street']

        for contact_param in ('phone', 'mobile', 'mail'):
            if contact_param in post_data:
                params['contact'][contact_param] = post_data[contact_param]

        r = self.call('FamilyService?wsdl', 'updateCoordinate',
                      numDossier=link.family_id,
                      numPerson=personID,
                      **params)
        return serialize_object(r)

    @endpoint(
        display_category=_('Invoices'),
        display_order=1,
        name='regie',
        perm='can_access',
        pattern=r'^(?P<regie_id>[\w-]+)/invoices/?$',
        example_pattern='{regie_id}/invoices',
        description=_("Get invoices to pay"),
        parameters={
            'NameID': {'description': _('Publik ID')},
            'regie_id': {'description': _('Regie identifier'), 'example_value': '42-42'}
        })
    def invoices(self, request, regie_id, NameID):
        invoices = [i for i in self.get_invoices(
            regie_id=regie_id, name_id=NameID) if not i['paid']]
        return {'data': invoices}

    @endpoint(
        display_category=_('Invoices'),
        display_order=2,
        name='regie',
        perm='can_access',
        pattern=r'^(?P<regie_id>[\w-]+)/invoices/history/?$',
        example_pattern='{regie_id}/invoices/history',
        description=_("Get invoices already paid"),
        parameters={
            'NameID': {'description': _('Publik ID')},
            'regie_id': {'description': _('Regie identifier'), 'example_value': '42-42'}
        })
    def invoices_history(self, request, regie_id, NameID):
        invoices = [i for i in self.get_invoices(
            regie_id=regie_id, name_id=NameID) if i['paid']]
        return {'data': invoices}

    @endpoint(
        display_category=_('Invoices'),
        display_order=3,
        name='regie',
        perm='can_access',
        pattern=r'^(?P<regie_id>[\w-]+)/invoice/(?P<invoice_id>(historical-)?\w+-\d+)/?$',
        example_pattern='{regie_id}/invoice/{invoice_id}',
        description=_('Get invoice details'),
        parameters={
            'NameID': {'description': _('Publik ID')},
            'regie_id': {'description': _('Regie identifier'), 'example_value': '1'},
            'invoice_id': {'description': _('Invoice identifier'), 'example_value': '42-42'}
        })
    def invoice(self, request, regie_id, invoice_id, NameID):
        for invoice in self.get_invoices(regie_id=regie_id, name_id=NameID):
            if invoice['id'] == invoice_id:
                return {'data': invoice}

    @endpoint(
        display_category=_('Invoices'),
        display_order=4,
        name='regie',
        perm='can_access',
        pattern=r'^(?P<regie_id>[\w-]+)/invoice/(?P<invoice_id>(historical-)?\w+-\d+)/pdf/?$',
        example_pattern='{regie_id}/invoice/{invoice_id}/pdf',
        description=_('Get invoice as a PDF file'),
        parameters={
            'NameID': {'description': _('Publik ID')},
            'regie_id': {'description': _('Regie identifier'), 'example_value': '1'},
            'invoice_id': {'description': _('Invoice identifier'), 'example_value': '42-42'}
        })
    def invoice_pdf(self, request, regie_id, invoice_id, **kwargs):
        # TODO to implement
        pass

    @endpoint(
        perm='can_access',
        description=_('Get activity list'),
        name='activity-list',
        parameters={
            'NameID': {'description': _('Publik ID')},
            'personID': {'description': _('Person ID')},
            'school_year': {'description': _('School year')},
        })
    def activity_list(self, request, NameID, personID, school_year=None, start_datetime=None,
                      end_datetime=None):
        link = self.get_link(NameID)
        family_data = self.get_family_data(link.family_id)
        if personID not in [c['id'] for c in family_data['childInfoList']]:
            raise APIError('Child not found', err_code='not-found')
        if not school_year:
            school_year = utils.get_default_school_year()
        if not start_datetime:
            start_datetime = timezone.now()
        if not end_datetime:
            end_datetime = start_datetime + timezone.timedelta(days=62)
        r = self.call('ActivityService?wsdl', 'readActivityList',
                      schoolyear=school_year, numPerson=personID,
                      dateStartCalend=start_datetime,
                      dateEndCalend=end_datetime)
        activities = serialize_object(r)
        return {'data': [utils.normalize_activity(a) for a in activities]}


class Link(models.Model):
    resource = models.ForeignKey(Maelis, on_delete=models.CASCADE)
    name_id = models.CharField(blank=False, max_length=256)
    family_id = models.CharField(blank=False, max_length=128)
    created = models.DateTimeField(auto_now_add=True)
    updated = models.DateTimeField(auto_now=True)

    class Meta:
        unique_together = ('resource', 'name_id')
