# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2020  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
import logging
import re

from django.db import models
from django.utils import six
from django.utils.module_loading import import_string
from django.utils.translation import ugettext_lazy as _

from passerelle.base.models import BaseResource
from passerelle.compat import json_loads
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError

SEND_SCHEMA = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    "type": "object",
    'required': ['message', 'from', 'to'],
    'properties': {
        'message': {
            'description': 'String message',
            'type': 'string',
        },
        'from': {
            'description': 'Sender number',
            'type': 'string',
        },
        'to': {
            'description': 'Destination numbers',
            "type": "array",
            "items": {
                'type': 'string',
                'pattern': r'^\+?\d+$'
            },
        },
    }
}


class SMSResource(BaseResource):
    category = _('SMS Providers')
    documentation_url = 'https://doc-publik.entrouvert.com/admin-fonctionnel/les-tutos/configuration-envoi-sms/'

    _can_send_messages_description = _('Sending messages is limited to the following API users:')

    default_country_code = models.CharField(verbose_name=_('Default country code'), max_length=3,
                                            default=u'33')
    default_trunk_prefix = models.CharField(verbose_name=_('Default trunk prefix'), max_length=2,
                                            default=u'0')  # Yeah France first !
    # FIXME: add regexp field, to check destination and from format
    max_message_length = models.IntegerField(_('Maximum message length'), default=160)

    manager_view_template_name = 'passerelle/manage/messages_service_view.html'

    @classmethod
    def get_management_urls(cls):
        return import_string('passerelle.sms.urls.management_urlpatterns')

    def clean_numbers(self, destinations):
        numbers = []
        for dest in destinations:
            # most gateways needs the number prefixed by the country code, this is
            # really unfortunate.
            dest = dest.strip()
            number = ''.join(re.findall('[0-9]', dest))
            if dest.startswith('+'):
                number = '00' + number
            elif number.startswith('00'):
                # assumes 00 is international access code, remove it
                pass
            elif number.startswith(self.default_trunk_prefix):
                number = '00' + self.default_country_code + number[len(self.default_trunk_prefix):]
            else:
                raise APIError('phone number %r is unsupported (no international prefix, '
                               'no local trunk prefix)' % number)
            numbers.append(number)
        return numbers

    @endpoint(perm='can_send_messages', methods=['post'],
              description=_('Send a SMS message'),
              parameters={'nostop': {'description': _('Do not send STOP instruction'), 'example_value': '1'}},
              post={'request_body': {'schema': {'application/json': SEND_SCHEMA}}})
    def send(self, request, post_data, nostop=None):
        post_data['message'] = post_data['message'][:self.max_message_length]
        post_data['to'] = self.clean_numbers(post_data['to'])
        logging.info('sending SMS to %r from %r', post_data['to'], post_data['from'])
        stop = nostop is None  # ?nostop in not in query string
        self.add_job('send_job',
                     text=post_data['message'], sender=post_data['from'], destinations=post_data['to'],
                     stop=stop)
        return {'err': 0}

    def send_job(self, *args, **kwargs):
        self.send_msg(**kwargs)
        SMSLog.objects.create(appname=self.get_connector_slug(), slug=self.slug)

    class Meta:
        abstract = True


@six.python_2_unicode_compatible
class SMSLog(models.Model):
    timestamp = models.DateTimeField(auto_now_add=True)
    appname = models.CharField(max_length=128, verbose_name='appname', null=True)
    slug = models.CharField(max_length=128, verbose_name='slug', null=True)

    def __str__(self):
        return '%s %s %s' % (self.timestamp, self.appname, self.slug)
