# -*- coding: utf-8 -*-
# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2019 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from urllib.parse import urljoin

import requests

from django.db import models
from django.utils.translation import ugettext_lazy as _

from passerelle.base.models import BaseResource, HTTPResource
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError

OUI_NON_ENUM = {'enum': ['Oui', 'Non']}
DATE_SCHEMA = {'type': 'string', 'format': 'date'}


class RSA13Resource(BaseResource, HTTPResource):
    category = _('Business Process Connectors')

    webservice_base_url = models.URLField(_('Webservice Base URL'))

    log_requests_errors = False

    class Meta:
        verbose_name = _('RSA CD13')

    def request_raw(self, method, path, **kwargs):
        email = kwargs.pop('email', None)
        ip = kwargs.pop('ip', None)
        headers = kwargs.setdefault('headers', {})
        if email:
            headers['X-CD13-Email'] = email
        if ip:
            headers['X-CD13-IP'] = ip
        full_path = urljoin(self.webservice_base_url, '/api/') + path
        try:
            response = self.requests.request(method, full_path, **kwargs)
            try:
                response.json()
            except ValueError:
                response.raise_for_status()
                raise requests.RequestException('JSON expected', response=response)
        except requests.RequestException as e:
            raise APIError('Server is down: %s' % e)
        return response

    def request(self, method, path, **kwargs):
        response = self.request_raw(method, path, **kwargs)
        content = response.json()
        if content.get('err') != 0:
            err_desc = content.get('err_code') or 'misc-error'
            raise APIError(err_desc, data=content)
        response.raise_for_status()
        return content

    def get(self, path, **kwargs):
        return self.request('get', path, **kwargs)

    def post(self, path, **kwargs):
        return self.request('post', path, **kwargs)

    def check_status(self):
        response = self.request_raw('GET', 'check')
        if response.json().get('ping') != 'pong':
            raise APIError('ping/pong expected received: "%s"' % repr(response)[:1024])

    def parameters(update=()):
        d = {
            'email': {
                'description': _('Publik known email'),
                'example_value': 'john.doe@example.com',
            },
            'ip': {
                'description': _('Publik client IP'),
                'example_value': '88.67.23.45',
            },
        }
        d.update(update)
        return d

    def response_schema(schema):
        return {
            'type': 'object',
            'required': ['err'],
            'properties': {
                'err': {'enum': [0, 1]},
                'data': schema,
            },
        }

    @endpoint(
        description=_('Get nomenclature'),
        long_description=_('Domain can be: MOTICLODAC, MOTIF_FIN_ACC, RESULTAT_RDV, RELANCE_RDV'),
        perm='can_access',
        pattern=r'^(?P<domain>[A-Z_]{1,30})/$',
        example_pattern='{domain}/',
        parameters=parameters(
            {
                'domain': {
                    'description': _('Nomenclature domain'),
                    'example_value': 'MOTICLODAC',
                }
            }
        ),
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'required': ['id', 'text'],
                    'properties': {
                        'id': {
                            'enum': [0, 1],
                        },
                        'text': {
                            'type': 'string',
                        },
                    },
                },
            }
        ),
    )
    def nomenclature(self, request, domain, email, ip=None):
        return self.get('cg_ref_code/domain/%s/' % domain, email=email, ip=ip)

    @endpoint(
        description=_('List of platforms'),
        perm='can_access',
        parameters=parameters(),
        display_category=_('Platform'),
        display_order=1,
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'required': ['id', 'dsp', 'name', 'role'],
                    'properties': {
                        'id': {
                            'type': 'integer',
                        },
                        'name': {
                            'type': 'string',
                        },
                        'dsp': {
                            'type': 'string',
                        },
                        'role': {
                            'enum': ['Coordonnateur', 'Accompagnateur'],
                        },
                    },
                },
            }
        ),
    )
    def platform(self, request, email, ip=None):
        return self.get('platform/', email=email, ip=ip)

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/$',
        example_pattern='{platform_id}/',
        description=_('Platform details'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                }
            }
        ),
        display_category=_('Platform'),
        display_order=2,
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'required': [
                        'id',
                        'dsp',
                        'name',
                        'adr1',
                        'adr2',
                        'adr3',
                        'adr4',
                        'adr5',
                        'adr6',
                        'tel',
                        'queries',
                    ],
                    'properties': {
                        'id': {
                            'type': 'integer',
                        },
                        'name': {
                            'type': 'string',
                        },
                        'dsp': {
                            'type': 'string',
                        },
                        'adr1': {
                            'type': 'string',
                        },
                        'adr2': {
                            'type': 'string',
                        },
                        'adr3': {
                            'type': 'string',
                        },
                        'adr4': {
                            'type': 'string',
                        },
                        'adr5': {
                            'type': 'string',
                        },
                        'adr6': {
                            'type': 'string',
                        },
                        'tel': {
                            'type': 'string',
                        },
                        'queries': {
                            'type': 'array',
                            'items': {
                                'type': 'object',
                                'required': ['id', 'name', 'count'],
                                'properties': {
                                    'id': {
                                        'type': 'integer',
                                    },
                                    'name': {
                                        'type': 'string',
                                    },
                                    'count': {
                                        'type': 'integer',
                                    },
                                },
                            },
                        },
                    },
                },
            }
        ),
    )
    def platform_details(self, request, platform_id, email, ip=None):
        return self.get('platform/%s/' % platform_id, email=email, ip=ip)

    @endpoint(
        name='platform',
        methods=['get', 'post'],
        pattern=r'^(?P<platform_id>[0-9]{1,10})/referent/$',
        example_pattern='{platform_id}/',
        description_get=_('Get platform referents'),
        description_post=_('Create platform referent'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                }
            }
        ),
        display_category=_('Platform'),
        display_order=3,
        post={
            'request_body': {
                'schema': {
                    'application/json': {
                        'type': 'object',
                        'required': ['email', 'nom', 'prenom', 'tel'],
                        'properties': {
                            'email': {
                                'type': 'string',
                                'maxLength': 78,
                                'pattern': '^.*@.*$',
                            },
                            'nom': {
                                'type': 'string',
                                'maxLength': 28,
                            },
                            'prenom': {
                                'type': 'string',
                                'maxLength': 32,
                            },
                            'tel': {
                                'type': 'string',
                                'maxLength': 10,
                                'pattern': '^[0-9]{0,10}$',
                            },
                        },
                    },
                },
            }
        },
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'required': [
                        'id',
                        'nom',
                        'prenom',
                        'tel',
                        'email',
                        'role',
                        'status',
                    ],
                    'properties': {
                        'id': {
                            'type': 'integer',
                        },
                        'nom': {
                            'type': 'string',
                        },
                        'prenom': {
                            'type': 'string',
                        },
                        'tel': {
                            'type': 'string',
                        },
                        'email': {
                            'type': 'string',
                        },
                        'role': {
                            'enum': ['Coordonnateur', 'Accompagnateur'],
                        },
                        'statut': {
                            'enum': ['Actif', 'Inactif'],
                        },
                    },
                },
            }
        ),
    )
    def platform_referent(self, request, platform_id, email, ip=None, post_data=None):
        if request.method == 'GET':
            return self.get('platform/%s/referent/' % platform_id, email=email, ip=ip)
        else:
            return self.post('platform/%s/referent/' % platform_id, email=email, ip=ip, json=post_data)
    # BUG, methods and post are incompatible
    platform_referent.endpoint_info.methods.append('get')

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/beneficiaire/$',
        example_pattern='{platform_id}/',
        description_get=_('Get platform beneficiaries'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'page': {
                    'description': _('Page number'),
                    'example_value': '1',
                },
                'query': {
                    'description': _('Query numeric identifier'),
                    'example_value': '2',
                },
                'nom': {
                    'description': _('Beneficiary last name'),
                    'example_value': 'Doe',
                },
                'prenom': {
                    'description': _('Beneficiary last name'),
                    'example_value': 'John',
                },
                'matricule': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '1234',
                },
                'referent': {
                    'description': _('Referent numeric identifier'),
                    'example_value': '5678',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=4,
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'required': [
                        'id',
                        'civilite',
                        'nom',
                        'prenom',
                        'date_naissance',
                        'actif',
                        'matricule',
                        'commune',
                        'code_pi',
                        'referent',
                        'date_deb_affectation',
                        'consulte',
                        'toppersdrodevorsa',
                    ],
                    'properties': {
                        'id': {
                            'type': 'integer',
                        },
                        'civilite': {
                            'enum': ['MR', 'MME'],
                        },
                        'nom': {
                            'type': 'string',
                        },
                        'prenom': {
                            'type': 'string',
                        },
                        'date_naissance': {
                            'type': 'string',
                            'format': 'date',
                        },
                        'actif': OUI_NON_ENUM,
                        'matricule': {
                            'type': 'string',
                        },
                        'code_postal': {
                            'type': 'string',
                        },
                        'commune': {
                            'type': 'string',
                        },
                        'communcode_pi': {
                            'type': 'string',
                        },
                        'referent': {
                            'type': 'string',
                        },
                        'date_deb_affectation': DATE_SCHEMA,
                        'consulte': OUI_NON_ENUM,
                        'toppersdrodevorsa': OUI_NON_ENUM,
                    },
                },
            }
        ),
    )
    def platform_beneficiaire(
        self,
        request,
        platform_id,
        email,
        ip=None,
        page=None,
        query=None,
        nom=None,
        prenom=None,
        matricule=None,
        referent=None,
    ):
        params = {}
        for key in ['page', 'query', 'nom', 'prenom', 'matricule', 'referent']:
            if key in locals():
                params[key] = locals()[key]
        return self.get('platform/%s/beneficiaire/' % platform_id, email=email, ip=ip, params=params)

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/',
        description=_('Get beneficiary details'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=5,
        json_schema_response=response_schema(
            {
                'type': 'object',
                'required': [
                    'id',
                    'civilite',
                    'nom',
                    'prenom',
                    'date_naissance',
                    'actif',
                    'matricule',
                    'code_pi',
                    'referent',
                    'adresse',
                    'age',
                    'commentaire_ref',
                    'conjoint',
                    'droit',
                    'enfants',
                    'lib_code_pi',
                    'nomnaiss',
                    'numdemrsa',
                    'situation_familiale',
                ],
                'properties': {
                    'id': {'type': 'integer'},
                    'civilite': {'enum': ['MR', 'MME']},
                    'nom': {'type': 'string'},
                    'prenom': {'type': 'string'},
                    'date_naissance': DATE_SCHEMA,
                    'actif': OUI_NON_ENUM,
                    'matricule': {'type': 'string'},
                    'code_pi': {'type': 'string'},
                    'referent': {
                        'type': 'object',
                        'required': ['id', 'nom', 'prenom'],
                        'properties': {
                            'id': {'type': 'integer'},
                            'nom': {'type': 'string'},
                            'prenom': {'type': 'string'},
                        },
                    },
                    'adresse': {
                        'type': 'object',
                        'required': ['adr2', 'adr3', 'adr4', 'adr5', 'adr6'],
                        'properties': {
                            'adr2': {'type': 'string'},
                            'adr3': {'type': 'string'},
                            'adr4': {'type': 'string'},
                            'adr5': {'type': 'string'},
                            'adr6': {'type': 'string'},
                        },
                    },
                    'age': {'type': 'string'},
                    'commentaire_ref': {'type': 'string'},
                    'conjoint': {
                        'type': 'object',
                        'required': ['id', 'age', 'prenom', 'nom', 'plateforme'],
                        'properties': {
                            'id': {'type': 'integer'},
                            'age': {'type': 'string'},
                            'nom': {'type': 'string'},
                            'prenom': {'type': 'string'},
                            'plateforme': {'type': 'string'},
                        },
                    },
                    'droit': {
                        'type': 'object',
                        'required': ['date_demande', 'etat', 'motif', 'toppersdrodevorsa'],
                        'properties': {
                            'date_demande': {'type': 'string', 'format': 'date'},
                            'etat': {'type': 'string'},
                            'motif': {'type': 'string'},
                            'toppersdrodevorsa': OUI_NON_ENUM,
                        },
                    },
                    'enfants': {
                        'type': 'array',
                        'items': {
                            'type': 'object',
                            'required': ['nom', 'prenom', 'age'],
                            'properties': {
                                'age': {'type': 'string'},
                                'nom': {'type': 'string'},
                                'prenom': {'type': 'string'},
                            },
                        },
                    },
                    'lib_code_pi': {'type': 'string'},
                    'nomnaiss': {'type': 'string'},
                    'numdemrsa': {'type': 'string'},
                    'situation_familiale': {
                        'type': 'object',
                        'required': ['date_debut', 'libelle'],
                        'properties': {
                            'date_debut': {'type': 'string', 'format': 'date'},
                            'libelle': {'type': 'string'},
                        },
                    },
                },
            }
        ),
    )
    def platform_beneficiaire_detail(self, request, platform_id, beneficiary_id, email, ip=None):
        return self.get('platform/%s/beneficiaire/%s/' % (platform_id, beneficiary_id), email=email, ip=ip)

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/transport/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/transport/',
        description=_('Get beneficiary transport details'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=6,
        json_schema_response=response_schema(
            {
                'type': 'object',
                'required': ['cumuls'],
                'properties': {
                    'cumuls': {
                        'type': 'array',
                        'items': {
                            'type': 'object',
                            'required': ['duree', 'type'],
                            'properties': {
                                'duree': {'type': 'integer'},
                                'type': {'type': 'string'},
                            },
                        },
                    }
                },
            }
        ),
    )
    def platform_beneficiaire_transport(self, request, platform_id, beneficiary_id, email, ip=None):
        return self.get(
            'platform/%s/beneficiaire/%s/transport/' % (platform_id, beneficiary_id), email=email, ip=ip
        )

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/contrat/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/contrat/',
        description=_('Get beneficiary contracts'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=7,
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'required': [
                        'id',
                        'clos',
                        'date_deb',
                        'date_fin',
                        'decision',
                        'duree',
                        'operateur',
                        'plateform',
                        'referent',
                        'retab',
                    ],
                    'properties': {
                        'id': {'type': 'integer'},
                        'clos': OUI_NON_ENUM,
                        'date_deb': DATE_SCHEMA,
                        'date_fin': DATE_SCHEMA,
                        'decision': {'type': 'string'},
                        'duree': {'type': 'integer'},
                        'operateur': {'type': 'string'},
                        'plateforme': {'type': 'string'},
                        'retab': OUI_NON_ENUM,
                    },
                },
            }
        ),
    )
    def platform_beneficiaire_contrat(self, request, platform_id, beneficiary_id, email, ip=None):
        return self.get(
            'platform/%s/beneficiaire/%s/contrat/' % (platform_id, beneficiary_id), email=email, ip=ip
        )

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/'
        r'beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/'
        r'contrat/(?P<contract_id>[0-9]{1,10})/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/contrat/{contract_id}/',
        description=_('Get beneficiary contract details'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
                'contract_id': {
                    'description': _('Contract numeric identifier'),
                    'example_value': '7',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=8,
        json_schema_response=response_schema(
            {
                'type': 'object',
                'required': [
                    'id',
                    'commentaire',
                    'date_clos',
                    'date_cvs',
                    'date_deb',
                    'date_fin',
                    'date_retab',
                    'date_sign',
                    'decision',
                    'duree',
                    'motif_cvs',
                    'operateur',
                    'plateforme',
                    'referent',
                    'type_contrat',
                ],
                'properties': {
                    'id': {'type': 'integer'},
                    'commentaire': {'type': 'string'},
                    'date_clos': DATE_SCHEMA,
                    'date_cvs': DATE_SCHEMA,
                    'date_deb': DATE_SCHEMA,
                    'date_fin': DATE_SCHEMA,
                    'date_retab': DATE_SCHEMA,
                    'date_sign': DATE_SCHEMA,
                    'decision': {'type': 'string'},
                    'duree': {'type': 'integer'},
                    'motif_cvs': {'type': 'string'},
                    'operateur': {'type': 'string'},
                    'plateforme': {'type': 'string'},
                    'referent': {
                        'type': 'object',
                        'required': ['commentaire', 'nom', 'prenom'],
                        'properties': {
                            'commentaire': {'type': 'string'},
                            'nom': {'type': 'string'},
                            'prenom': {'type': 'string'},
                        },
                    },
                    'type_contrat': {'type': 'string'},
                },
            }
        ),
    )
    def platform_beneficiaire_contrat_detail(
        self, request, platform_id, beneficiary_id, contract_id, email, ip=None
    ):
        return self.get(
            'platform/%s/beneficiaire/%s/contrat/%s/' % (platform_id, beneficiary_id, contract_id),
            email=email,
            ip=ip,
        )

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/'
        r'beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/'
        r'action/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/action/',
        description=_('Get beneficiary actions'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=9,
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'required': [
                        'id',
                        'contrat_id',
                    ],
                    'properties': {
                        'id': {'type': 'integer'},
                        'contrat_id': {'type': 'integer'},
                        'libelle': {'type': 'string'},
                        'date_preconisation': DATE_SCHEMA,
                        'date_deb': DATE_SCHEMA,
                        'date_fin': DATE_SCHEMA,
                        'validation': {
                            'enum': ['En cours', 'Oui', 'Non'],
                        },
                        'clos': OUI_NON_ENUM,
                    },
                }
            }
        ),
    )
    def platform_beneficiaire_action(
        self, request, platform_id, beneficiary_id, email, ip=None
    ):
        return self.get(
            'platform/%s/beneficiaire/%s/action/' % (platform_id, beneficiary_id),
            email=email,
            ip=ip,
        )

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/'
        r'beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/'
        r'action/(?P<action_id>[0-9]{1,10})/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/contrat/{action_id}/',
        description=_('Get beneficiary action details'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
                'action_id': {
                    'description': _('Action numeric identifier'),
                    'example_value': '7',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=10,
        json_schema_response=response_schema(
            {
                'type': 'object',
                'required': [
                    'id',
                    'contrat_id',
                ],
                'properties': {
                    'id': {'type': 'integer'},
                    'contrat_id': {'type': 'integer'},
                    'sac': {'type': 'string'},
                    'libelle': {'type': 'string'},
                    'date_preconisation': DATE_SCHEMA,
                    'date_deb': DATE_SCHEMA,
                    'date_fin': DATE_SCHEMA,
                    'date_cloture': DATE_SCHEMA,
                    'moticlodac': {'type': 'string'},
                    'lib_moticlodac': {'type': 'string'},
                    'validation': {
                        'enum': ['En cours', 'Oui', 'Non'],
                    },
                    'financement': {
                        'properties': {
                            'montant_demande': {'type': 'integer'},
                            'montant_accorde': {'type': 'integer'},
                        }
                    },
                    'commentaire_ref': {'type': 'string'},
                },
            }
        ),
    )
    def platform_beneficiaire_action_detail(
        self, request, platform_id, beneficiary_id, action_id, email, ip=None
    ):
        return self.get(
            'platform/%s/beneficiaire/%s/action/%s/' % (platform_id, beneficiary_id, action_id),
            email=email,
            ip=ip,
        )

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/'
        r'beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/'
        r'fondsaide/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/fondsaide/',
        description=_('Get beneficiary help funds'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=11,
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'properties': {
                        'id': {'type': 'integer'},
                        'cod_tfi': {'type': 'string'},
                        'lib_tfi': {'type': 'string'},
                        'demande': {
                            'type': 'object',
                            'properties': {
                                'montant': {'type': 'number'},
                                'date': DATE_SCHEMA,
                            }
                        },
                        'avis_pi': {
                            'type': 'object',
                            'properties': {
                                'montant': {'type': 'number'},
                                'date': DATE_SCHEMA,
                                'avis': {'type': 'string'},
                            }
                        },
                        'avis_sai': {
                            'type': 'object',
                            'properties': {
                                'montant': {'type': 'number'},
                                'date': DATE_SCHEMA,
                            }
                        },
                        'clos': OUI_NON_ENUM,
                    },
                },
            }
        ),
    )
    def platform_beneficiaire_fondsaide(
        self, request, platform_id, beneficiary_id, email, ip=None
    ):
        return self.get(
            'platform/%s/beneficiaire/%s/fondsaide/' % (platform_id, beneficiary_id),
            email=email,
            ip=ip,
        )

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/'
        r'beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/'
        r'fondsaide/(?P<fondsaide_id>[0-9]{1,10})/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/fondsaide/{fondsaide_id}/',
        description=_('Get beneficiary help fund details'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
                'fondsaide_id': {
                    'description': _('Help fund numeric identifier'),
                    'example_value': '7',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=12,
        json_schema_response=response_schema(
            {
                "type": "object",
                "properties": {
                    "avis_pi": {
                        "type": "object",
                        "properties": {
                            "avis": {"type": "string"},
                            "date": DATE_SCHEMA,
                            "montant": {"type": "number"},
                        }
                    },
                    "budget": {
                        "type": "object",
                        "properties": {
                            "date_reception": {"type": "string"},
                            "justificatifs": {
                                "type": "array",
                                "items": {
                                    "type": "object",
                                    "properties": {
                                        "conforme": {"type": "string"},
                                        "date_reception": DATE_SCHEMA,
                                        "date_relance": DATE_SCHEMA,
                                        "num_versement": {"type": "integer"},
                                        "reception": {"type": "string"},
                                        "type": {"type": "string"}
                                    }
                                }
                            },
                            "nombre_versements": {"type": "integer"},
                        }
                    },
                    "cloture": {
                        "type": "object",
                        "properties": {
                            "date_cloture": DATE_SCHEMA,
                            "date_relance": DATE_SCHEMA,
                        }
                    },
                    "code_tfi": {"type": "string"},
                    "decision_sai": {
                        "type": "object",
                        "properties": {
                            "date": DATE_SCHEMA,
                            "decision": {"type": "string"},
                            "montant": {"type": "number"}
                        }
                    },
                    "demande": {
                        "type": "object",
                        "properties": {
                            "date": DATE_SCHEMA,
                            "montant": {"type": "number"},
                        }
                    },
                    "id": {"type": "integer"},
                    "lib_tfi": {"type": "string"},
                    "recours": {
                        "type": "object",
                        "properties": {
                            "date_decision": DATE_SCHEMA,
                            "date_demande": DATE_SCHEMA,
                            "decision": {"type": "string"},
                            "montant": {"type": "string"}
                        }
                    }
                }
            }
        ),
    )
    def platform_beneficiaire_fondsaide_detail(
        self, request, platform_id, beneficiary_id, fondsaide_id, email, ip=None
    ):
        return self.get(
            'platform/%s/beneficiaire/%s/fondsaide/%s/' % (platform_id, beneficiary_id, fondsaide_id),
            email=email,
            ip=ip,
        )

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/'
        r'beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/'
        r'affectation/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/affectation/',
        description=_('Get beneficiary affectations'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=13,
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'properties': {
                        'id': {'type': 'integer'},
                        'dispositif': {'type': 'string'},
                        'plateforme': {'type': 'string'},
                        'code_pi': {'type': 'string'},
                        'referent': {'type': 'string'},
                        'date_deb': DATE_SCHEMA,
                        'origine': {'type': 'string'},
                    },
                },
            }
        ),
    )
    def platform_beneficiaire_affectation(
        self, request, platform_id, beneficiary_id, email, ip=None
    ):
        return self.get(
            'platform/%s/beneficiaire/%s/affectation/' % (platform_id, beneficiary_id),
            email=email,
            ip=ip,
        )

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/'
        r'beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/'
        r'affectation/(?P<affectation_id>[0-9]{1,10})/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/affectation/{affectation_id}/',
        description=_('Get beneficiary affectation details'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
                'affectation_id': {
                    'description': _('Help fund numeric identifier'),
                    'example_value': '7',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=14,
        json_schema_response=response_schema(
            {
                'type': 'object',
                'properties': {
                    'id': {'type': 'integer'},
                    'dispositif': {'type': 'string'},
                    'plateforme': {'type': 'string'},
                    'referent': {
                        'type': 'object',
                        'properties': {
                            'nom': {'type': 'string'},
                            'prenom': {'type': 'string'},
                        }
                    },
                    'code_pi': {'type': 'string'},
                    'date_deb': DATE_SCHEMA,
                    'origine': {'type': 'string'},
                    'erreur': OUI_NON_ENUM,
                    'date_deb': DATE_SCHEMA,
                    'origin': {'type': 'string'},
                    'prescripteur': {
                        'type': 'object',
                        'properties': {
                            'type': {'type': 'string'},
                            'dispositif': {'type': 'string'},
                            'plateforme': {'type': 'string'},
                        },
                    },
                    'rendez-vous': {
                        'type': 'object',
                        'properties': {
                            'date_prise': DATE_SCHEMA,
                            'relance': {
                                'type': 'object',
                                'properties': {
                                    'date': DATE_SCHEMA,
                                    'motif': {'type': 'string'},
                                    'lib_motif': {'type': 'string'},
                                },
                            },
                            'date_reelle': DATE_SCHEMA,
                            'resultat': {'type': 'string'},
                            'lib_resultat': {'type': 'string'},
                        }
                    },
                    'fin': {
                        'type': 'object',
                        'properties': {
                            'date': DATE_SCHEMA,
                            'motif': {'type': 'string'},
                            'lib_motif': {'type': 'string'},
                        }
                    },
                    'commentaire_ref': {'type': 'string'},
                },
            }
        ),
    )
    def platform_beneficiaire_affectation_detail(
        self, request, platform_id, beneficiary_id, affectation_id, email, ip=None
    ):
        return self.get(
            'platform/%s/beneficiaire/%s/affectation/%s/' % (platform_id, beneficiary_id, affectation_id),
            email=email,
            ip=ip,
        )

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/'
        r'beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/'
        r'convo/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/convo/',
        description=_('Get beneficiary convocations'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=15,
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    "type": "object",
                    "properties": {
                        "convos_par_motif": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "nombre": {"type": "integer"},
                                    "motif": {"type": "string"},
                                }
                            },
                        },
                        "derniere_consequence": {
                            "type": "object",
                            "properties": {
                                "date": DATE_SCHEMA,
                                "consequence": {"type": "string"},
                            }
                        }
                    }
                },
            }
        ),
    )
    def platform_beneficiaire_convo(
        self, request, platform_id, beneficiary_id, email, ip=None
    ):
        return self.get(
            'platform/%s/beneficiaire/%s/convo/' % (platform_id, beneficiary_id),
            email=email,
            ip=ip,
        )

    @endpoint(
        name='platform',
        pattern=r'^(?P<platform_id>[0-9]{1,10})/'
        r'beneficiaire/(?P<beneficiary_id>[0-9]{1,10})/'
        r'emploi/$',
        example_pattern='{platform_id}/beneficiaire/{beneficiary_id}/emploi/',
        description=_('Get beneficiary employments'),
        perm='can_access',
        parameters=parameters(
            {
                'platform_id': {
                    'description': _('Platform numeric identifier'),
                    'example_value': '11',
                },
                'beneficiary_id': {
                    'description': _('Beneficiary numeric identifier'),
                    'example_value': '12',
                },
            }
        ),
        display_category=_('Platform'),
        display_order=16,
        json_schema_response=response_schema(
            {
                'type': 'array',
                'items': {
                    "type": "object",
                    "properties": {
                        "id": {"type": "string"},
                        "code_axe": {"type": "string"},
                        "lib_axe": {"type": "string"},
                        "code_rome": {"type": "string"},
                        "lib_rome": {"type": "string"},
                        "code_categorie": {"type": "string"},
                        "lib_categorie": {"type": "string"},
                        "lib_secteur": {"type": "string"},
                        "lib_niveau": {"type": "string"},
                        "lib_modalite": {"type": "string"},
                        "date_inscription": DATE_SCHEMA,
                        "date_sortie": DATE_SCHEMA,
                        "motif_sortie": {"type": "string"},
                        "date_dernier_ent": DATE_SCHEMA,
                    },
                },
            }
        ),
    )
    def platform_beneficiaire_emploi(
        self, request, platform_id, beneficiary_id, email, ip=None
    ):
        return self.get(
            'platform/%s/beneficiaire/%s/emploi/' % (platform_id, beneficiary_id),
            email=email,
            ip=ip,
        )
