# coding: utf-8
# Passerelle - uniform access to data and services
# Copyright (C) 2019  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import unicode_literals

from django.db import models
from django.utils.translation import ugettext_lazy as _

from passerelle.utils.api import endpoint
from passerelle.utils.xml import JSONSchemaFromXMLSchema

import xmlschema

from . import abstract


class DDPACSSchema(JSONSchemaFromXMLSchema):
    type_map = {
        'CiviliteType': 'civilite',
    }
    civilite_map = {
        'Monsieur': 'M',
        'Madame': 'MME',
    }

    @classmethod
    def schema_civilite(cls):
        return {
            'type': 'string',
            'enum': ['Madame', 'Monsieur'],
        }

    def encode_civilite(self, obj):
        try:
            return self.civilite_map[obj]
        except KeyError:
            raise xmlschema.XMLSchemaValidationError(self, obj, reason='civilite invalide')

    def decode_civilite(self, data):
        for key, value in self.civilite_map.items():
            if data.text == value:
                return xmlschema.ElementData(tag=data.tag, text=key, content=data.content, attributes=data.attributes)
        raise xmlschema.XMLSchemaValidationError(self, data, reason='civilite invalide %s')


class Resource(abstract.Resource):
    category = _('Civil Status Connectors')
    xsd_root_element = 'PACS'
    flow_type = 'depotDossierPACS'
    doc_type = 'flux-pacs'
    xmlschema_class = DDPACSSchema

    class Meta:
        verbose_name = _('PACS request (MDEL DDPACS)')

    @endpoint(perm='can_access',
              methods=['post'],
              description=_('Create request'),
              post={
                  'request_body': {
                      'schema': {
                          'application/json': None
                      }
                  }
              })
    def create(self, request, post_data):
        return self._handle_create(request, post_data)

Resource.create.endpoint_info.post['request_body']['schema']['application/json'] = Resource.get_create_schema()


class Demand(abstract.Demand):
    resource = models.ForeignKey(Resource)

    class Meta:
        verbose_name = _('MDEL compatible DDPACS request')
