# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2018 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django.db import models
from django.template import Template, Context
from django.utils.six.moves.urllib import parse as urlparse
from django.utils.translation import ugettext_lazy as _

from passerelle.utils.jsonresponse import APIError
from passerelle.utils.api import endpoint
from passerelle.base.models import BaseResource, HTTPResource


class ArcGISError(APIError):
    pass


class ArcGIS(BaseResource, HTTPResource):
    category = _('Geographic information system')

    base_url = models.URLField(_('Webservice Base URL'))

    class Meta:
        verbose_name = _('ArcGIS REST API')

    @endpoint(name='mapservice-query',
              description=_('Map Service Query'),
              perm='can_access',
              parameters={
                  'folder': {
                      'description': _('Folder name'),
                      'example_value': 'Specialty',
                  },
                  'service': {
                      'description': _('Service name'),
                      'example_value': 'ESRI_StateCityHighway_USA',
                  },
                  'layer': {
                      'description': _('Layer or table name'),
                      'example_value': '1',
                  },
                  'lat': {'description': _('Latitude')},
                  'lon': {'description': _('Longitude')},
                  'latmin': {'description': _('Minimal latitude (envelope)')},
                  'lonmin': {'description': _('Minimal longitude (envelope)')},
                  'latmax': {'description': _('Maximal latitude (envelope)')},
                  'lonmax': {'description': _('Maximal longitude (envelope)')},
                  'q': {'description': _('Search text in display field')},
                  'template': {
                      'description': _('Django template for text attribute'),
                      'example_value': '{{ attributes.STATE_NAME }} ({{ attributes.STATE_ABBR }})',
                  },
                  'id_template': {
                      'description': _('Django template for id attribute'),
                  },
                  'full': {
                      'description': _('Returns all ArcGIS informations (geometry, metadata)'),
                      'type': 'bool',
                  },
              })
    def mapservice_query(self, request, service, layer='0', folder='', lat=None, lon=None,
                         latmin=None, lonmin=None, latmax=None, lonmax=None, q=None,
                         template=None, id_template=None, full=False, **kwargs):
        url = urlparse.urljoin(self.base_url, 'services/')
        if folder:
            url = urlparse.urljoin(url, folder + '/')
        url = urlparse.urljoin(url, service + '/MapServer/' + layer + '/query')

        # build query params
        # cf https://developers.arcgis.com/rest/services-reference/query-map-service-layer-.htm
        params = {
            'f': 'json',
            'inSR': '4326',
            'outSR': '4326',
            'outFields': '*',
        }
        if lat and lon:
            try:
                lon, lat = float(lon), float(lat)
            except (ValueError,):
                raise APIError('<lon> and <lat> must be floats', http_status=400)
            params['geometry'] = '{},{}'.format(lon, lat)
            params['geometryType'] = 'esriGeometryPoint'
        elif latmin and lonmin and latmax and lonmax:
            try:
                lonmin, latmin = float(lonmin), float(latmin)
                lonmax, latmax = float(lonmax), float(latmax)
            except (ValueError,):
                raise APIError('<lonmin> <latmin> <lonmax> and <latmax> must be floats',
                               http_status=400)
            params['geometry'] = '{},{},{},{}'.format(lonmin, latmin, lonmax, latmax)
            params['geometryType'] = 'esriGeometryEnvelope'
        if q is not None:
            params['text'] = q
        # consider all remaining parameters as ArcGIS ones
        params.update(kwargs)
        if 'where' not in params and 'text' not in params:
            params['where'] = '1=1'
        if 'distance' in params and 'units' not in params:
            params['units'] = 'esriSRUnit_Meter'

        response = self.requests.get(url, params=params)

        # errors
        if response.status_code // 100 != 2:
            raise ArcGISError('ArcGIS returned status code %s' % response.status_code)
        try:
            infos = response.json()
        except (ValueError,):
            raise ArcGISError('ArcGIS returned invalid JSON content: %r' % response.content)
        if 'error' in infos:
            err_desc = infos['error'].get('message') or 'unknown ArcGIS error'
            raise ArcGISError(err_desc, data=infos)

        features = infos.pop('features', [])
        id_fieldname = infos.get('objectIdFieldName') or 'OBJECTID'
        text_fieldname = infos.get('displayFieldName')
        if infos.get('fieldAliases'):
            aliases = {v: k for k, v in infos['fieldAliases'].items()}
        else:
            aliases = {}

        # data is the features list, with 'id' and 'text' entries
        data = []

        def get_feature_attribute(feature, attribute):
            if attribute in feature['attributes']:
                return feature['attributes'][attribute]
            return feature['attributes'].get(aliases.get(attribute))

        if template:
            template = Template(template)
        if id_template:
            id_template = Template(id_template)
        for n, feature in enumerate(features):
            if 'attributes' in feature:
                feature['id'] = '%s' % get_feature_attribute(feature, id_fieldname)
                feature['text'] = '%s' % get_feature_attribute(feature, text_fieldname)
            else:
                feature['id'] = feature['text'] = '%d' % (n+1)
            if template:
                feature['text'] = template.render(Context(feature))
            if id_template:
                feature['id'] = id_template.render(Context(feature))
            if not full and 'geometry' in feature:
                del feature['geometry']
            data.append(feature)

        if full:
            return {'data': data, 'metadata': infos}
        return {'data': data}

    @endpoint(name='district',
              description=_('Districts in Nancy Town (deprecated)'),
              parameters={
                  'lat': {'description': _('Latitude')},
                  'lon': {'description': _('Longitude')},
              })
    def district(self, request, lon=None, lat=None):
        if 'NANCY_Grc' in self.base_url:
            # Nancy URL used to contains folder, service and layer, remove them
            self.base_url = 'https://geoservices.grand-nancy.org/arcgis/rest/'
        features = self.mapservice_query(request, folder='public', service='NANCY_Grc', layer='0',
                                         template='{{ attributes.NOM }}',
                                         id_template='{{ attributes.NUMERO }}',
                                         lon=lon, lat=lat)['data']
        if not features:
            raise APIError('No features found.')
        for feature in features:
            del feature['attributes']
        feature['id'] = int(feature['id'])
        if len(features) == 1:
            return {'data': features[0]}
        return {'data': features}
