# -*- coding: utf-8 -*-
# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2017  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import datetime
import xml.etree.ElementTree as ET

from django.db import models
from django.http import Http404
from django.utils.translation import ugettext_lazy as _

from passerelle.base.models import BaseResource
from passerelle.utils.api import endpoint

class AirQuality(BaseResource):
    category = _('Misc')
    api_description = _(u'''
        This API provides a unique format for the air quality data of various places.
        (But only supports the Rhône-Alpes region for now).
        ''')

    class Meta:
        verbose_name = _('Air Quality')

    @endpoint(pattern='^(?P<country>\w+)/(?P<city>\w+)/$',
              example_pattern='{country}/{city}/',
              parameters={
                  'country': {'description': _('Country Code'), 'example_value': 'fr'},
                  'city': {'description': _('City Name'), 'example_value': 'lyon'},
              })
    def details(self, request, country, city, **kwargs):
        methods = {
            ('fr', 'albertville'): 'air_rhonealpes',
            ('fr', 'annecy'): 'air_rhonealpes',
            ('fr', 'bourg-en-bresse'): 'air_rhonealpes',
            ('fr', 'chambery'): 'air_rhonealpes',
            ('fr', 'chamonix'): 'air_rhonealpes',
            ('fr', 'grenoble'): 'air_rhonealpes',
            ('fr', 'lyon'): 'air_rhonealpes',
            ('fr', 'roanne'): 'air_rhonealpes',
            ('fr', 'saint-etienne'): 'air_rhonealpes',
            ('fr', 'valence'): 'air_rhonealpes',
            ('fr', 'vienne'): 'air_rhonealpes',
        }
        local_method = methods.get((country, city))
        if not local_method:
            raise Http404()
        return getattr(self, local_method)(request, country, city, **kwargs)

    def air_rhonealpes(self, request, country, city, **kwargs):
        response = self.requests.get('https://www.atmo-auvergnerhonealpes.fr/site/indice/XML/ATMO/now/%s' % city)
        atmo = ET.fromstring(response.content)
        data = {
            'latest': {
                'date': datetime.datetime.strptime(
                    atmo.find('JOUR_ATMO/MESURE/INDICE/JOUR_INDICE').attrib['jour'],
                    '%d/%m/%Y').strftime('%Y-%m-%d'),
                'value': atmo.findtext('JOUR_ATMO/MESURE/INDICE/VALEUR'),
            },
            'comment': atmo.findtext('JOUR_ATMO/COMMENTAIRE').strip(),
        }
        if atmo.findtext('JOUR_ATMO/PREVISION_J1/INDICE/VALEUR') != '-':
            data.update({
                'forecast': {
                    'j1': {
                        'date': datetime.datetime.strptime(
                            atmo.findtext('JOUR_ATMO/PREVISION_J1/INDICE/JOUR_INDICE'),
                            '%d/%m/%Y').strftime('%Y-%m-%d'),
                        'value': atmo.findtext('JOUR_ATMO/PREVISION_J1/INDICE/VALEUR'),
                    },
                },
            })
        return {'data': data, 'err': 0}
