# -*- coding: utf-8 -*-
import json
import requests

from django.utils.six import string_types
from django.utils.translation import ugettext_lazy as _
from django.db import models

from passerelle.utils.jsonresponse import APIError
from passerelle.base.models import BaseResource
from passerelle.sms import SMSGatewayMixin


class ChoositSMSGateway(BaseResource, SMSGatewayMixin):
    key = models.CharField(verbose_name=_('Key'), max_length=64)
    default_country_code = models.CharField(verbose_name=_('Default country code'), max_length=3,
                                            default=u'33')
    default_trunk_prefix = models.CharField(verbose_name=_('Default trunk prefix'), max_length=2,
                                            default=u'0')
    # FIXME: add regexp field, to check destination and from format

    TEST_DEFAULTS = {
        'create_kwargs': {
            'key': '1234',
        },
        'test_vectors': [
            {
                'response': '',
                'result': {
                    'err': 1,
                    'err_desc': 'Choosit error: some destinations failed',
                    'data': [
                        [u'0033688888888', u'Choosit error: bad JSON response'],
                        [u'0033677777777', u'Choosit error: bad JSON response'],
                    ]
                }
            },
            {
                'response': {
                    'error': 'not ok',
                },
                'result': {
                    'err': 1,
                    'err_desc': 'Choosit error: some destinations failed',
                    'data': [
                        [u'0033688888888', u'Choosit error: not ok'],
                        [u'0033677777777', u'Choosit error: not ok'],
                    ],
                }
            },
            {
                'response': {
                    'result': u'Envoi terminé',
                    'sms_id': 1234,
                },
                'result': {
                    'err': 0,
                    'data': [
                        [u'0033688888888', {'result': u'Envoi terminé', 'sms_id': 1234}],
                        [u'0033677777777', {'result': u'Envoi terminé', 'sms_id': 1234}],
                    ],
                }
            }

        ],
    }
    URL = 'http://sms.choosit.com/webservice'

    class Meta:
        verbose_name = 'Choosit'
        db_table = 'sms_choosit'

    @classmethod
    def get_verbose_name(cls):
        return cls._meta.verbose_name

    def send_msg(self, text, sender, destinations, **kwargs):
        """Send a SMS using the Choosit provider"""
        # from http://sms.choosit.com/documentation_technique.html
        # unfortunately it lacks a batch API...
        destinations = self.clean_numbers(destinations,
                                          self.default_country_code,
                                          self.default_trunk_prefix)
        results = []
        for dest in destinations:
            params = {
                'key': self.key,
                'recipient': dest,
                'content': text[:160],
            }
            data = {'data': json.dumps(params)}
            try:
                r = self.requests.post(self.URL, data=data)
            except requests.RequestException as e:
                results.append('Choosit error: %s' % e)
            else:
                try:
                    output = r.json()
                except ValueError as e:
                    results.append('Choosit error: bad JSON response')
                else:
                    if not isinstance(output, dict):
                        results.append('Choosit error: JSON response is not a dict %r' % output)
                    elif 'error' in output:
                        results.append(u'Choosit error: %s' % output['error'])
                    else:
                        results.append(output)
        if any(isinstance(result, string_types) for result in results):
            raise APIError('Choosit error: some destinations failed',
                           data=list(zip(destinations, results)))
        return list(zip(destinations, results))
