# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2016 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import absolute_import

import inspect

from django.core.urlresolvers import reverse
from django.utils import six
from django.utils.safestring import mark_safe

# make APIError available from this module
from .jsonresponse import APIError


class endpoint(object):
    do_not_call_in_templates = True

    def __init__(self, serializer_type='json-api', perm=None, methods=['get'], name=None, pattern=None,
                 wrap_response=False,
                 description=None,
                 description_get=None,
                 description_post=None,
                 description_patch=None,
                 long_description=None,
                 long_description_get=None,
                 long_description_post=None,
                 long_description_patch=None,
                 example_pattern=None,
                 parameters=None,
                 cache_duration=None,
                 post=None):
        self.perm = perm
        self.methods = methods
        self.serializer_type = serializer_type
        self.pattern = pattern
        self.name = name
        self.wrap_response = wrap_response
        self.descriptions = {
            'get': description_get or description,
            'post': description_post or description,
            'patch': description_patch or description,
        }
        self.long_descriptions = {
            'get': long_description_get or long_description,
            'post': long_description_post or long_description,
            'patch': long_description_patch or long_description,
        }
        self.example_pattern = example_pattern
        self.parameters = parameters
        self.cache_duration = cache_duration
        self.post = post
        if post:
            self.methods = ['post']
            if post.get('description'):
                self.descriptions['post'] = post.get('description')
            if post.get('long_description'):
                self.long_descriptions['post'] = post.get('long_description')

    def __call__(self, func):
        func.endpoint_info = self
        if not self.name:
            self.name = func.func_name if six.PY2 else func.__name__
        self.func = func
        return func

    def get_example_params(self):
        return dict([(x, self.parameters[x]['example_value']) for x in self.parameters or {}
                     if x in self.parameters and 'example_value' in self.parameters[x]])

    def get_query_parameters(self):
        query_parameters = []
        for param, param_value in self.get_example_params().items():
            if param in (self.example_pattern or ''):
                continue
            query_parameters.append((param, param_value))
        return query_parameters

    def example_url(self):
        kwargs = {
            'connector': self.object.get_connector_slug(),
            'slug': self.object.slug,
            'endpoint': self.name,
        }
        if self.example_pattern:
            kwargs['rest'] = self.example_pattern.format(**self.get_example_params())

        query_string = ''
        query_parameters = self.get_query_parameters()
        if query_parameters:
            query_string = '?' + '&'.join(['%s=%s' % x for x in query_parameters])

        return reverse('generic-endpoint', kwargs=kwargs) + query_string

    def example_url_as_html(self):
        kwargs = {
            'connector': self.object.get_connector_slug(),
            'slug': self.object.slug,
            'endpoint': self.name,
        }
        if self.example_pattern:
            kwargs['rest'] = self.example_pattern.format(
                    **dict([(x, '$%s$' % x) for x in self.get_example_params().keys()]))

        url = reverse('generic-endpoint', kwargs=kwargs)
        for param in self.get_example_params():
            url = url.replace('$%s$' % param,  '<i class="varname">%s</i>' % param)

        query_string = ''
        query_parameters = self.get_query_parameters()
        if query_parameters:
            query_string = '?' + '&amp;'.join(['%s=<i class="varname">%s</i>' % (x[0], x[0]) for x in query_parameters])

        return mark_safe(url + query_string)

    def has_params(self):
        argspec = inspect.getargspec(self.func)
        return len(argspec.args) > 2 # (self, request)

    @property
    def description(self):
        return self.descriptions.get(self.http_method)

    @property
    def long_description(self):
        return self.long_descriptions.get(self.http_method)

    @property
    def body_schemas(self):
        if (self.http_method == 'post'
                and self.post
                and 'request_body' in self.post
                and 'schema' in self.post['request_body']):
            return self.post['request_body']['schema']
        return {}

    def get_params(self):
        params = []
        defaults = dict(zip(
            reversed(inspect.getargspec(self.func).args),
            reversed(inspect.getargspec(self.func).defaults or [])))
        for param in inspect.getargspec(self.func).args[2:]:
            if param == 'post_data':
                continue
            param_info = {'name': param}
            if self.parameters and param in self.parameters and self.parameters[param].get('description'):
                param_info['description'] = self.parameters[param].get('description')
            if param in defaults:
                param_info['optional'] = True
                param_info['default_value'] = defaults[param]
            params.append(param_info)
        return params
