# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2016  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import base64
import binascii
import functools
import httplib2
import re

from cmislib import CmisClient
from cmislib.exceptions import CmisException
from cmislib.exceptions import ObjectNotFoundException
from cmislib.exceptions import PermissionDeniedException
from cmislib.exceptions import UpdateConflictException
from django.db import models
from django.utils.translation import ugettext_lazy as _
from django.utils.six import StringIO
from django.utils.six.moves.urllib import error as urllib2

from passerelle.base.models import BaseResource
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError


SPECIAL_CHARS = '!#$%&+-^_`~;[]{}+=~'
FILE_PATH_PATTERN = r'^(/|(/[\w%s]+)+)$' % re.escape(SPECIAL_CHARS)
FILE_NAME_PATTERN = r'[\w%s\.]+$' % re.escape(SPECIAL_CHARS)


UPLOAD_SCHEMA = {
    'type': 'object',
    'properties': {
        'file': {
            'type': 'object',
            'properties': {
                'filename': {
                    'type': 'string',
                    'pattern': FILE_NAME_PATTERN,
                },
                'content': {'type': 'string'},
                'content_type': {'type': 'string'},
            },
            'required': ['content']
        },
        'filename': {
            'type': 'string',
            'pattern': FILE_NAME_PATTERN,
        },
        'path': {
            'type': 'string',
            'pattern': FILE_PATH_PATTERN,
        },
    },
    'required': ['file', 'path']
}


class CmisConnector(BaseResource):
    cmis_endpoint = models.URLField(
        max_length=400, verbose_name=_('CMIS Atom endpoint'),
        help_text=_('URL of the CMIS Atom endpoint'))
    username = models.CharField(max_length=128, verbose_name=_('Service username'))
    password = models.CharField(max_length=128, verbose_name=_('Service password'))
    category = _('Business Process Connectors')

    class Meta:
        verbose_name = _('CMIS connector')

    def get_description_fields(self):
        fields = super(CmisConnector, self).get_description_fields()
        return [(x[0], x[1]) for x in fields if x[0].name != 'password']

    @endpoint(
        perm='can_access',
        post={
            'request_body': {
                'schema': {
                    'application/json': UPLOAD_SCHEMA,
                }
            }
        })
    def uploadfile(self, request, post_data):
        error, error_msg, data = self._validate_inputs(post_data)
        if error:
            self.logger.debug("received invalid data: %s" % error_msg)
            raise APIError(error_msg, http_status=400)
        filename = data.get('filename') or data['file']['filename']
        self.logger.info("received file_name: '%s', file_path: '%s'", filename, data["path"])
        cmis_gateway = CMISGateway(self.cmis_endpoint, self.username, self.password, self.logger)
        doc = cmis_gateway.create_doc(
            filename,
            data['path'],
            data['file_byte_content'])
        return {'data': {'properties': doc.properties}}

    def _validate_inputs(self, data):
        """ process dict
        return a tuple (error, error_msg, data)
        """
        file_ = data['file']
        if 'filename' not in file_ and 'filename' not in data:
            return True, '"filename" or "file[\'filename\']" is required', None

        try:
            data['file_byte_content'] = base64.b64decode(file_['content'])
        except (TypeError, binascii.Error):
            return True, '"file[\'content\']" must be a valid base64 string', None

        return False, '', data


def wrap_cmis_error(f):
    @functools.wraps(f)
    def wrapper(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        except (urllib2.URLError, httplib2.HttpLib2Error) as e:
            # FIXME urllib2 still used for cmslib 0.5 compat
            raise APIError("connection error: %s" % e)
        except PermissionDeniedException as e:
            raise APIError("permission denied: %s" % e)
        except UpdateConflictException as e:
            raise APIError("update conflict: %s" % e)
        except CmisException as e:
            raise APIError("cmis binding error: %s" % e)
    return wrapper


class CMISGateway(object):

    def __init__(self, cmis_endpoint, username, password, logger):
        self._cmis_client = CmisClient(cmis_endpoint, username, password)
        self._logger = logger

    def _get_or_create_folder(self, file_path):
        repo = self._cmis_client.defaultRepository
        try:
            self._logger.debug("searching '%s'" % file_path)
            res = repo.getObjectByPath(file_path)
            self._logger.debug("'%s' found" % file_path)
            return res
        except ObjectNotFoundException:
            self._logger.debug("'%s' not found" % file_path)
            basepath = ""
            folder = repo.rootFolder
            for path_part in file_path.strip('/').split('/'):
                basepath += '/%s' % path_part
                try:
                    self._logger.debug("searching '%s'" % basepath)
                    folder = repo.getObjectByPath(basepath)
                    self._logger.debug("'%s' found" % basepath)
                except ObjectNotFoundException:
                    self._logger.debug("'%s' not found" % basepath)
                    folder = folder.createFolder(path_part)
                    self._logger.debug("create folder '%s'" % basepath)
            return folder

    @wrap_cmis_error
    def create_doc(self, file_name, file_path, file_byte_content):
        folder = self._get_or_create_folder(file_path)
        return folder.createDocument(file_name, contentFile=StringIO(file_byte_content))
