# passerelle - uniform access to multiple data sources and services
# Copyright (C) 2017  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

'''Gateway to API-Particulier web-service from SGMAP:
   https://particulier.api.gouv.fr/
'''

from collections import OrderedDict
import requests

from django.db import models
from django.utils import six
from django.utils.six.moves.urllib import parse
from django.utils.translation import ugettext_lazy as _

from passerelle.base.models import BaseResource
from passerelle.utils.api import endpoint
from passerelle.utils.jsonresponse import APIError


class APIParticulier(BaseResource):
    PLATFORMS = [
        {
            'name': 'prod',
            'label': _('Production'),
            'url': 'https://particulier.api.gouv.fr/api/'
        },
        {
            'name': 'test',
            'label': _('Test'),
            'url': 'https://particulier-test.api.gouv.fr/api/',
        },
    ]
    PLATFORMS = OrderedDict([(platform['name'], platform) for platform in PLATFORMS])

    platform = models.CharField(
        verbose_name=_('Platform'),
        max_length=8,
        choices=[(key, platform['label']) for key, platform in PLATFORMS.items()])

    api_key = models.CharField(
        max_length=256,
        default='',
        blank=True,
        verbose_name=_('API key'))

    @property
    def url(self):
        return self.PLATFORMS[self.platform]['url']

    def get(self, path, **kwargs):
        user = kwargs.pop('user', None)
        url = parse.urljoin(self.url, path)
        headers = {'X-API-KEY': self.api_key}
        if user:
            headers['X-User'] = user
        response = None
        try:
            response = self.requests.get(
                url,
                headers=headers,
                **kwargs)
        except requests.RequestException as e:
            raise APIError(
                u'API-particulier platform "%s" connection error: %s' %
                (self.platform, response.status_code),
                data={
                    'platform': self.platform,
                    'error': six.text_type(e),
                })
        try:
            data = response.json()
        except ValueError as e:
            content = repr(response.content[:1000])
            raise APIError(
                u'API-particulier platform "%s" returned non-JSON content with status %s: %s' %
                (self.platform, response.status_code, content),
                data={
                    'status_code': response.status_code,
                    'exception': six.text_type(e),
                    'platform': self.platform,
                    'content': content,
                })
        if response.status_code != 200:
            if data.get('error') == 'not_found':
                return {
                    'err': 1,
                    'err_desc': data.get('message', 'not-found'),
                }
            raise APIError(
                u'API-particulier platform "%s" returned a non 200 status %s: %s' %
                (self.platform, response.status_code, data),
                data={
                    'status_code': response.status_code,
                    'platform': self.platform,
                    'content': data,
                })
        return {
            'err': 0,
            'data': data,
        }

    @endpoint(perm='can_access',
              description=_('Get citizen\'s fiscal informations'),
              parameters={
                  'numero_fiscal': {
                      'description': _('fiscal identifier'),
                      'example_value': '1562456789521',
                  },
                  'reference_avis': {
                      'description': _('tax notice number'),
                      'example_value': '1512456789521',
                  },
                  'user': {
                      'description': _('requesting user'),
                      'example_value': 'John Doe (agent)',
                  },
              })
    def impots_svair(self, request, numero_fiscal, reference_avis, user=None):
        if not numero_fiscal.strip() or not reference_avis.strip():
            raise APIError('missing numero_fiscal or reference_avis', status_code=400)
        return self.get('impots/svair', params={
            'numeroFiscal': numero_fiscal,
            'referenceAvis': reference_avis[:13],
        }, user=user)

    @endpoint(perm='can_access',
              description=_('Get citizen\'s fiscal address'),
              parameters={
                  'numero_fiscal': {
                      'description': _('fiscal identifier'),
                      'example_value': '1562456789521',
                  },
                  'reference_avis': {
                      'description': _('tax notice number'),
                      'example_value': '1512456789521',
                  },
                  'user': {
                      'description': _('requesting user'),
                      'example_value': 'John Doe (agent)',
                  },
              })
    def impots_adresse(self, request, numero_fiscal, reference_avis, user=None):
        if not numero_fiscal.strip() or not reference_avis.strip():
            raise APIError('missing numero_fiscal or reference_avis', status_code=400)
        return self.get('impots/adress', params={
            'numeroFiscal': numero_fiscal,
            'referenceAvis': reference_avis[:13],
        }, user=user)

    @endpoint(perm='can_access',
              description=_('Get family allowances recipient informations'),
              parameters={
                  'code_postal': {
                      'description': _('postal code'),
                      'example_value': '99148',
                  },
                  'numero_allocataire': {
                      'description': _('recipient identifier'),
                      'example_value': '0000354',
                  },
                  'user': {
                      'description': _('requesting user'),
                      'example_value': 'John Doe (agent)',
                  },
              })
    def caf_famille(self, request, code_postal, numero_allocataire, user=None):
        if not code_postal.strip() or not numero_allocataire.strip():
            raise APIError('missing code_postal or numero_allocataire', status_code=400)
        return self.get('caf/famille', params={
            'codePostal': code_postal,
            'numeroAllocataire': numero_allocataire,
        }, user=user)

    category = _('Business Process Connectors')

    class Meta:
        verbose_name = _('API Particulier')
