# -*- coding: utf-8 -*-

from httmock import HTTMock
import json
import mock
import pytest

from passerelle.contrib.lille_urban_card.models import LilleUrbanCard
from passerelle.utils.jsonresponse import APIError

import utils

@pytest.fixture
def connector(db):
    return utils.setup_access_rights(
        LilleUrbanCard.objects.create(
            slug='test',
            base_url='http://localhost',
            username='test',
            password='secret'))


TOKEN_ERROR_RESPONSE = '{"erreur":"Authentification échouée"}'
TOKEN_RESPONSE = '{"token": "eyJhbGciO..."}'

def mocked_http(url, request):
    if url.path == '/clu/ws/auth/connexion':
        return {'content': TOKEN_RESPONSE, 'status_code': 200}
    if url.path == '/clu/ws/demanderCarte':
        content = {
            'n_demande_clu': 10000005,
            'request': json.loads(request.body),  # for tests
        }
        return {'content': json.dumps(content), 'status_code': 200}
    if url.path.startswith('/clu/ws/consulterDemande/'):
        content = {
            "n_demande_clu": '...',
            "statut": "200",
            "statut_desc": "CARTE_PRODUITE_EXPEDIEE",
            "date": "2019-01-01 00:00:00",
        }
        return {'content': json.dumps(content), 'status_code': 200}


@mock.patch('passerelle.utils.Request.post')
def test_get_token(mocked_post, app, connector):
    with pytest.raises(APIError):
        mocked_post.return_value = utils.FakedResponse(content=TOKEN_ERROR_RESPONSE, status_code=400)
        connector.get_token()
    mocked_post.return_value = utils.FakedResponse(content=TOKEN_RESPONSE, status_code=200)
    connector.get_token()


def test_csp(app, connector):
    endpoint = utils.generic_endpoint_url('lille-urban-card', 'csp', slug=connector.slug)
    resp = app.get(endpoint)
    assert resp.json.get('data')

def test_card_request(app, connector):
    endpoint = utils.generic_endpoint_url('lille-urban-card', 'card_request', slug=connector.slug)
    with HTTMock(mocked_http):
        resp = app.post_json(endpoint, params={
            'civilite': 'Monsieur',
            'code_postal': '59000',
            'ville': 'Lille',
            'photo': {'content': 'xxx'},
            'telephone': '01.02.03.04.05',
            'recevoir_journal_senior': 'Non',
            'service_zoo': 'Oui',
            'service_zoo_newsletter': 'Non',
        })
        request = resp.json['data']['request']
        assert request['civilite'] == 1
        assert request['code_postal'] == 59000
        assert request['ville'] == 'LILLE'
        assert request['photo'] == 'xxx'
        assert request['telephone'] == '0102030405'
        assert request['services'] == [{'service': 'zoo', 'newsletter': 0}]
        assert resp.json['data']['n_demande_clu'] == 10000005

def test_card_status(app, connector):
    endpoint = utils.generic_endpoint_url('lille-urban-card', 'card_status', slug=connector.slug)
    with HTTMock(mocked_http):
        resp = app.get(endpoint + '?n_demande_clu=1234')
        assert resp.json['data']['statut_desc'] == 'CARTE_PRODUITE_EXPEDIEE'
