import mock
import pytest

import urlparse
from requests.exceptions import HTTPError

from django.contrib.contenttypes.models import ContentType

from passerelle.base.models import ApiUser, AccessRight
from passerelle.apps.clicrdv.models import ClicRdv


@pytest.fixture
def connector(db):
    return ClicRdv.objects.create(slug='test', group_id='5242', apikey='test', username='test',
                        password='test')


@mock.patch('passerelle.utils.Request.request')
def test_request_call(mocked_request, app, connector):
    mocked_request.json.return_value = "foo"
    connector.request('bar')
    assert mocked_request.call_count == 1
    req = mocked_request.call_args[0][1]
    assert req.startswith('https://sandbox.clicrdv.com/api/v1/groups/5242/bar')


@mock.patch('passerelle.utils.Request.request')
def test_interventionsets(mocked_request, app, connector):
    response = mock.Mock()
    response.json.return_value = {
            "totalRecords": 2,
            "records": [
                {
                    "sort": 1,
                    "publicname": "Une Demande de Passeport",
                    "name": "Demande",
                    "id": 7032,
                    "group_id": 5242,
                },
                {
                    "sort": 2,
                    "publicname": "Un Retrait de Passeport",
                    "name": "Retrait",
                    "id": 7033,
                    "group_id": 5242,
                },
            ]}
    mocked_request.return_value = response
    resp = app.get('/clicrdv/test/interventionsets/')
    assert len(resp.json.get('data')) == 2
    assert resp.json.get('data')[0]['text'] == 'Une Demande de Passeport'


@mock.patch('passerelle.utils.Request.request')
def test_interventionsets_details(mocked_request, app, connector):
    response = mock.Mock()
    response.json.return_value = {
            "totalRecords": 2,
            "records": [
                {
                    "sort": 1,
                    "publicname": "pour une personne",
                    "description": None,
                    "name": "1 personne",
                    "interventionset_id": 7032,
                    "group_id": 5242,
                    "id": 63258,
                    "abbr": "1 demande"
                    },
                {
                    "sort": 2,
                    "publicname": "pour deuxs personnes",
                    "description": None,
                    "name": "2 personnes",
                    "interventionset_id": 7032,
                    "group_id": 5242,
                    "id": 63259,
                    "abbr": "2 demandes"
                },
            ]}
    mocked_request.return_value = response
    resp = app.get('/clicrdv/test/interventionsets/7032/')
    assert len(resp.json.get('data')) == 2
    assert resp.json.get('data')[0]['text'] == 'pour une personne'


@mock.patch('passerelle.utils.Request.request')
def test_interventions_get_datetimes(mocked_request, app, connector):
    response = mock.Mock()
    response.json.return_value = {"availabletimeslots": []}
    mocked_request.return_value = response
    resp = app.get('/clicrdv/test/interventions/63258/dates/')
    assert resp.json.get('data') == []
    assert resp.json.get('err') == 0
    assert mocked_request.call_count == 1
    url = mocked_request.call_args[0][1]
    # https://sandbox.clicrdv.com/api/v1/groups/5242/availabletimeslots?
    #   intervention_ids[]=63258&start=2016-09-21&end=2017-09-22&apikey=test&format=json
    scheme, netloc, path, params, query, fragment = urlparse.urlparse(url)
    query = urlparse.parse_qs(query, keep_blank_values=True)
    assert scheme == 'https'
    assert netloc == 'sandbox.clicrdv.com'
    assert path == '/api/v1/groups/5242/availabletimeslots'
    assert params == ''
    assert fragment == ''
    assert query['intervention_ids[]'] == ['63258']
    assert 'start' in query
    assert 'end' in query
    assert query['apikey'] == ['test']
    assert query['format'] == ['json']

    response.json.return_value = {"availabletimeslots": [
        { "start": "2016-09-21 12:34:56" },
        { "start": "2016-09-22 11:22:33" }
    ]}
    mocked_request.return_value = response
    resp = app.get('/clicrdv/test/interventions/63258/dates/').json
    assert mocked_request.call_count == 2
    assert resp.get('err') == 0
    assert len(resp.get('data')) == 2
    assert resp['data'][0] == {'id': '2016-09-21', 'text': '21 September 2016'}
    assert resp['data'][1] == {'id': '2016-09-22', 'text': '22 September 2016'}

    response.json.return_value = {"availabletimeslots": [
            { "start": "2016-09-22 11:22:33" },
            { "start": "2016-09-21 12:34:56" }
    ]} # will be sorted
    mocked_request.return_value = response
    resp = app.get('/clicrdv/test/interventions/63258/datetimes/').json
    assert mocked_request.call_count == 3
    assert resp.get('err') == 0
    assert len(resp.get('data')) == 2
    assert resp['data'][0] == {'id': '2016-09-21-12:34:56', 'text': '21 September 2016 12:34'}
    assert resp['data'][1] == {'id': '2016-09-22-11:22:33', 'text': '22 September 2016 11:22'}

    response.json.return_value = {"availabletimeslots": [
            { "start": "2016-09-21 12:34:56" },
            { "start": "2016-09-21 11:22:33" }
    ]} # will be sorted
    mocked_request.return_value = response
    resp = app.get('/clicrdv/test/interventions/63258/2016-09-21/times').json
    assert mocked_request.call_count == 4
    url = mocked_request.call_args[0][1]
    scheme, netloc, path, params, query, fragment = urlparse.urlparse(url)
    query = urlparse.parse_qs(query, keep_blank_values=True)
    assert query['start'] == ['2016-09-21 00:00:00']
    assert query['end'] == ['2016-09-21 23:59:59']
    assert resp.get('err') == 0
    assert len(resp.get('data')) == 2
    assert resp['data'][0] == {'id': '11:22:33', 'text': '11:22'}
    assert resp['data'][1] == {'id': '12:34:56', 'text': '12:34'}

@mock.patch('passerelle.utils.Request.request')
def test_cancel_appointment(mocked_request, app, connector):
    obj_type = ContentType.objects.get_for_model(ClicRdv)
    apiuser = ApiUser.objects.create(username='apiuser', keytype='API',
                                     key='apiuser')
    AccessRight.objects.create(codename='can_manage_appointment',
                resource_type=obj_type, resource_pk=connector.pk,
                apiuser=apiuser)

    resp = app.get('/clicrdv/test/63258/cancel?apikey=apiuser').json
    assert mocked_request.call_count == 1
    assert resp['data']['success']

@mock.patch('passerelle.utils.Request.request',
            side_effect=HTTPError('{"msg": "cancel failed"}'))
def test_failed_cancel_appointment(mocked_request, app, connector):
    obj_type = ContentType.objects.get_for_model(ClicRdv)
    apiuser = ApiUser.objects.create(username='apiuser', keytype='API',
                                key='apiuser')
    AccessRight.objects.create(codename='can_manage_appointment',
                resource_type=obj_type, resource_pk=connector.pk,
                apiuser=apiuser)
    resp = app.get('/clicrdv/test/63258/cancel?apikey=apiuser').json
    assert mocked_request.call_count == 1
    assert resp.get('err') == 0
    assert resp['data']
    assert resp['data']['error'] == {'msg': 'cancel failed'}


@mock.patch('passerelle.utils.Request.request',
            side_effect=HTTPError('[{"error": "creation failed"}]'))
def test_failed_appointment_creation(mocked_request, app, connector):
    obj_type = ContentType.objects.get_for_model(ClicRdv)
    apiuser = ApiUser.objects.create(username='apiuser', keytype='API',
                                key='apiuser')
    AccessRight.objects.create(codename='can_manage_appointment',
                resource_type=obj_type, resource_pk=connector.pk,
                apiuser=apiuser)

    data = {'fields': {'clicrdv_date_raw': '2017-01-01' , 'clicrdv_time_raw': '12:00:00',
                       'firstname': 'Foo', 'lastname': 'Bar', 'email': 'foobar@example.com'}}
    resp = app.post_json('/clicrdv/test/interventions/63258/create?apikey=apiuser', params=data).json
    assert resp['data']
    assert not resp['data']['success']
    assert resp['data']['error'] == 'creation failed'
