# -*- coding: utf-8 -*-

# tests/test_api_particulier.py
# Copyright (C) 2017  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import pytest
from httmock import urlmatch, HTTMock, response

from django.core.urlresolvers import reverse

from passerelle.apps.api_particulier.models import APIParticulier

from utils import make_resource, endpoint_get

SVAIR_RESPONSE = {
    "declarant1": {
        "nom": "Martin",
        "nomNaissance": "Martin",
        "prenoms": "Pierre",
        "dateNaissance": "22/03/1985"
    },
    "declarant2": {
        "nom": "Martin",
        "nomNaissance": "Honore",
        "prenoms": "Marie",
        "dateNaissance": "03/04/1986"
    },
    "foyerFiscal": {
        "annee": 2015,
        "adresse": "12 rue Balzac 75008 Paris"
    },
    "dateRecouvrement": "10/10/2015",
    "dateEtablissement": "08/07/2015",
    "nombreParts": 2,
    "situationFamille": "Marié(e)s",
    "nombrePersonnesCharge": 2,
    "revenuBrutGlobal": 29880,
    "revenuImposable": 29880,
    "impotRevenuNetAvantCorrections": 2165,
    "montantImpot": 2165,
    "revenuFiscalReference": 29880,
    "anneeImpots": "2015",
    "anneeRevenus": "2014"
}

IMPOTS_ADRESSE = {
    "adresses": [
        {
            "adresse": {
                "citycode": "75108",
                "street": "Rue Balzac",
                "name": "12 Rue Balzac",
                "housenumber": "12",
                "city": "Paris",
                "type": "housenumber",
                "context": "75, Île-de-France",
                "score": 0.9401454545454544,
                "label": "12 Rue Balzac 75008 Paris",
                "postcode": "75008"
            },
            "geometry": {
                "type": "Point",
                "coordinates": [
                    2.300816,
                    48.873951
                ]
            }
        }
    ],
    "declarant1": {
        "nom": "Martin",
        "nomNaissance": "Martin",
        "prenoms": "Pierre",
        "dateNaissance": "22/03/1985"
    },
    "declarant2": {
        "nom": "Martin",
        "nomNaissance": "Honore",
        "prenoms": "Marie",
        "dateNaissance": "03/04/1986"
    },
    "foyerFiscal": {
        "annee": 2015,
        "adresse": "12 rue Balzac 75008 Paris"
    }
}

CAF_FAMILLE = {
    "adresse": {
        "codePostalVille": "12345 CONDAT",
        "complementIdentiteGeo": "ESCALIER B",
        "identite": "Madame MARIE DUPONT",
        "numeroRue": "123 RUE BIDON",
        "pays": "FRANCE"
    },
    "allocataires": [
        {
            "dateDeNaissance": "12111971",
            "nomPrenom": "MARIE DUPONT",
            "sexe": "F"
        },
        {
            "dateDeNaissance": "18101969",
            "nomPrenom": "JEAN DUPONT",
            "sexe": "M"
        }
    ],
    "annee": 2017,
    "enfants": [
        {
            "dateDeNaissance": "11122016",
            "nomPrenom": "LUCIE DUPONT",
            "sexe": "F"
        }
    ],
    "mois": 4,
    "quotientFamilial": 1754
}


@urlmatch(netloc=r'^particulier.*\.api\.gouv\.fr$',
          path=r'^/api/impots/svair$')
def api_particulier_impots_svair(url, request):
    return response(200, SVAIR_RESPONSE, request=request)


@urlmatch(netloc=r'^particulier.*\.api\.gouv\.fr$',
          path=r'^/api/impots/adress$')
def api_particulier_impots_adresse(url, request):
    return response(200, IMPOTS_ADRESSE, request=request)


@urlmatch(netloc=r'^particulier.*\.api\.gouv\.fr$',
          path=r'^/api/caf/famille$')
def api_particulier_caf_famille(url, request):
    return response(200, CAF_FAMILLE, request=request)


@urlmatch(netloc=r'^particulier.*\.api\.gouv\.fr$')
def api_particulier_error_500(url, request):
    return response(500, 'something bad happened', request=request)


@urlmatch(netloc=r'^particulier.*\.api\.gouv\.fr$')
def api_particulier_error_not_json(url, request):
    return response(200, 'something bad happened', request=request)


@urlmatch(netloc=r'^particulier.*\.api\.gouv\.fr$')
def api_particulier_error_not_found(url, request):
    return response(404, {
        'error': 'not_found',
        'message': u'Les paramètres fournis sont incorrects ou ne correspondent pas à un avis'
    }, request=request)


@pytest.yield_fixture
def mock_api_particulier():
    with HTTMock(api_particulier_impots_svair, api_particulier_impots_adresse, api_particulier_caf_famille):
        yield None


@pytest.fixture
def resource(db):
    return make_resource(
        APIParticulier,
        slug='test',
        title='API Particulier Prod',
        description='API Particulier Prod',
        platform='test',
        api_key='83c68bf0b6013c4daf3f8213f7212aa5')


def test_error(app, resource, mock_api_particulier):
    with HTTMock(api_particulier_error_500):
        def do(endpoint, params):
            resp = endpoint_get(
                '/api-particulier/test/%s' % endpoint,
                app,
                resource,
                endpoint,
                params=params)
            assert resp.status_code == 200
            assert resp.json['err'] == 1
            assert resp.json['data']['status_code'] == 500
        vector = [
            (['impots_svair', 'impots_adresse'], {
                'numero_fiscal': 12,
                'reference_avis': 15,
            }),
            (['caf_famille'], {
                'code_postal': 12,
                'numero_allocataire': 15
            }),
        ]
        for endpoints, params in vector:
            for endpoint in endpoints:
                do(endpoint, params)
    with HTTMock(api_particulier_error_not_json):
        def do(endpoint, params):
            resp = endpoint_get(
                '/api-particulier/test/%s' % endpoint,
                app,
                resource,
                endpoint,
                params=params)
            assert resp.status_code == 200
            assert resp.json['err'] == 1
            assert resp.json['data']['exception'] == 'No JSON object could be decoded'
        vector = [
            (['impots_svair', 'impots_adresse'], {
                'numero_fiscal': 12,
                'reference_avis': 15,
            }),
            (['caf_famille'], {
                'code_postal': 12,
                'numero_allocataire': 15
            }),
        ]
        for endpoints, params in vector:
            for endpoint in endpoints:
                do(endpoint, params)
    with HTTMock(api_particulier_error_not_found):
        def do(endpoint, params):
            resp = endpoint_get(
                '/api-particulier/test/%s' % endpoint,
                app,
                resource,
                endpoint,
                params=params)
            assert resp.status_code == 200
            assert resp.json['err'] == 1
            assert resp.json['err_desc'].endswith(u'à un avis')
        vector = [
            (['impots_svair', 'impots_adresse'], {
                'numero_fiscal': 12,
                'reference_avis': 15,
            }),
            (['caf_famille'], {
                'code_postal': 12,
                'numero_allocataire': 15
            }),
        ]
        for endpoints, params in vector:
            for endpoint in endpoints:
                do(endpoint, params)


def test_impots_svair(app, resource, mock_api_particulier):
    resp = endpoint_get(
        '/api-particulier/test/impots_svair',
        app,
        resource,
        'impots_svair',
        params={
            'numero_fiscal': 12,
            'reference_avis': 15,
            'user': 'John Doe',
        })
    assert resp.json['data']['montantImpot'] == 2165


def test_impots_adresse(app, resource, mock_api_particulier):
    resp = endpoint_get(
        '/api-particulier/test/impots_adresse',
        app,
        resource,
        'impots_adresse',
        params={
            'numero_fiscal': 12,
            'reference_avis': 15,
            'user': 'John Doe',
        })
    assert resp.json['data']['adresses'][0]['adresse']['citycode'] == '75108'


def test_caf_famille(app, resource, mock_api_particulier):
    resp = endpoint_get(
        '/api-particulier/test/caf_famille',
        app,
        resource,
        'caf_famille',
        params={
            'code_postal': '99148',
            'numero_allocataire': '000354',
            'user': 'John Doe',
        })
    assert resp.json['data']['adresse']['codePostalVille'] == '12345 CONDAT'


def test_detail_page(app, resource):
    response = app.get(reverse('view-connector', kwargs={
        'connector': 'api-particulier',
        'slug': 'test',
    }))
    assert 'API Particulier Prod' in response.content
    assert 'family allowance' in response.content
    assert 'fiscal information' in response.content
    assert 'fiscal address' in response.content

