# -*- coding: utf-8 -*-

import mock
import pytest

from django.core.urlresolvers import reverse
from django.utils import timezone
from django.utils.http import urlencode

from passerelle.contrib.agoraplus.models import AgoraPlus, AgoraPlusLink
from passerelle.contrib.agoraplus.normalize import DATE_FORMAT

import utils


pytestmark = pytest.mark.django_db

DATETIME_FORMAT_START = DATE_FORMAT + 'T%H:'  # test only date+hour, without min and seconds

NAME_ID = 'foobarnameid'

AGORAPLUS_PERISCOL_RESPONSE = """
{"DATA": {"INSCRIPTION_INFO": [
  {"ID": 38193,
   "ACTIVITY": {"ID": 97,
                "DESCRIPTION": "Accueil De Loisirs Vacances Élémentaire Toulon"
   },
   "START_DATE": "2015-09-01T00:00:00",
   "END_DATE": "2016-08-31T00:00:00"
   }]
 },
 "TYPE_RESULT": 1,
 "ERROR": {
   "ID": "0",
   "MESSAGE": ""
  }
}
"""

AGORAPLUS_PERISCOL_PLANNING_RESPONSE = """{
 "DATA": {
   "PLANNING_DAYS": [
   {"DAY": "2016-10-20T00:00:00",
    "START_TIME": "00:00",
    "END_TIME": "00:00"
   },
   {
    "DAY": "2016-10-21T00:00:00",
    "START_TIME": "00:00",
    "END_TIME": "00:00"
    }]
  },
  "TYPE_RESULT": 1,
  "ERROR": {
    "ID": "0",
    "MESSAGE": ""
  }
}
"""

@pytest.fixture
def connector():
    conn = AgoraPlus.objects.create(title='test', slug='test',
                                    url='https://agoraplus.com/',
                                    login='test', oauth_consumer_key='secret')
    utils.setup_access_rights(conn)
    return conn

@pytest.fixture
def link(connector):
    return AgoraPlusLink.objects.create(resource=connector, name_id=NAME_ID,
                                        login='foo', password='bar')

class FakedAgoraPlusResponse:
    def __init__(self, content):
        self.content = content

    def __call__(self, method, url, **kwargs):
        request = utils.FakedResponse(headers={'Content-Type': 'application/json'})
        if url == 'https://agoraplus.com/Auth/':
            return utils.FakedResponse(status_code=200, content='{"token": "random"}',
                                       request=request, headers={})
        return utils.FakedResponse(status_code=200, content=self.content, headers={},
                                   request=request)


@mock.patch('passerelle.utils.RequestSession.request')
def test_child_periscol_enrollments(mocked_get, app, connector, link):
    mocked_get.side_effect = FakedAgoraPlusResponse(AGORAPLUS_PERISCOL_RESPONSE)
    r = app.get(reverse('agoraplus-child-periscol-enrollments',
                        kwargs={'slug': connector.slug, 'child_id': '1'}),
                params={'service_id': '7'})
    # check remote webservice call args
    assert urlencode((('p_id_service', '7'),)) in mocked_get.call_args[0][1]
    date = timezone.now() + timezone.timedelta(days=35)
    assert urlencode((('p_date', date.strftime(DATETIME_FORMAT_START)),)) in mocked_get.call_args[0][1]
    assert r.json['data']

    r = app.get(reverse('agoraplus-child-periscol-enrollments',
                        kwargs={'slug': connector.slug, 'child_id': '1'}),
                params={'service_id': '6', 'start_days': '10'})
    # check remote webservice call args
    assert urlencode((('p_id_service', '6'),)) in mocked_get.call_args[0][1]
    date = timezone.now() + timezone.timedelta(days=10)
    assert urlencode((('p_date', date.strftime(DATETIME_FORMAT_START)),)) in mocked_get.call_args[0][1]
    assert r.json['data']

    r = app.get(reverse('agoraplus-child-periscol-enrollments',
                        kwargs={'slug': connector.slug, 'child_id': '1'}),
                params={'service_id': '6'})
    date = timezone.now() + timezone.timedelta(days=35)
    assert urlencode((('p_date', date.strftime(DATETIME_FORMAT_START)),)) in mocked_get.call_args[0][1]


@mock.patch('passerelle.utils.RequestSession.request')
def test_child_periscol_enrollment_planning(mocked_get, app, connector, link):
    mocked_get.side_effect = FakedAgoraPlusResponse(AGORAPLUS_PERISCOL_PLANNING_RESPONSE)
    r = app.get(reverse('agoraplus-periscol-child-enrollment-planning',
                        kwargs={'slug': connector.slug, 'child_id': '1', 'enrollment_id': '232'}),
                params={'reserved_day': '1'})
    # check remote webservice call args
    assert urlencode((('reserved_day', '1'),)) in mocked_get.call_args[0][1]
    date = timezone.now() + timezone.timedelta(days=2)
    assert urlencode((('start_date', date.strftime(DATETIME_FORMAT_START)),)) in mocked_get.call_args[0][1]
    assert r.json['data']

    r = app.get(reverse('agoraplus-periscol-child-enrollment-planning',
                        kwargs={'slug': connector.slug, 'child_id': '1', 'enrollment_id': '232'}),
                params={'reserved_day': '-1', 'start_days': '10', 'end_days': '31'})
    # check remote webservice call args
    assert urlencode((('reserved_day', '-1'),)) in mocked_get.call_args[0][1]
    date = timezone.now() + timezone.timedelta(days=10)
    assert urlencode((('start_date', date.strftime(DATETIME_FORMAT_START)),)) in mocked_get.call_args[0][1]
    date = timezone.now() + timezone.timedelta(days=31)
    assert urlencode((('end_date', date.strftime(DATETIME_FORMAT_START)),)) in mocked_get.call_args[0][1]
    assert r.json['data']

    r = app.get(reverse('agoraplus-periscol-child-enrollment-planning',
                        kwargs={'slug': connector.slug, 'child_id': '1', 'enrollment_id': '232'}),
                params={'reserved_day': '-1'})
    date = timezone.now() + timezone.timedelta(days=2)
    assert urlencode((('start_date', date.strftime(DATETIME_FORMAT_START)),)) in mocked_get.call_args[0][1]
    date = timezone.now() + timezone.timedelta(days=365)
    assert urlencode((('end_date', date.strftime(DATETIME_FORMAT_START)),)) in mocked_get.call_args[0][1]
