# passerelle-imio-liege-lisrue - passerelle connector to Lisrue webservice
# Copyright (C) 2016  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unicodedata
import urlparse
import requests

from django.core.urlresolvers import reverse
from django.db import models
from django.utils.translation import ugettext_lazy as _

from passerelle.base.models import BaseResource
from passerelle.utils.api import endpoint


class ImioLiegeLisrue(BaseResource):
    service_url = models.CharField(max_length=128, blank=False,
            verbose_name=_('Service URL'),
            help_text=_('SIG Web Service URL (ex: https://e-services.liege.be:8443/)'))
    include_all_of_belgium = models.BooleanField(default=True,
            verbose_name=_('Include all of Belgium'))
    verify_cert = models.BooleanField(default=True,
            verbose_name=_('Check HTTPS Certificate validity'))

    category = _('Geographic information system')

    class Meta:
        verbose_name = _('Liege Lisrue Service')

    @classmethod
    def get_verbose_name(cls):
        return cls._meta.verbose_name

    @classmethod
    def get_icon_class(cls):
        return 'gis'

    @classmethod
    def get_connector_slug(cls):
        return 'imio-liege-lisrue'

    @endpoint()
    def voies(self, request, q=None, **kwargs):
        url = self.service_url
        if self.include_all_of_belgium:
            url += 'jsonlisrue/'
        else:
            url += 'jsonlisrue2/'

        if q:
            q = unicodedata.normalize('NFKD', request.GET['q']).encode('ascii', 'ignore')
            url += q.lower()

        result = requests.get(url, headers={'Accept': 'application/json'},
                verify=self.verify_cert).json()
        if isinstance(result['rues'], list):
            lisrues = result['rues']
        elif isinstance(result['rues'], dict) and 'return' in result['rues']:
            lisrues = [result['rues']['return']]
        elif isinstance(result['rues'], dict) and 'rue' in result['rues']:
            lisrues = [result['rues']['rue']]
        else:
            lisrues = []

        streets = []
        known_street_labels = {}
        for item in lisrues:
            if item.get('rue'):
                street_label = item.get('rue')
            elif item.get('libelleMinuscule'):
                street_label = '%s %s' % (
                        item.get('particuleMinuscule') or '',
                        item.get('libelleMinuscule'))
            else:
                continue
            if street_label in known_street_labels:
                continue
            known_street_labels[street_label] = True
            streets.append({
                'id': item.get('codeRue'),
                'text': street_label.strip(),
                })
        return {'data': streets}
