import datetime
import decimal
from unittest import mock

import pytest

from lingo.agendas.chrono import ChronoError
from lingo.agendas.models import Agenda, CheckType, CheckTypeGroup
from lingo.invoicing import utils
from lingo.invoicing.errors import PayerDataError, PayerError
from lingo.invoicing.models import (
    Campaign,
    CampaignAsyncJob,
    DraftInvoiceLine,
    DraftJournalLine,
    JournalLine,
    Pool,
    PoolAsyncJob,
    Regie,
)

pytestmark = pytest.mark.django_db


def test_get_booked_dates_from_journal():
    regie = Regie.objects.create(label='Regie')
    campaign = Campaign.objects.create(
        regie=regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
        fine_config={
            'cat1': {'kind': 'user'},
        },
    )
    pool = Pool.objects.create(
        campaign=campaign,
        draft=True,
    )

    analyzed_campaign = Campaign.objects.create(
        regie=regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
    )
    analyzed_pool = Pool.objects.create(
        campaign=analyzed_campaign,
        draft=False,
    )

    check_types = {}

    # no journal lines
    assert utils.get_booked_dates_from_journal(
        pool=pool,
        analyzed_pool=analyzed_pool,
        check_types=check_types,
        category='cat1',
    ) == ({}, [])

    # create journal lines
    journal_line1 = JournalLine.objects.create(
        event_date=datetime.date(2022, 9, 1),
        event={
            'agenda': 'agenda-1',
            'slug': 'event-1',
            'primary_event': 'primary-event-1',
        },
        pricing_data={
            'booking_details': {'check_type': 'foo', 'check_type_group': 'foobar', 'status': 'presence'}
        },
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
    )
    journal_line2 = JournalLine.objects.create(
        event_date=datetime.date(2022, 9, 2),
        event={
            'agenda': 'agenda-1',
            'slug': 'event-1',
            'primary_event': 'primary-event-1',
        },
        pricing_data={'booking_details': {'status': 'presence'}},
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:2',  # other user
    )
    journal_line3 = JournalLine.objects.create(
        event_date=datetime.date(2022, 9, 2),
        event={
            'agenda': 'agenda-1',
            'slug': 'event-1',
            'primary_event': 'primary-event-1',
        },
        pricing_data={'booking_details': {'status': 'presence'}},
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
    )
    journal_line4 = JournalLine.objects.create(
        event_date=datetime.date(2022, 9, 3),
        event={
            'agenda': 'agenda-1',
            'slug': 'event-1',
            'primary_event': 'primary-event-1',
        },
        pricing_data={'booking_details': {'status': 'absence'}},
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
    )
    journal_line5 = JournalLine.objects.create(
        event_date=datetime.date(2022, 9, 4),
        event={
            'agenda': 'agenda-1',
            'slug': 'event-1',
            'primary_event': 'primary-event-1',
        },
        pricing_data={
            'booking_details': {
                'check_type': 'unexpected',
                'check_type_group': 'foobar',
                'status': 'presence',
            }
        },
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
    )
    JournalLine.objects.create(
        event_date=datetime.date(2022, 9, 5),
        event={
            'agenda': 'agenda-1',
            'slug': 'event-1',
            'primary_event': 'primary-event-1',
        },
        pricing_data={
            'booking_details': {'check_type': 'foo', 'check_type_group': 'foobar', 'status': 'presence'}
        },
        amount=1,
        status='error',  # wrong status
        pool=analyzed_pool,
        user_external_id='user:1',
    )
    JournalLine.objects.create(
        event_date=datetime.date(2022, 9, 6),
        event={
            'agenda': 'agenda-1',
            'slug': 'event-1',
            'primary_event': 'primary-event-1',
        },
        pricing_data={'booking_details': {'status': 'foo'}},  # wrong status
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
    )
    journal_line8 = JournalLine.objects.create(
        event_date=datetime.date(2022, 9, 1),
        event={
            'agenda': 'agenda-2',
            'slug': 'event-2',
        },
        pricing_data={
            'booking_details': {'check_type': 'foo', 'check_type_group': 'foobar', 'status': 'presence'}
        },
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
    )
    JournalLine.objects.create(
        event_date=datetime.date(2022, 9, 2),
        event={
            'agenda': 'other-category-agenda',  # wrong agenda
            'slug': 'event-2',
        },
        pricing_data={
            'booking_details': {'check_type': 'foo', 'check_type_group': 'foobar', 'status': 'presence'}
        },
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
    )
    JournalLine.objects.create(
        event_date=datetime.date(2022, 9, 3),
        event={
            'agenda': 'no-category-agenda',  # wrong agenda
            'slug': 'event-2',
        },
        pricing_data={
            'booking_details': {'check_type': 'foo', 'check_type_group': 'foobar', 'status': 'presence'}
        },
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
    )

    # no agenda
    assert utils.get_booked_dates_from_journal(
        pool=pool,
        analyzed_pool=analyzed_pool,
        check_types=check_types,
        category='cat1',
    ) == ({}, [])

    # create agendas
    Agenda.objects.create(label='Agenda 1', slug='agenda-1', category_slug='cat1')
    Agenda.objects.create(label='Agenda 2', slug='agenda-2', category_slug='cat1')
    Agenda.objects.create(label='Other category agenda', slug='other-category-agenda', category_slug='cat2')
    Agenda.objects.create(label='No category agenda', slug='no-category-agenda')

    assert utils.get_booked_dates_from_journal(
        pool=pool,
        analyzed_pool=analyzed_pool,
        check_types=check_types,
        category='cat1',
    ) == (
        {
            'user:1': {
                'user:1': {
                    'agenda-1@primary-event-1': {
                        '2022-09-01': {'status': 'booked', 'line': journal_line1},
                        '2022-09-02': {'status': 'booked', 'line': journal_line3},
                        '2022-09-03': {'status': 'booked', 'line': journal_line4},
                        '2022-09-04': {'status': 'booked', 'line': journal_line5},
                    },
                    'agenda-2@event-2': {
                        '2022-09-01': {'status': 'booked', 'line': journal_line8},
                    },
                }
            },
            'user:2': {
                'user:2': {
                    'agenda-1@primary-event-1': {
                        '2022-09-02': {'status': 'booked', 'line': journal_line2},
                    }
                }
            },
        },
        [],
    )

    # with check types
    group = CheckTypeGroup.objects.create(label='foobar')
    CheckType.objects.create(label='foo', group=group, kind='presence')
    check_type = CheckType.objects.create(label='unexpected', group=group, kind='presence')
    group.unexpected_presence = check_type
    group.save()
    check_types = {(c.slug, c.group.slug, c.kind): c for c in CheckType.objects.select_related('group').all()}
    assert utils.get_booked_dates_from_journal(
        pool=pool,
        analyzed_pool=analyzed_pool,
        check_types=check_types,
        category='cat1',
    ) == (
        {
            'user:1': {
                'user:1': {
                    'agenda-1@primary-event-1': {
                        '2022-09-01': {'status': 'booked', 'line': journal_line1},
                        '2022-09-02': {'status': 'booked', 'line': journal_line3},
                        '2022-09-03': {'status': 'booked', 'line': journal_line4},
                        '2022-09-04': {'status': 'unexpected', 'line': journal_line5},
                    },
                    'agenda-2@event-2': {
                        '2022-09-01': {'status': 'booked', 'line': journal_line8},
                    },
                }
            },
            'user:2': {
                'user:2': {
                    'agenda-1@primary-event-1': {
                        '2022-09-02': {'status': 'booked', 'line': journal_line2},
                    }
                }
            },
        },
        [],
    )

    # with family kind
    campaign.fine_config = {
        'cat1': {'kind': 'family', 'extra_data_family_id_key': 'family_id'},
    }

    # family_id not found
    assert utils.get_booked_dates_from_journal(
        pool=pool,
        analyzed_pool=analyzed_pool,
        check_types=check_types,
        category='cat1',
    ) == (
        {},
        [
            journal_line1,
            journal_line2,
            journal_line3,
            journal_line4,
            journal_line5,
            journal_line8,
        ],
    )

    JournalLine.objects.update(booking={'extra_data': {}})
    assert utils.get_booked_dates_from_journal(
        pool=pool,
        analyzed_pool=analyzed_pool,
        check_types=check_types,
        category='cat1',
    ) == (
        {},
        [
            journal_line1,
            journal_line2,
            journal_line3,
            journal_line4,
            journal_line5,
            journal_line8,
        ],
    )

    # only one family
    JournalLine.objects.update(booking={'extra_data': {'family_id': 'family:1'}})
    assert utils.get_booked_dates_from_journal(
        pool=pool,
        analyzed_pool=analyzed_pool,
        check_types=check_types,
        category='cat1',
    ) == (
        {
            'family:1': {
                'user:1': {
                    'agenda-1@primary-event-1': {
                        '2022-09-01': {'status': 'booked', 'line': journal_line1},
                        '2022-09-02': {'status': 'booked', 'line': journal_line3},
                        '2022-09-03': {'status': 'booked', 'line': journal_line4},
                        '2022-09-04': {'status': 'unexpected', 'line': journal_line5},
                    },
                    'agenda-2@event-2': {
                        '2022-09-01': {'status': 'booked', 'line': journal_line8},
                    },
                },
                'user:2': {
                    'agenda-1@primary-event-1': {
                        '2022-09-02': {'status': 'booked', 'line': journal_line2},
                    }
                },
            }
        },
        [],
    )

    # two families, and missing family on one line
    JournalLine.objects.filter(pk=journal_line2.pk).update(booking={'extra_data': {'family_id': 'family:2'}})
    JournalLine.objects.filter(pk=journal_line1.pk).update(booking={'extra_data': {'family_id': ''}})
    assert utils.get_booked_dates_from_journal(
        pool=pool,
        analyzed_pool=analyzed_pool,
        check_types=check_types,
        category='cat1',
    ) == (
        {
            'family:2': {
                'user:2': {
                    'agenda-1@primary-event-1': {
                        '2022-09-02': {'status': 'booked', 'line': journal_line2},
                    }
                },
            },
            'family:1': {
                'user:1': {
                    'agenda-1@primary-event-1': {
                        '2022-09-02': {'status': 'booked', 'line': journal_line3},
                        '2022-09-03': {'status': 'booked', 'line': journal_line4},
                        '2022-09-04': {'status': 'unexpected', 'line': journal_line5},
                    },
                    'agenda-2@event-2': {
                        '2022-09-01': {'status': 'booked', 'line': journal_line8},
                    },
                },
            },
        },
        [journal_line1],
    )


@mock.patch('lingo.invoicing.utils.get_subscriptions')
def test_get_user_subscriptions_error(mock_subscriptions):
    mock_subscriptions.side_effect = ChronoError('foo baz')
    with pytest.raises(ChronoError):
        utils.get_user_subscriptions(
            booked_dates={
                'fam1': {
                    'user:1': {
                        'events': {
                            'agenda-1@event': 'foo',
                        },
                    },
                }
            },
            fine_config={
                'date_school_period_start': '2022-09-01',
                'date_school_period_end': '2023-09-01',
            },
        )


@mock.patch('lingo.invoicing.utils.get_subscriptions')
def test_get_user_subscriptions(mock_subscriptions):
    assert utils.get_user_subscriptions(booked_dates={}, fine_config={}) == {}

    assert (
        utils.get_user_subscriptions(
            booked_dates={
                'fam1': {
                    'user:1': {
                        'agenda-1@event': 'foo',
                        'agenda-2@event': 'foo',
                    },
                    'user:2': {
                        'agenda-1@event': 'foo',
                    },
                    'user:3': {
                        'agenda-1@event': 'foo',
                    },
                },
                'fam2': {
                    'user:1': {
                        'agenda-1@event': 'foo',
                    },
                    'user:2': {
                        'agenda-3@event': 'foo',
                    },
                },
            },
            fine_config={
                'date_school_period_start': '2022-09-01',
                'date_school_period_end': '2023-09-01',
            },
        )
        == {}
    )
    assert mock_subscriptions.call_args_list == [
        mock.call(agenda_slug='agenda-1', date_start='2022-09-01', date_end='2023-09-01'),
        mock.call(agenda_slug='agenda-2', date_start='2022-09-01', date_end='2023-09-01'),
        mock.call(agenda_slug='agenda-3', date_start='2022-09-01', date_end='2023-09-01'),
    ]

    mock_subscriptions.side_effect = [
        [
            {
                'user_external_id': 'user:1',
                'user_first_name': 'User1',
                'user_last_name': 'Name1',
                'date_start': '2022-08-15',
                'date_end': '2022-09-02',
            },
            {
                'user_external_id': 'user:1',
                'user_first_name': 'Foo Bar',
                'user_last_name': '',
                'date_start': '2022-09-02',
                'date_end': '2022-09-03',
            },
            {
                'user_external_id': 'user:2',
                'user_first_name': '',
                'user_last_name': '',
                'date_start': '2022-09-02',
                'date_end': '2022-09-03',
            },
        ],
        [
            {
                'user_external_id': 'user:1',
                'user_first_name': 'User1 Name1',
                'user_last_name': '',
                'date_start': '2022-08-01',
                'date_end': '2022-10-01',
            },
        ],
    ]
    assert (
        utils.get_user_subscriptions(
            booked_dates={
                'fam1': {
                    'user:1': {
                        'agenda-1@event': 'foo',
                        'agenda-2@event': 'foo',
                    },
                }
            },
            fine_config={
                'date_school_period_start': '2022-09-01',
                'date_school_period_end': '2023-09-01',
            },
        )
    ) == {
        'user:1': [
            {
                'date_start': '2022-08-01',
                'date_end': '2022-10-01',
                'agenda_slug': 'agenda-2',
            },
            {
                'date_start': '2022-08-15',
                'date_end': '2022-09-02',
                'agenda_slug': 'agenda-1',
            },
            {
                'date_start': '2022-09-02',
                'date_end': '2022-09-03',
                'agenda_slug': 'agenda-1',
            },
        ],
        'user:2': [
            {
                'date_start': '2022-09-02',
                'date_end': '2022-09-03',
                'agenda_slug': 'agenda-1',
            }
        ],
    }


@mock.patch('lingo.invoicing.utils.get_user_subscriptions')
def test_get_unexpected_presences_to_invoice(mock_user_subscriptions):
    mock_user_subscriptions.return_value = {
        'user:1': [
            {
                'date_start': '2022-09-01',
                'date_end': '2023-09-01',
                'agenda_slug': 'agenda-1',
            },
            {
                'date_start': '2022-09-01',
                'date_end': '2023-09-01',
                'agenda_slug': 'agenda-2',
            },
        ],
        'user:2': [
            {
                'date_start': '2022-09-01',
                'date_end': '2023-09-01',
                'agenda_slug': 'agenda-1',
            },
        ],
        'user:3': [
            {
                'date_start': '2022-09-01',
                'date_end': '2023-09-01',
                'agenda_slug': 'agenda-1',
            },
        ],
    }

    assert utils.get_unexpected_presences_to_invoice(booked_dates={}, fine_config={}) == {}

    # fam1:
    # user:1 has 2 unexpected presences, 09-04 & 09-01
    # user:2 has 1 unexpected presence, 09-02
    # user:3 has no unexpected presence
    # fam2:
    # user:1 has 1 unexpected presence, 09-04
    # user:2 has no unexpected presence
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-1@event-1': {
                    '2022-09-04': {'status': 'unexpected', 'line': 'foo1'},
                    '2022-09-02': {'status': 'booked', 'line': 'foo2'},
                },
                'agenda-2@event-2': {
                    '2022-09-01': {'status': 'unexpected', 'line': 'foo3'},
                    '2022-09-04': {'status': 'booked', 'line': 'foo4'},
                },
            },
            'user:2': {
                'agenda-1@event-1': {
                    '2022-09-02': {'status': 'unexpected', 'line': 'foo5'},
                }
            },
            'user:3': {
                'agenda-1@event-1': {'3033-09-03': {'status': 'booked', 'line': 'foo6'}},
            },
        },
        'fam2': {
            'user:1': {
                'agenda-1@event-1': {
                    '2022-09-02': {'status': 'booked', 'line': 'foo7'},
                    '2022-09-04': {'status': 'unexpected', 'line': 'foo8'},
                },
            },
            'user:2': {
                'agenda-1@event-1': {'2022-09-02': {'status': 'booked', 'line': 'foo9'}},
            },
        },
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config={
            'date_school_period_start': '2022-09-01',
            'date_school_period_end': '2023-09-01',
            'waiting_period_first_subscription': '0',
            'waiting_period_other_subscriptions': '0',
        },
    ) == {
        'fam1': {
            '2022-09-01': 'foo3',
            '2022-09-02': 'foo5',
            '2022-09-04': 'foo1',
        },
        'fam2': {
            '2022-09-04': 'foo8',
        },
    }


@mock.patch('lingo.invoicing.utils.get_user_subscriptions')
def test_get_unexpected_presences_to_invoice_with_waiting_periods(mock_user_subscriptions):
    mock_user_subscriptions.return_value = {
        'user:1': [
            {
                'date_start': '2022-09-01',
                'date_end': '2022-12-31',
                'agenda_slug': 'agenda-1',
            },
            {
                'date_start': '2023-01-01',
                'date_end': '2023-06-01',
                'agenda_slug': 'agenda-2',
            },
            {
                'date_start': '2023-06-01',
                'date_end': '2023-09-01',
                'agenda_slug': 'agenda-3',
            },
        ],
    }

    fine_config = {
        'date_school_period_start': '2022-09-01',
        'date_school_period_end': '2023-09-01',
        'waiting_period_first_subscription': '10',
        'waiting_period_other_subscriptions': '5',
    }

    # unexpected presence is contained by first subscription, in waiting period
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-1@event-1': {
                    '2022-09-04': {'status': 'unexpected', 'line': 'foo'},
                },
            },
        }
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config=fine_config,
    ) == {'fam1': {}}
    assert mock_user_subscriptions.call_args_list == [
        mock.call(booked_dates=booked_dates, fine_config=fine_config),
    ]

    # unexpected presence is contained by first subscription, in waiting period
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-1@event-1': {
                    '2022-09-10': {'status': 'unexpected', 'line': 'foo'},
                },
            },
        }
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config=fine_config,
    ) == {'fam1': {}}

    # unexpected presence is contained by first subscription, out of waiting period
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-1@event-1': {
                    '2022-09-11': {'status': 'unexpected', 'line': 'foo'},
                },
            },
        }
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config=fine_config,
    ) == {'fam1': {'2022-09-11': 'foo'}}

    # unexpected presence is contained by second subscription, in waiting period
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-2@event-1': {
                    '2023-01-05': {'status': 'unexpected', 'line': 'foo'},
                },
            },
        }
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config=fine_config,
    ) == {'fam1': {}}

    # unexpected presence is contained by second subscription, out of waiting period
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-2@event-1': {
                    '2023-01-06': {'status': 'unexpected', 'line': 'foo'},
                },
            },
        }
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config=fine_config,
    ) == {'fam1': {'2023-01-06': 'foo'}}

    # unexpected presence is contained by third subscription, in waiting period
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-3@event-1': {
                    '2023-06-05': {'status': 'unexpected', 'line': 'foo'},
                },
            },
        }
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config=fine_config,
    ) == {'fam1': {}}

    # unexpected presence is contained by third subscription, out of waiting period
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-3@event-1': {
                    '2023-06-06': {'status': 'unexpected', 'line': 'foo'},
                },
            },
        }
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config=fine_config,
    ) == {'fam1': {'2023-06-06': 'foo'}}

    # no subscription matching event
    mock_user_subscriptions.return_value = {
        'user:1': [
            {
                'date_start': '2022-09-01',
                'date_end': '2022-09-04',  # before
                'agenda_slug': 'agenda-1',
            },
            {
                'date_start': '2022-09-01',
                'date_end': '2022-09-05',
                'agenda_slug': 'agenda-2',  # other agenda
            },
            {
                'date_start': '2023-09-05',  # after
                'date_end': '2023-10-01',
                'agenda_slug': 'agenda-1',
            },
        ],
    }
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-1@event-1': {
                    '2022-09-04': {'status': 'unexpected', 'line': 'foo'},
                },
            },
        }
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config=fine_config,
    ) == {'fam1': {}}

    # matching subscription starts before school period (and it works)
    mock_user_subscriptions.return_value = {
        'user:1': [
            {
                'date_start': '2022-08-23',
                'date_end': '2022-10-01',
                'agenda_slug': 'agenda-1',
            },
        ],
    }
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-1@event-1': {
                    '2022-09-01': {'status': 'unexpected', 'line': 'foo'},
                },
            },
        }
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config=fine_config,
    ) == {'fam1': {}}
    booked_dates = {
        'fam1': {
            'user:1': {
                'agenda-1@event-1': {
                    '2022-09-02': {'status': 'unexpected', 'line': 'foo'},
                },
            },
        }
    }
    assert utils.get_unexpected_presences_to_invoice(
        booked_dates=booked_dates,
        fine_config=fine_config,
    ) == {'fam1': {'2022-09-02': 'foo'}}


@mock.patch('lingo.invoicing.utils.get_booked_dates_from_journal')
@mock.patch('lingo.invoicing.utils.get_unexpected_presences_to_invoice')
def test_apply_fines(mock_unexpected, mock_booked_dates):
    other_regie = Regie.objects.create(label='Other regie')
    analyzed_campaign = Campaign.objects.create(
        regie=other_regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
    )
    analyzed_pool = Pool.objects.create(
        campaign=analyzed_campaign,
        draft=False,
    )

    journal_line1 = JournalLine.objects.create(
        label='Event 1',
        event_date=datetime.date(2022, 9, 1),
        event={'agenda': 'agenda-1'},
        booking={'foo': 'bar'},
        pricing_data={'foo': 'baz'},
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
        user_first_name='User1',
        user_last_name='Name1',
        accounting_code='414141',
    )
    journal_line2 = JournalLine.objects.create(
        label='Event 2',
        event_date=datetime.date(2022, 9, 4),
        event={'agenda': 'agenda-2'},
        booking={'foo': 'bar'},
        pricing_data={'foo': 'baz'},
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:2',
        user_first_name='User2',
        user_last_name='Name2',
        accounting_code='414142',
    )
    journal_line3 = JournalLine.objects.create(
        label='Event 3',
        event_date=datetime.date(2022, 9, 2),
        event={'agenda': 'agenda-3'},
        booking={'foo': 'bar'},
        pricing_data={'foo': 'baz'},
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:2',
        user_first_name='User2',
        user_last_name='Name2',
        accounting_code='414143',
    )
    journal_line4 = JournalLine.objects.create(
        label='Event 4',
        event_date=datetime.date(2022, 9, 4),
        event={'agenda': 'agenda-4'},
        booking={'foo': 'bar'},
        pricing_data={'foo': 'baz'},
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
        user_first_name='User1',
        user_last_name='Name1',
        accounting_code='414144',
    )
    journal_line5 = JournalLine.objects.create(
        label='Event 5',
        event_date=datetime.date(2022, 9, 5),
        event={'agenda': 'agenda-5'},
        booking={'foo': 'bar'},
        pricing_data={'foo': 'baz'},
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
        user_first_name='User1',
        user_last_name='Name1',
        payer_external_id='payer:1',
        payer_first_name='First1',
        payer_last_name='Last1',
        payer_address='41 rue des kangourous\n99999 Kangourou Ville',
        payer_email='email1',
        payer_phone='phone1',
        payer_direct_debit=True,
        accounting_code='414144',
    )

    regie = Regie.objects.create(label='Regie')
    campaign = Campaign.objects.create(
        regie=regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
        fine_config={
            'cat1': {
                'amount': '15',
            },
            'cat2': {
                'amount': '7.5',
            },
        },
        analyzed_campaign=analyzed_campaign,
    )
    pool = Pool.objects.create(
        campaign=campaign,
        draft=True,
    )
    pjob = PoolAsyncJob.objects.create(
        pool=pool,
        status='registered',
        users={'user:1': ('User1', 'Name1'), 'user:2': ('User2', 'Name2')},
    )

    group = CheckTypeGroup.objects.create(label='foobar')
    CheckType.objects.create(label='foo', group=group, kind='presence')
    check_type = CheckType.objects.create(label='unexpected', group=group, kind='presence')
    group.unexpected_presence = check_type
    group.save()
    check_types = {(c.slug, c.group.slug, c.kind): c for c in CheckType.objects.select_related('group').all()}

    mock_booked_dates.return_value = ({'foo': 'baz'}, [])
    mock_unexpected.return_value = {}

    def get_payer(ap, r, user_external_id):
        return {
            'user:1': 'payer:1',
            'user:2': 'payer:2',
        }.get(user_external_id)

    def get_payer_data(ap, r, payer_external_id):
        return {
            'payer:1': {
                'first_name': 'First1',
                'last_name': 'Last1',
                'address': '41 rue des kangourous\n99999 Kangourou Ville',
                'email': 'email1',
                'phone': 'phone1',
                'direct_debit': False,
            },
            'payer:2': {
                'first_name': 'First2',
                'last_name': 'Last2',
                'address': '42 rue des kangourous\n99999 Kangourou Ville',
                'email': 'email2',
                'phone': 'phone2',
                'direct_debit': True,
            },
        }.get(payer_external_id)

    payer_patch = mock.patch.object(Regie, 'get_payer_external_id', autospec=True)
    payer_data_patch = mock.patch.object(Regie, 'get_payer_data', autospec=True)
    with payer_patch as mock_payer, payer_data_patch as mock_payer_data:
        mock_payer.side_effect = get_payer
        mock_payer_data.side_effect = get_payer_data

        # not fine campaign
        assert (
            utils.apply_fines(
                pool=pool,
            )
            == []
        )
        assert mock_booked_dates.call_args_list == []
        assert mock_unexpected.call_args_list == []

        # fine campaign
        campaign.fine_campaign = True
        campaign.save()
        assert (
            utils.apply_fines(
                pool=pool,
            )
            == []
        )
        assert mock_booked_dates.call_args_list == [
            mock.call(pool=pool, analyzed_pool=analyzed_pool, category='cat1', check_types=check_types),
            mock.call(pool=pool, analyzed_pool=analyzed_pool, category='cat2', check_types=check_types),
        ]
        assert mock_unexpected.call_args_list == [
            mock.call(booked_dates={'foo': 'baz'}, fine_config={'amount': '15'}),
            mock.call(booked_dates={'foo': 'baz'}, fine_config={'amount': '7.5'}),
        ]

        # check generated lines
        mock_booked_dates.side_effect = [
            (
                {
                    'fam1': {},
                    'fam2': {},
                    'fam3': {},
                },
                [],
            ),
            (
                {
                    'fam1': {},
                    'fam2': {},
                },
                [journal_line5],
            ),
        ]
        mock_unexpected.side_effect = [
            {
                'fam1': {
                    '2022-09-01': journal_line1,
                    # no fine applied for it, as a fine is already applied for the first one
                    '2022-09-04': journal_line2,
                },
                'fam2': {
                    '2022-09-02': journal_line3,
                },
                'fam3': {},
            },
            {
                'fam1': {
                    '2022-09-04': journal_line4,
                },
                'fam2': {},
            },
        ]

        lines = utils.apply_fines(
            pool=pool,
            job=pjob,
        )
        assert pjob.total_count == 5
        assert pjob.current_count == 5
        lines = DraftJournalLine.objects.filter(pk__in=[li.pk for li in lines]).order_by('pk')
        assert len(lines) == 4
        assert lines[0].event_date == datetime.date(2022, 9, 1)
        assert lines[0].slug == 'agenda-1@cat1-fine'
        assert lines[0].label == 'Event 1'
        assert lines[0].description == ''
        assert lines[0].amount == 15
        assert lines[0].quantity == 1
        assert lines[0].quantity_type == 'units'
        assert lines[0].user_external_id == 'user:1'
        assert lines[0].user_first_name == 'User1'
        assert lines[0].user_last_name == 'Name1'
        assert lines[0].payer_external_id == 'payer:1'
        assert lines[0].payer_first_name == 'First1'
        assert lines[0].payer_last_name == 'Last1'
        assert lines[0].payer_address == '41 rue des kangourous\n99999 Kangourou Ville'
        assert lines[0].payer_phone == 'phone1'
        assert lines[0].payer_email == 'email1'
        assert lines[0].payer_direct_debit is False
        assert lines[0].event == {'agenda': 'agenda-1'}
        assert lines[0].booking == {'foo': 'bar'}
        assert lines[0].pricing_data == {
            'fine': {'category': 'cat1', 'config': {'amount': '15'}, 'user_or_family': 'fam1'}
        }
        assert lines[0].accounting_code == '414141'
        assert lines[0].status == 'success'
        assert lines[0].pool == pool
        assert lines[0].from_injected_line is None
        assert lines[1].event_date == datetime.date(2022, 9, 2)
        assert lines[1].slug == 'agenda-3@cat1-fine'
        assert lines[1].label == 'Event 3'
        assert lines[1].description == ''
        assert lines[1].amount == 15
        assert lines[1].quantity == 1
        assert lines[1].quantity_type == 'units'
        assert lines[1].user_external_id == 'user:2'
        assert lines[1].user_first_name == 'User2'
        assert lines[1].user_last_name == 'Name2'
        assert lines[1].payer_external_id == 'payer:2'
        assert lines[1].payer_first_name == 'First2'
        assert lines[1].payer_last_name == 'Last2'
        assert lines[1].payer_address == '42 rue des kangourous\n99999 Kangourou Ville'
        assert lines[1].payer_phone == 'phone2'
        assert lines[1].payer_email == 'email2'
        assert lines[1].payer_direct_debit is True
        assert lines[1].event == {'agenda': 'agenda-3'}
        assert lines[1].booking == {'foo': 'bar'}
        assert lines[1].pricing_data == {
            'fine': {'category': 'cat1', 'config': {'amount': '15'}, 'user_or_family': 'fam2'}
        }
        assert lines[1].accounting_code == '414143'
        assert lines[1].status == 'success'
        assert lines[1].pool == pool
        assert lines[1].from_injected_line is None
        assert lines[2].event_date == datetime.date(2022, 9, 4)
        assert lines[2].slug == 'agenda-4@cat2-fine'
        assert lines[2].label == 'Event 4'
        assert lines[2].description == ''
        assert lines[2].amount == decimal.Decimal('7.5')
        assert lines[2].quantity == 1
        assert lines[2].quantity_type == 'units'
        assert lines[2].user_external_id == 'user:1'
        assert lines[2].user_first_name == 'User1'
        assert lines[2].user_last_name == 'Name1'
        assert lines[2].payer_external_id == 'payer:1'
        assert lines[2].payer_first_name == 'First1'
        assert lines[2].payer_last_name == 'Last1'
        assert lines[2].payer_address == '41 rue des kangourous\n99999 Kangourou Ville'
        assert lines[2].payer_phone == 'phone1'
        assert lines[2].payer_email == 'email1'
        assert lines[2].payer_direct_debit is False
        assert lines[2].event == {'agenda': 'agenda-4'}
        assert lines[2].booking == {'foo': 'bar'}
        assert lines[2].pricing_data == {
            'fine': {'config': {'amount': '7.5'}, 'category': 'cat2', 'user_or_family': 'fam1'}
        }
        assert lines[2].accounting_code == '414144'
        assert lines[2].status == 'success'
        assert lines[2].pool == pool
        assert lines[2].from_injected_line is None
        assert lines[3].event_date == datetime.date(2022, 9, 5)
        assert lines[3].slug == 'agenda-5@cat2-fine'
        assert lines[3].label == 'Event 5'
        assert lines[3].description == ''
        assert lines[3].amount == decimal.Decimal('7.5')
        assert lines[3].quantity == 0
        assert lines[3].quantity_type == 'units'
        assert lines[3].user_external_id == 'user:1'
        assert lines[3].user_first_name == 'User1'
        assert lines[3].user_last_name == 'Name1'
        assert lines[3].payer_external_id == 'payer:1'
        assert lines[3].payer_first_name == 'First1'
        assert lines[3].payer_last_name == 'Last1'
        assert lines[3].payer_address == '41 rue des kangourous\n99999 Kangourou Ville'
        assert lines[3].payer_phone == 'phone1'
        assert lines[3].payer_email == 'email1'
        assert lines[3].payer_direct_debit is True
        assert lines[3].event == {'agenda': 'agenda-5'}
        assert lines[3].booking == {'foo': 'bar'}
        assert lines[3].pricing_data == {'error': 'FineFamilyIDError', 'error_details': ''}
        assert lines[3].accounting_code == '414144'
        assert lines[3].status == 'warning'
        assert lines[3].pool == pool
        assert lines[3].from_injected_line is None

        mock_booked_dates.side_effect = None
        mock_unexpected.side_effect = [
            {},
            {
                'fam1': {
                    '2022-09-04': journal_line2,
                }
            },
        ]

        lines = utils.apply_fines(
            pool=pool,
        )
        lines = DraftJournalLine.objects.filter(pk__in=[li.pk for li in lines]).order_by('pk')
        assert len(lines) == 1


@mock.patch('lingo.invoicing.utils.get_booked_dates_from_journal')
@mock.patch('lingo.invoicing.utils.get_unexpected_presences_to_invoice')
@mock.patch('lingo.invoicing.models.Regie.get_payer_external_id')
def test_apply_fines_get_payer_id_error(mock_payer, mock_unexpected, mock_booked_dates):
    other_regie = Regie.objects.create(label='Other regie')
    analyzed_campaign = Campaign.objects.create(
        regie=other_regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
    )
    analyzed_pool = Pool.objects.create(
        campaign=analyzed_campaign,
        draft=False,
    )
    journal_line = JournalLine.objects.create(
        label='Event 1',
        event_date=datetime.date(2022, 9, 2),
        event={'agenda': 'agenda-1'},
        booking={'foo': 'bar'},
        pricing_data={'foo': 'baz'},
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
        user_first_name='User1',
        user_last_name='Name1',
        accounting_code='414141',
    )

    regie = Regie.objects.create(label='Regie')
    campaign = Campaign.objects.create(
        regie=regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
        fine_campaign=True,
        fine_config={'cat1': {'amount': 15}},
        analyzed_campaign=analyzed_campaign,
    )
    pool = Pool.objects.create(
        campaign=campaign,
        draft=True,
    )

    mock_booked_dates.return_value = (
        {
            'user:1': {
                'user:1': {
                    'agenda-1@event-1': {
                        '2022-09-02': {'status': 'unexpected', 'line': 'foo'},
                    },
                },
            },
        },
        [],
    )
    mock_unexpected.return_value = {
        'user:1': {
            '2022-09-02': journal_line,
        }
    }
    mock_payer.side_effect = PayerError(details={'foo': 'bar'})
    lines = utils.apply_fines(
        pool=pool,
    )
    lines = DraftJournalLine.objects.filter(pk__in=[li.pk for li in lines]).order_by('pk')
    assert len(lines) == 1
    assert lines[0].event_date == datetime.date(2022, 9, 2)
    assert lines[0].slug == 'agenda-1@cat1-fine'
    assert lines[0].label == 'Event 1'
    assert lines[0].description == ''
    assert lines[0].amount == 15
    assert lines[0].quantity == 0
    assert lines[0].quantity_type == 'units'
    assert lines[0].user_external_id == 'user:1'
    assert lines[0].user_first_name == 'User1'
    assert lines[0].user_last_name == 'Name1'
    assert lines[0].payer_external_id == 'unknown'
    assert lines[0].payer_first_name == ''
    assert lines[0].payer_last_name == ''
    assert lines[0].payer_address == ''
    assert lines[0].payer_phone == ''
    assert lines[0].payer_email == ''
    assert lines[0].payer_direct_debit is False
    assert lines[0].event == {'agenda': 'agenda-1'}
    assert lines[0].booking == {'foo': 'bar'}
    assert lines[0].pricing_data == {
        'fine': {'config': {'amount': 15}, 'category': 'cat1', 'user_or_family': 'user:1'},
        'error': 'PayerError',
        'error_details': {'foo': 'bar'},
    }
    assert lines[0].accounting_code == '414141'
    assert lines[0].status == 'error'
    assert lines[0].pool == pool
    assert lines[0].from_injected_line is None


@mock.patch('lingo.invoicing.utils.get_booked_dates_from_journal')
@mock.patch('lingo.invoicing.utils.get_unexpected_presences_to_invoice')
@mock.patch('lingo.invoicing.models.Regie.get_payer_external_id')
@mock.patch('lingo.invoicing.models.Regie.get_payer_data')
def test_apply_fines_get_payer_data_error(mock_payer_data, mock_payer, mock_unexpected, mock_booked_dates):
    other_regie = Regie.objects.create(label='Other regie')
    analyzed_campaign = Campaign.objects.create(
        regie=other_regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
    )
    analyzed_pool = Pool.objects.create(
        campaign=analyzed_campaign,
        draft=False,
    )
    journal_line = JournalLine.objects.create(
        label='Event 1',
        event_date=datetime.date(2022, 9, 2),
        event={'agenda': 'agenda-1'},
        booking={'foo': 'bar'},
        pricing_data={'foo': 'baz'},
        amount=1,
        status='success',
        pool=analyzed_pool,
        user_external_id='user:1',
        user_first_name='User1',
        user_last_name='Name1',
        accounting_code='414141',
    )

    regie = Regie.objects.create(label='Regie')
    campaign = Campaign.objects.create(
        regie=regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
        fine_campaign=True,
        fine_config={'cat1': {'amount': 15}},
        analyzed_campaign=analyzed_campaign,
    )
    pool = Pool.objects.create(
        campaign=campaign,
        draft=True,
    )

    mock_booked_dates.return_value = (
        {
            'user:1': {
                'user:1': {
                    'events': {
                        'agenda-1@event-1': {
                            'accounting_code': '414141',
                            'dates': {'2022-09-02': 'unexpected'},
                        },
                    },
                    'user_external_id': 'user:1',
                    'user_first_name': 'User1',
                    'user_last_name': 'Name1',
                },
            },
        },
        [],
    )
    mock_unexpected.return_value = {
        'user:1': {
            '2022-09-02': journal_line,
        }
    }
    mock_payer.return_value = 'payer:1'
    mock_payer_data.side_effect = PayerDataError(details={'key': 'foobar', 'reason': 'foo'})
    lines = utils.apply_fines(
        pool=pool,
    )
    lines = DraftJournalLine.objects.filter(pk__in=[li.pk for li in lines]).order_by('pk')
    assert len(lines) == 1
    assert lines[0].event_date == datetime.date(2022, 9, 2)
    assert lines[0].slug == 'agenda-1@cat1-fine'
    assert lines[0].label == 'Event 1'
    assert lines[0].description == ''
    assert lines[0].amount == 15
    assert lines[0].quantity == 0
    assert lines[0].quantity_type == 'units'
    assert lines[0].user_external_id == 'user:1'
    assert lines[0].user_first_name == 'User1'
    assert lines[0].user_last_name == 'Name1'
    assert lines[0].payer_external_id == 'payer:1'
    assert lines[0].payer_first_name == ''
    assert lines[0].payer_last_name == ''
    assert lines[0].payer_address == ''
    assert lines[0].payer_phone == ''
    assert lines[0].payer_email == ''
    assert lines[0].payer_direct_debit is False
    assert lines[0].event == {'agenda': 'agenda-1'}
    assert lines[0].booking == {'foo': 'bar'}
    assert lines[0].pricing_data == {
        'fine': {'config': {'amount': 15}, 'category': 'cat1', 'user_or_family': 'user:1'},
        'error': 'PayerDataError',
        'error_details': {'key': 'foobar', 'reason': 'foo'},
    }
    assert lines[0].accounting_code == '414141'
    assert lines[0].status == 'error'
    assert lines[0].pool == pool
    assert lines[0].from_injected_line is None


def test_generate_invoices_from_lines():
    Agenda.objects.create(label='Agenda 1')
    regie = Regie.objects.create(label='Regie')
    campaign = Campaign.objects.create(
        regie=regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
    )
    pool = Pool.objects.create(
        campaign=campaign,
        draft=True,
        status='running',
    )
    line = DraftJournalLine.objects.create(
        label='Event 1',
        slug='agenda-1@foobar',
        event_date=datetime.date(2022, 9, 1),
        event={
            'agenda': 'agenda-1',
            'primary_event': 'event-1',
            'start_datetime': '2022-09-01T12:00:00+02:00',
        },
        pricing_data={'fine': {'config': {'amount': '7.5'}, 'category': 'cat2', 'user_or_family': 'fam1'}},
        amount=15,
        accounting_code='424242',
        user_external_id='user:1',
        user_first_name='UserFirst1',
        user_last_name='UserLast1',
        payer_external_id='payer:1',
        payer_first_name='First1',
        payer_last_name='Last1',
        payer_address='41 rue des kangourous\n99999 Kangourou Ville',
        payer_direct_debit=False,
        status='success',
        pool=pool,
    )

    invoices = utils.generate_invoices_from_lines(pool=pool)
    assert len(invoices) == 1
    line = DraftJournalLine.objects.get()
    assert DraftInvoiceLine.objects.count() == 1
    iline = DraftInvoiceLine.objects.get()
    assert iline.event_date == campaign.date_start
    assert iline.label == 'Event 1'
    assert iline.quantity == 1
    assert iline.unit_amount == 15
    assert iline.details == {}
    assert iline.event_slug == 'agenda-1@foobar'
    assert iline.event_label == 'Event 1'
    assert iline.agenda_slug == 'agenda-1'
    assert iline.activity_label == 'Agenda 1'
    assert iline.description == 'Fine'
    assert iline.accounting_code == '424242'
    assert iline.user_external_id == 'user:1'
    assert iline.user_first_name == 'UserFirst1'
    assert iline.user_last_name == 'UserLast1'
    assert iline.pool == pool
    assert iline == line.invoice_line


@mock.patch('lingo.invoicing.utils.apply_fines')
@mock.patch('lingo.invoicing.utils.generate_invoices_from_lines')
def test_generate_invoices(mock_generate, mock_fines):
    regie = Regie.objects.create(label='Regie')
    campaign = Campaign.objects.create(
        regie=regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
        fine_campaign=True,
    )

    # check only calls between functions
    with mock.patch('lingo.invoicing.models.CampaignAsyncJob.run') as mock_run:
        campaign.generate()
        assert mock_run.call_args_list == [mock.call()]

    pool = Pool.objects.latest('pk')
    assert pool.campaign == campaign
    assert pool.draft is True

    assert CampaignAsyncJob.objects.count() == 1
    assert PoolAsyncJob.objects.count() == 0
    cjob = CampaignAsyncJob.objects.get()
    assert cjob.campaign == campaign
    assert cjob.params == {'draft_pool_id': pool.pk, 'force_cron': False}
    assert cjob.action == 'generate'
    assert cjob.status == 'registered'
    with mock.patch('lingo.invoicing.models.PoolAsyncJob.run') as mock_run:
        cjob.run()
        assert mock_run.call_args_list == [mock.call(), mock.call()]
    assert cjob.status == 'completed'
    assert cjob.total_count == 2
    assert cjob.current_count == 2
    assert CampaignAsyncJob.objects.count() == 1
    assert PoolAsyncJob.objects.count() == 2
    pjob1, pjob2 = list(PoolAsyncJob.objects.all())
    assert pjob1.pool == pool
    assert pjob1.campaign_job == cjob
    assert pjob1.params == {'force_cron': False}
    assert pjob1.action == 'generate_fines'
    assert pjob1.status == 'registered'
    assert pjob2.pool == pool
    assert pjob2.campaign_job == cjob
    assert pjob2.params == {'force_cron': False}
    assert pjob2.action == 'finalize_invoices'
    assert pjob2.status == 'registered'

    pjob2.run()
    assert pjob2.status == 'waiting'

    pjob1.run()
    assert pjob1.status == 'completed'
    pjob2.run()
    assert pjob2.status == 'completed'

    assert mock_fines.call_args_list == [mock.call(pool=pool, job=pjob1)]
    assert mock_generate.call_args_list == [mock.call(pool=pool, job=pjob2)]


@mock.patch('lingo.invoicing.utils.apply_fines')
@mock.patch('lingo.invoicing.utils.generate_invoices_from_lines')
def test_generate_invoices_errors(mock_generate, mock_fines):
    regie = Regie.objects.create(label='Regie')
    campaign = Campaign.objects.create(
        regie=regie,
        date_start=datetime.date(2022, 9, 1),
        date_end=datetime.date(2022, 10, 1),
        date_publication=datetime.date(2022, 10, 1),
        date_payment_deadline=datetime.date(2022, 10, 31),
        date_due=datetime.date(2022, 10, 31),
        date_debit=datetime.date(2022, 11, 15),
        fine_campaign=True,
    )

    mock_fines.side_effect = ChronoError('foo baz')
    campaign.generate()
    pool = Pool.objects.latest('pk')
    assert pool.status == 'failed'
    assert pool.exception == 'foo baz'

    mock_fines.side_effect = Exception('foo baz')
    campaign.generate()
    pool = Pool.objects.latest('pk')
    assert pool.status == 'failed'
    assert pool.exception.startswith('Traceback (most recent call last):\n')
    assert pool.exception.endswith('Exception: foo baz\n')
