# hobo - portal to configure and deploy applications
# Copyright (C) 2015-2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import io
import json
import tarfile

from django.core.exceptions import ValidationError
from django.core.files.base import ContentFile
from django.utils.text import slugify
from django.utils.translation import gettext_lazy as _
from PIL import Image, UnidentifiedImageError

from .models import Application, AsyncJob, Element, Relation, Version


def check_install(fd, application=None, spool=True):
    try:
        with tarfile.open(fileobj=fd) as tar:
            try:
                manifest = json.loads(tar.extractfile('manifest.json').read().decode())
            except KeyError:
                raise ValidationError({'bundle': _('Invalid tar file, missing manifest.')})
            if application and application.slug != manifest.get('slug'):
                raise ValidationError(
                    {
                        'bundle': _('Can not update this application, wrong slug (%s).')
                        % manifest.get('slug'),
                    }
                )
            icon = manifest.get('icon')
            if icon:
                Image.open(tar.extractfile(icon))
            raw_slug = manifest.get('slug')
            app_slug = slugify(raw_slug)
            if app_slug != raw_slug:
                raise ValidationError({'bundle': _('Invalid tar file: invalid slug in metadata.')})

            # do not reuse deleted app
            Application.objects.filter(slug=manifest.get('slug'), marked_for_deletion=True).delete()

            app, created = Application.objects.get_or_create(
                slug=manifest.get('slug'), defaults={'name': manifest.get('application')}
            )
            application = app
            app.name = manifest.get('application')
            app.description = manifest.get('description')
            app.documentation_url = manifest.get('documentation_url', '')
            app.authors = manifest.get('authors', '')
            app.license = manifest.get('license', '')
            if created:
                # mark as non-editable only newly deployed applications, this allows
                # overwriting a local application and keep on developing it.
                app.editable = False
            app.save()
            if icon:
                app.icon.save(icon, tar.extractfile(icon), save=True)
            else:
                app.icon.delete()
    except tarfile.TarError:
        raise ValidationError({'bundle': _('Invalid tar file.')})
    except UnidentifiedImageError:
        raise ValidationError({'bundle': _('Invalid icon file.')})

    # always create a new version on install or if previous version has not the same number
    version_number = manifest.get('version_number') or 'unknown'
    latest_version = app.version_set.order_by('last_update_timestamp').last()
    if latest_version and latest_version.number == version_number:
        version = latest_version
    else:
        version = Version(application=app)
    version.number = version_number
    version.notes = manifest.get('version_notes') or ''
    fd.seek(0)
    version.bundle.save('%s.tar' % app.slug, content=ContentFile(fd.read()))
    version.save()

    # check if some objects where locally modified or already exist outside the application
    job = AsyncJob(
        label=_('Check installation'),
        application=application,
        version=version,
        action='check-first-install' if created else 'check-install',
    )
    job.save()
    job.run(spool=spool)
    return job


def deploy(version, application=None, spool=True):
    application = application if application is not None else version.application

    # create elements and relations
    version.bundle.seek(0)
    tar_io = io.BytesIO(version.bundle.read())
    with tarfile.open(fileobj=tar_io) as tar:
        manifest = json.loads(tar.extractfile('manifest.json').read().decode())
        application.relation_set.all().delete()
        for element_dict in manifest.get('elements'):
            element, dummy = Element.objects.get_or_create(
                type=element_dict['type'],
                slug=element_dict['slug'],
                defaults={'name': element_dict['name']},
            )
            element.name = element_dict['name']
            element.save()

            relation = Relation(
                application=application,
                element=element,
                auto_dependency=element_dict['auto-dependency'],
            )
            relation.set_error('not-installed')
            relation.save()
    version.bundle.seek(0)

    # and run deployment
    job = AsyncJob(
        label=_('Deploying application bundle'),
        application=application,
        version=version,
        action='deploy',
    )
    job.save()
    job.run(spool=spool)
    return job
