# hobo - portal to configure and deploy applications
# Copyright (C) 2015-2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import base64

from django.http import Http404, HttpResponse
from django.utils.encoding import force_str
from rest_framework import permissions
from rest_framework.response import Response
from rest_framework.views import APIView

from hobo.rest_permissions import IsAdminUser, IsAPIClient

from .views import get_app_version

APIAdminOrAuthUser = (IsAPIClient & IsAdminUser) | ((~IsAPIClient) & permissions.IsAuthenticated)


class ApplicationDownloadAPI(APIView):
    permission_classes = (APIAdminOrAuthUser,)
    raw = False

    def get_permissions(self):
        try:
            self.version = get_app_version(self.kwargs['app_slug'], number=self.kwargs.get('number'))
        except Http404:
            self.version = None
        else:
            if self.version.application.license:
                # if there is a license then it's a free software. we always allow to download it
                return [permissions.AllowAny()]
        return super().get_permissions()

    def get(self, request, app_slug, number=None):
        if self.version is None:
            if self.raw:
                # return a real http error
                raise Http404
            return Response({'err': 1, 'err_desc': 'not found'})
        self.version.bundle.seek(0)
        content = self.version.bundle.read()
        if self.raw:
            resp = HttpResponse(
                content,
                headers={
                    'Content-Type': 'application/x-tar',
                    'Content-Disposition': f'attachment; filename="{app_slug}-{self.version.number}.tar"',
                },
            )
        else:
            resp = Response(
                {
                    'err': 0,
                    'data': {
                        'filename': '%s-%s.tar' % (app_slug, self.version.number),
                        'content_type': 'application/x-tar',
                        'content': force_str(base64.b64encode(content)),
                    },
                }
            )
        # allow to access version number without fetching the response body
        resp.headers['X-Application-Version'] = self.version.number
        return resp


application_download = ApplicationDownloadAPI.as_view()
application_download_raw = ApplicationDownloadAPI.as_view(raw=True)
