# hobo - portal to configure and deploy applications
# Copyright (C) 2015-2016 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import itertools
import os
import time

import pytest
from django.template.engine import Engine
from django.test import override_settings
from tenant_schemas.utils import tenant_context

from hobo.multitenant.template_loader import FilesystemLoader


def test_combo_user_portal_theme_loading_order(settings, make_tenant):
    templates = list(settings.TEMPLATES)
    templates[0]['OPTIONS']['loaders'] = ['hobo.multitenant.template_loader.FilesystemLoader']
    templates[0]['APP_DIRS'] = False

    template_file = 'some_template.html'
    template_dirs = ('/', '/tpl')
    # avoid DjangoCachedLoader hits

    with override_settings(TEMPLATES=templates):
        cur_tenant = make_tenant('combo.examples.net', 'user')
        with tenant_context(cur_tenant):
            settings.TEMPLATE_VARS['theme'] = 'publik'
            fsl = FilesystemLoader(Engine.get_default())
            paths = [origin.name for origin in fsl.get_template_sources(template_file, template_dirs)]
            assert paths == [
                '/combo.examples.net/templates/variants/publik/portal-user/some_template.html',
                '/combo.examples.net/templates/variants/publik/some_template.html',
                '/combo.examples.net/theme/templates/variants/publik/portal-user/some_template.html',
                '/combo.examples.net/theme/templates/variants/publik/some_template.html',
                '/combo.examples.net/templates/some_template.html',
                '/combo.examples.net/theme/templates/some_template.html',
                '/tpl/combo.examples.net/templates/variants/publik/portal-user/some_template.html',
                '/tpl/combo.examples.net/templates/variants/publik/some_template.html',
                '/tpl/combo.examples.net/theme/templates/variants/publik/portal-user/some_template.html',
                '/tpl/combo.examples.net/theme/templates/variants/publik/some_template.html',
                '/tpl/combo.examples.net/templates/some_template.html',
                '/tpl/combo.examples.net/theme/templates/some_template.html',
            ]


@pytest.mark.parametrize(
    'service_id,service_slug,template_name_override',
    [
        ('welco', 'test', None),
        ('authentic', 'other', None),
        ('combo', 'user', None),
        ('combo', 'agent', None),
        ('combo', 'another', None),
        ('combo', 'another', 'custom_theme'),
    ],
)
@pytest.mark.parametrize('theme', ('publik', 'sometheme', ''))
def test_combo_theme_loading(settings, make_tenant, service_id, service_slug, template_name_override, theme):
    templates = list(settings.TEMPLATES)
    templates[0]['OPTIONS']['loaders'] = ['hobo.multitenant.template_loader.FilesystemLoader']
    templates[0]['APP_DIRS'] = False

    template_file = 'some_template.html'
    template_dirs = ('/', '/tpl')
    # avoid DjangoCachedLoader hits
    tenant_host = 'test%f.examples.net' % time.time()

    with override_settings(TEMPLATES=templates):
        cur_tenant = make_tenant(tenant_host, service_slug)
        with tenant_context(cur_tenant):
            settings.TEMPLATE_VARS['theme'] = theme
            if template_name_override:
                settings.KNOWN_SERVICES[service_id][service_slug]['template_name'] = template_name_override
            template_name = settings.KNOWN_SERVICES[service_id][service_slug].get('template_name', '')
            fsl = FilesystemLoader(Engine.get_default())
            paths = {origin.name for origin in fsl.get_template_sources(template_file, template_dirs)}

            directories = list(
                itertools.chain(
                    *[
                        [os.path.join(tpl_dir, tenant_host, known_dir) for tpl_dir in template_dirs]
                        for known_dir in ('templates', 'theme/templates')
                    ]
                )
            )

            dirnames = ['']
            if theme:
                dirnames += ['variants/%s' % theme]
                if template_name:
                    dirnames.append('variants/%s/%s' % (theme, template_name))

            expected = set(
                itertools.chain(
                    *[[os.path.join(tpl_dir, dirname) for tpl_dir in directories] for dirname in dirnames]
                )
            )
            if service_slug == 'agent':
                expected = {os.path.join(d, 'portal-agent') for d in expected} | expected

            expected = {os.path.join(d, template_file) for d in expected}
            assert expected == paths
