# hobo - © Entr'ouvert

import contextlib
import ctypes
import signal
import threading
import time


class Timeout(Exception):
    pass


def raise_exception_in_thread(thread, exception):
    ctypes.pythonapi.PyThreadState_SetAsyncExc(ctypes.c_ulong(thread.ident), ctypes.py_object(exception))


def sigwinch_handler(signum, frame):
    raise Timeout()


def waiting_thread_target(wait, deadline, thread, stop_event, use_signal):
    while not stop_event.wait(wait):
        if time.time() > deadline:
            # belt and suspenders
            if use_signal:
                signal.pthread_kill(thread.ident, signal.SIGWINCH)
            raise_exception_in_thread(thread, Timeout)
            wait = 1


@contextlib.contextmanager
def timeout(seconds, use_signal=True):
    '''Setup a signal handler and a waiting thread to raise an exception in the
    current thread. It's not asyncio safe.

    It's not perfect, as you could completely ignore exceptions in a loop like
    that, do not do that:

    while True:
        try:
            do()
        except Exception:
            continue

    or reset the SIGWINCH handler and call time.sleep(100000), but in most
    cases, it will work.
    '''
    use_signal = use_signal and threading.current_thread() is threading.main_thread()
    if use_signal:
        old = signal.signal(signal.SIGWINCH, sigwinch_handler)
    current_thread = threading.current_thread()
    deadline = time.time() + seconds
    stop_event = threading.Event()

    waiting_thread = threading.Thread(
        target=waiting_thread_target,
        args=(seconds, deadline, current_thread, stop_event, use_signal),
        daemon=True,
    )
    waiting_thread.start()

    try:
        yield
        if use_signal:
            signal.pthread_sigmask(signal.SIG_BLOCK, [signal.SIGWINCH])
    except Exception:
        if use_signal:
            signal.pthread_sigmask(signal.SIG_BLOCK, [signal.SIGWINCH])
        raise
    finally:
        stop_event.set()
        if use_signal:
            signal.signal(signal.SIGWINCH, old)
            signal.pthread_sigmask(signal.SIG_UNBLOCK, [signal.SIGWINCH])


if __name__ == '__main__':
    # Should print:
    #
    # Traceback (most recent call last):
    #   File "/home/bdauvergne/wd/eo/hobo/hobo/timeout.py", line 54, in <module>
    #     time.sleep(10)
    #     ~~~~~~~~~~^^^^
    #   File "/home/bdauvergne/wd/eo/hobo/hobo/timeout.py", line 19, in sigwinch_handler
    #     raise Timeout()
    # Harakiri

    with timeout(2):
        time.sleep(10)
        print('finished')
