# hobo - portal to configure and deploy applications © Entr'ouvert

import logging
import os
import re

from django.conf import settings
from django.db import connection
from django.http import FileResponse, HttpResponse, JsonResponse
from django.utils.deprecation import MiddlewareMixin

WELL_KNOWN_PREFIX = '/.well-known/'
WELL_KNOWN_SETTING = 'WELL_KNOWN'

FILENAME_RE = re.compile('^(?:[a-zA-Z0-9-][a-zA-Z0-9.-]*/)*[a-zA-Z0-9-][a-zA-Z0-9.-]*$')


class WellKnownMiddleware(MiddlewareMixin):
    def process_request(self, request):
        if not request.path.startswith(WELL_KNOWN_PREFIX):
            return None

        if not request.method == 'GET':
            return None

        well_known = getattr(settings, WELL_KNOWN_SETTING, {})
        filename = request.path[len(WELL_KNOWN_PREFIX) :]

        if not FILENAME_RE.match(filename):
            return None

        # WELL_KNOWN setting
        if filename in well_known:
            description = well_known[filename]

            content = description.get('content', '')
            if not isinstance(content, str):
                content_type = description.get('content_type', 'application/json')
                try:
                    return JsonResponse(content, content_type=content_type, safe=False)
                except Exception:
                    logging.exception('JSON serialization of /.well-known/%s failed', filename)
                    return JsonResponse({'err': 'JSON serialization failed'})
            else:
                content_type = description.get('content_type', 'text/plain')
                return HttpResponse(content, content_type=content_type)

        # /var/lib/*/tenants/*/well-known/ directory
        if hasattr(connection, 'tenant'):
            path = os.path.join(connection.tenant.get_directory(), 'well-known', filename)
            if os.access(path, os.R_OK):
                return FileResponse(open(path, 'rb'))
        return None
